package io.contek.tusk;

import com.clickhouse.client.*;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;
import java.time.Duration;
import java.util.List;
import java.util.function.Consumer;
import java.util.logging.Logger;

import static com.clickhouse.client.ClickHouseCredentials.fromUserAndPassword;
import static com.clickhouse.client.ClickHouseFormat.TabSeparatedWithNamesAndTypes;
import static com.clickhouse.client.ClickHouseProtocol.HTTP;
import static com.clickhouse.client.config.ClickHouseClientOption.CONNECTION_TIMEOUT;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.logging.Level.SEVERE;
import static java.util.logging.Logger.getLogger;

@ThreadSafe
public final class MetricClient {

  private static final Logger LOGGER = getLogger(MetricClient.class.getName());

  private final ClickHouseClient client;
  private final ClickHouseNode server;

  private MetricClient(ClickHouseClient client, ClickHouseNode server) {
    this.client = client;
    this.server = server;
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  @Nullable
  Schema describe(Table table) {
    try {
      String query = format("DESCRIBE TABLE %s", table.getName());
      ClickHouseResponse response =
          newRequest(table.getDatabase())
              .format(TabSeparatedWithNamesAndTypes)
              .query(query)
              .execute()
              .get();
      List<ClickHouseRecord> records = ImmutableList.copyOf(response.records());

      Schema.Builder result = Schema.newBuilder();
      for (ClickHouseRecord record : records) {
        String columnName = record.getValue("name").asString();
        String columnType = record.getValue("type").asString();
        ClickHouseDataType type = parseType(columnType);
        result.add(columnName, type);
      }

      return result.build();
    } catch (Throwable t) {
      LOGGER.log(SEVERE, format("Failed to describe table %s.", table), t);
      return null;
    }
  }

  void write(MetricData data, Consumer<Throwable> onError) {
    newRequest(data.getTable().getDatabase())
        .write()
        .table(data.getTable().getName())
        .format(data.getFormat())
        .data(data.getInputStream())
        .seal()
        .execute()
        .exceptionally(
            t -> {
              LOGGER.log(SEVERE, format("Failed to write data into table %s.", data.getTable()), t);
              onError.accept(t);
              return null;
            });
  }

  private ClickHouseRequest<?> newRequest(@Nullable String database) {
    ClickHouseRequest<?> request = client.connect(server);
    if (database != null) {
      request.use(database);
    }
    return request;
  }

  private static ClickHouseDataType parseType(String typeName) {
    int variantStartIndex = typeName.indexOf('(');
    if (variantStartIndex >= 0) {
      typeName = typeName.substring(0, variantStartIndex);
    }
    return ClickHouseDataType.of(typeName);
  }

  @NotThreadSafe
  public static final class Builder {

    private String host = "localhost";
    private String user = "default";
    private Duration timeout = Duration.ofSeconds(60);

    private String password;
    private String database;

    public Builder setHost(String host) {
      this.host = host;
      return this;
    }

    public Builder setUser(String user) {
      this.user = user;
      return this;
    }

    public Builder setTimeout(Duration timeout) {
      this.timeout = timeout;
      return this;
    }

    public Builder setPassword(@Nullable String password) {
      this.password = password;
      return this;
    }

    public Builder setDatabase(@Nullable String database) {
      this.database = database;
      return this;
    }

    public MetricClient build() {
      requireNonNull(timeout);

      ClickHouseClient client =
          ClickHouseClient.builder()
              .nodeSelector(ClickHouseNodeSelector.of(HTTP))
              .option(CONNECTION_TIMEOUT, (int) timeout.toMillis())
              .build();
      ClickHouseNode server = createConnection();
      return new MetricClient(client, server);
    }

    private ClickHouseNode createConnection() {
      requireNonNull(host);
      requireNonNull(user);

      return ClickHouseNode.builder()
          .host(host)
          .database(database)
          .credentials(fromUserAndPassword(user, password))
          .port(HTTP)
          .build();
    }

    private Builder() {}
  }
}
