package io.contek.tusk;

import com.clickhouse.client.ClickHouseException;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Shorts;

import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;
import java.lang.Enum;
import java.lang.String;
import java.math.BigInteger;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.logging.Logger;

import static com.clickhouse.client.ClickHouseDataType.String;
import static com.clickhouse.client.ClickHouseDataType.*;
import static com.clickhouse.client.data.BinaryStreamUtils.*;
import static java.lang.String.format;
import static java.util.concurrent.Executors.newSingleThreadScheduledExecutor;
import static java.util.concurrent.TimeUnit.SECONDS;
import static java.util.logging.Level.INFO;
import static java.util.logging.Level.SEVERE;
import static java.util.logging.Logger.getLogger;

@ThreadSafe
public final class Metric {

  private static final Logger LOGGER = getLogger(Metric.class.getName());

  private final Table table;
  private final SchemaProvider schema;
  private final TimeColumnCache timeColumn;
  private final EnvTagsCache envTags;
  private final EntryChecker checker;
  private final MetricFormatter formatter;
  private final MetricBatch batch;

  private final AtomicReference<Future<?>> task = new AtomicReference<>(null);
  private final ScheduledExecutorService scheduler = newSingleThreadScheduledExecutor();

  private Metric(
      Table table,
      SchemaProvider schema,
      TimeColumnCache timeColumn,
      EnvTagsCache envTags,
      EntryChecker checker,
      MetricFormatter formatter,
      BatchingConfig batching) {
    this.table = table;
    this.schema = schema;
    this.timeColumn = timeColumn;
    this.envTags = envTags;
    this.checker = checker;
    this.formatter = formatter;
    this.batch = new MetricBatch(table, batching);
  }

  public static Metric metric(String table) {
    return metric(null, table);
  }

  public static Metric metric(@Nullable String database, String table) {
    return metric(Table.newBuilder().setDatabase(database).setName(table).build());
  }

  public static Metric metric(Table table) {
    return metric(table, BatchingConfig.getDefault());
  }

  public static Metric metric(Table.Builder table, BatchingConfig batching) {
    return metric(table.build(), batching);
  }

  public static Metric metric(Table table, BatchingConfig batching) {
    SchemaProvider schema = new SchemaProvider(table);
    return new Metric(
        table,
        schema,
        new TimeColumnCache(table, schema),
        new EnvTagsCache(schema),
        new EntryChecker(schema),
        new MetricFormatter(table, schema),
        batching);
  }

  public EntryWriter newEntry() {
    return new EntryWriter(this::insert, timeColumn, envTags, checker);
  }

  private void insert(MetricRow row) {
    batch.add(row);
    scheduleIfIdle();
  }

  private void scheduleIfIdle() {
    if (batch.isImmediate()) {
      flush();
      return;
    }

    synchronized (task) {
      Future<?> future = task.get();
      if (future != null && !future.isDone()) {
        return;
      }
      schedule();
    }
  }

  private void flushAndSchedule() {
    boolean updated = flush();
    if (!updated) {
      return;
    }

    schedule();
  }

  private void schedule() {
    synchronized (task) {
      Future<?> future =
          scheduler.schedule(this::flushAndSchedule, batch.getPeriod().getSeconds(), SECONDS);
      task.set(future);
    }
  }

  private boolean flush() {
    MetricClient client = Tusk.getClient();
    if (client == null) {
      return false;
    }

    MetricData data;
    try {
      data = batch.export(formatter);
      if (data == null) {
        return false;
      }
    } catch (Throwable e) {
      LOGGER.log(SEVERE, "Failed to format metric data.", e);
      return false;
    }

    client.write(data, this::onWriteError);
    return true;
  }

  private void onWriteError(Throwable t) {
    if (t instanceof CompletionException) {
      t = t.getCause();
    }
    if (t instanceof ClickHouseException) {
      onClientError((ClickHouseException) t);
    }
  }

  private void onClientError(ClickHouseException e) {
    switch (e.getErrorCode()) {
      case 33:
      case 60:
        clearSchema();
        break;
    }
  }

  private void clearSchema() {
    if (schema.clear()) {
      timeColumn.clear();
      envTags.clear();
      LOGGER.log(INFO, format("Schema cache for %s is evicted.", table));
    }
  }

  @NotThreadSafe
  public static final class EntryWriter {

    private final Consumer<MetricRow> consumer;
    private final TimeColumnCache timeColumnCache;
    private final EnvTagsCache envTagsCache;
    private final EntryChecker checker;

    private final Map<String, Object> keyValues = new HashMap<>();

    public EntryWriter putInt8(String key, byte value) {
      checker.check(key, Int8);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putInt16(String key, long value) {
      return putInt16(key, Shorts.checkedCast(value));
    }

    public EntryWriter putInt16(String key, short value) {
      checker.check(key, Int16);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putInt32(String key, long value) {
      return putInt32(key, Ints.checkedCast(value));
    }

    public EntryWriter putInt32(String key, int value) {
      checker.check(key, Int32);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putInt64(String key, long value) {
      checker.check(key, Int64);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putUInt8(String key, boolean bool) {
      int unsignedByte = bool ? 1 : 0;
      return putUInt8(key, unsignedByte);
    }

    public EntryWriter putUInt8(String key, byte signedByte) {
      int unsignedByte = signedByte & 0xff;
      return putUInt8(key, unsignedByte);
    }

    public EntryWriter putUInt8(String key, int value) {
      if (value > U_INT8_MAX) {
        throw new IllegalArgumentException(key + ": " + value + " > " + U_INT8_MAX);
      }
      checker.check(key, UInt8);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putUInt16(String key, short signedShort) {
      int unsignedShort = signedShort & 0xffff;
      return putUInt16(key, unsignedShort);
    }

    public EntryWriter putUInt16(String key, long value) {
      return putUInt16(key, Ints.checkedCast(value));
    }

    public EntryWriter putUInt16(String key, int value) {
      if (value > U_INT16_MAX) {
        throw new IllegalArgumentException(key + ": " + value + " > " + U_INT16_MAX);
      }
      checker.check(key, UInt16);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putUInt32(String key, int value) {
      long unsignedInt = value & 0x00000000ffffffffL;
      return putUInt32(key, unsignedInt);
    }

    public EntryWriter putUInt32(String key, long value) {
      if (value > U_INT32_MAX) {
        throw new IllegalArgumentException(key + ": " + value + " > " + U_INT32_MAX);
      }
      checker.check(key, UInt32);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putUInt64(String key, long value) {
      return putUInt64(key, BigInteger.valueOf(value));
    }

    public EntryWriter putUInt64(String key, BigInteger value) {
      if (value.compareTo(U_INT64_MAX) > 0) {
        throw new IllegalArgumentException(key + ": " + value + " > " + U_INT64_MAX);
      }
      checker.check(key, UInt64);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putFloat32(String key, float value) {
      checker.check(key, Float32);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putFloat64(String key, double value) {
      checker.check(key, Float64);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putString(String key, Enum<?> value) {
      return putString(key, value.name());
    }

    public EntryWriter putString(String key, String value) {
      checker.check(key, String);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putDateTime64(String key, Instant value) {
      checker.check(key, DateTime64);
      keyValues.put(key, value);
      return this;
    }

    public EntryWriter putAll(Map<String, ?> fields) {
      this.keyValues.putAll(fields);
      return this;
    }

    public void write() {
      String timeColumn = timeColumnCache.get();
      if (timeColumn != null) {
        checker.check(timeColumn, DateTime64);
        keyValues.putIfAbsent(timeColumn, Tusk.getClock().instant());
      }

      Map<String, String> tags = envTagsCache.get();
      if (tags != null) {
        tags.forEach(keyValues::putIfAbsent);
      }

      consumer.accept(new MetricRow(keyValues));
    }

    private EntryWriter(
        Consumer<MetricRow> consumer,
        TimeColumnCache timeColumnCache,
        EnvTagsCache envTagsCache,
        EntryChecker checker) {
      this.consumer = consumer;
      this.timeColumnCache = timeColumnCache;
      this.envTagsCache = envTagsCache;
      this.checker = checker;
    }
  }
}
