/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.multitenant.quota;

import io.confluent.kafka.multitenant.quota.ClusterMetadata;
import io.confluent.kafka.multitenant.quota.TenantPartitionAssignor;
import io.confluent.kafka.multitenant.quota.TestCluster;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TenantPartitionAssignorTest {
    private TestCluster testCluster = new TestCluster();
    private TenantPartitionAssignor partitionAssignor;

    @Before
    public void setUp() {
        this.partitionAssignor = new TenantPartitionAssignor();
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
    }

    @Test
    public void testRackUnawareAssignment() {
        String topic = "tenant1_topicA";
        this.addNodes(5, 0);
        this.createTopic(topic, 10, 3);
        List<List<Integer>> expectedAssignment = Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 3), Arrays.asList(2, 3, 4), Arrays.asList(3, 4, 0), Arrays.asList(4, 0, 1), Arrays.asList(0, 2, 3), Arrays.asList(1, 3, 4), Arrays.asList(2, 4, 0), Arrays.asList(3, 0, 1), Arrays.asList(4, 1, 2));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.verifyAssignmentsAreBalanced(0, 0, 0);
    }

    @Test
    public void testRackUnawareAssignmentOneNodeCluster() {
        String topic = "tenant1_topicA";
        this.addNodes(1, 0);
        this.createTopic(topic, 5, 1);
        List<List<Integer>> expectedAssignment = Arrays.asList(Collections.singletonList(0), Collections.singletonList(0), Collections.singletonList(0), Collections.singletonList(0), Collections.singletonList(0));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
    }

    @Test
    public void testRackAwareAssignment() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0, "rack1");
        this.testCluster.addNode(1, "rack3");
        this.testCluster.addNode(2, "rack3");
        this.testCluster.addNode(3, "rack2");
        this.testCluster.addNode(4, "rack2");
        this.testCluster.addNode(5, "rack1");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 12, 3);
        List<List<Integer>> expectedAssignment = Arrays.asList(Arrays.asList(0, 1, 3), Arrays.asList(1, 3, 5), Arrays.asList(3, 5, 2), Arrays.asList(5, 2, 4), Arrays.asList(2, 4, 0), Arrays.asList(4, 0, 1), Arrays.asList(0, 2, 4), Arrays.asList(1, 4, 0), Arrays.asList(3, 0, 1), Arrays.asList(5, 1, 3), Arrays.asList(2, 3, 5), Arrays.asList(4, 5, 2));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.verifyAssignmentsAreBalanced(0, 0, 0);
    }

    @Test
    public void testCellAwareAndRackUnawareAssignment() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0, null, "cell1");
        this.testCluster.addNode(1, null, "cell1");
        this.testCluster.addNode(2, null, "cell1");
        this.testCluster.addNode(3, null, "cell2");
        this.testCluster.addNode(4, null, "cell2");
        this.testCluster.addNode(5, null, "cell2");
        this.testCluster.addNode(6, null, "cell3");
        this.testCluster.addNode(7, null, "cell3");
        this.testCluster.addNode(8, null, "cell3");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 9, 3);
        List<List<Integer>> expectedAssignment = Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 0), Arrays.asList(2, 0, 1), Arrays.asList(3, 4, 5), Arrays.asList(4, 5, 3), Arrays.asList(5, 3, 4), Arrays.asList(6, 7, 8), Arrays.asList(7, 8, 6), Arrays.asList(8, 6, 7));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.verifyAssignmentsAreBalanced(0, 0, 0);
    }

    @Test
    public void testCellAndRackAwareAssignment() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0, "rack1", "cell1");
        this.testCluster.addNode(1, "rack2", "cell1");
        this.testCluster.addNode(2, "rack3", "cell1");
        this.testCluster.addNode(3, "rack1", "cell2");
        this.testCluster.addNode(4, "rack2", "cell2");
        this.testCluster.addNode(5, "rack3", "cell2");
        this.testCluster.addNode(6, "rack1", "cell3");
        this.testCluster.addNode(7, "rack2", "cell3");
        this.testCluster.addNode(8, "rack3", "cell3");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 9, 3);
        List<List<Integer>> expectedAssignment = Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 0), Arrays.asList(2, 0, 1), Arrays.asList(3, 4, 5), Arrays.asList(4, 5, 3), Arrays.asList(5, 3, 4), Arrays.asList(6, 7, 8), Arrays.asList(7, 8, 6), Arrays.asList(8, 6, 7));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.verifyAssignmentsAreBalanced(0, 0, 0);
    }

    @Test
    public void testCellAwareAssignmentWithOneCellIsEquivalentToRackUnawareAssignment() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0);
        this.testCluster.addNode(1);
        this.testCluster.addNode(2);
        this.testCluster.addNode(3);
        this.testCluster.addNode(4);
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 10, 3);
        List expectedAssignment = this.testCluster.cluster().partitionsForTopic(topic);
        this.testCluster.addNode(0, null, "cell1");
        this.testCluster.addNode(1, null, "cell1");
        this.testCluster.addNode(2, null, "cell1");
        this.testCluster.addNode(3, null, "cell1");
        this.testCluster.addNode(4, null, "cell1");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 10, 3);
        List actualAssignment = this.testCluster.cluster().partitionsForTopic(topic);
        this.verifyPartitionsAssignment(expectedAssignment, actualAssignment);
    }

    @Test
    public void testCellAwareAssignmentWithOneCellIsEquivalentToRackAwareAssignment() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0, "rack1");
        this.testCluster.addNode(1, "rack3");
        this.testCluster.addNode(2, "rack3");
        this.testCluster.addNode(3, "rack2");
        this.testCluster.addNode(4, "rack1");
        this.testCluster.addNode(5, "rack2");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 12, 3);
        List expectedAssignment = this.testCluster.cluster().partitionsForTopic(topic);
        this.testCluster.addNode(0, "rack1", "cell1");
        this.testCluster.addNode(1, "rack3", "cell1");
        this.testCluster.addNode(2, "rack3", "cell1");
        this.testCluster.addNode(3, "rack2", "cell1");
        this.testCluster.addNode(4, "rack1", "cell1");
        this.testCluster.addNode(5, "rack2", "cell1");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 12, 3);
        List actualAssignment = this.testCluster.cluster().partitionsForTopic(topic);
        this.verifyPartitionsAssignment(expectedAssignment, actualAssignment);
    }

    @Test
    public void testCellAwareAssignmentFillEmptyCell() {
        String topicA = "tenant1_topicA";
        String topicB = "tenant1_topicB";
        this.testCluster.addNode(0, null, "cell1");
        this.testCluster.addNode(1, null, "cell1");
        this.testCluster.addNode(2, null, "cell1");
        this.testCluster.addNode(3, null, "cell1");
        this.testCluster.addNode(4, null, "cell2");
        this.testCluster.addNode(5, null, "cell2");
        this.testCluster.addNode(6, null, "cell2");
        this.testCluster.addNode(7, null, "cell2");
        this.testCluster.createPartitions(topicA, 0, Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 3), Arrays.asList(2, 3, 0), Arrays.asList(3, 0, 1)));
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topicB, 4, 3);
        List<List<Integer>> expectedAssignment = Arrays.asList(Arrays.asList(4, 5, 6), Arrays.asList(5, 6, 7), Arrays.asList(6, 7, 4), Arrays.asList(7, 4, 5));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topicB));
    }

    @Test
    public void testCellAwareAssignmentBalancesLeadersFillEmptyNodes() {
        String topicA = "tenant1_topicA";
        String topicB = "tenant1_topicB";
        this.testCluster.addNode(0, null, "cell1");
        this.testCluster.addNode(1, null, "cell1");
        this.testCluster.addNode(2, null, "cell1");
        this.testCluster.addNode(3, null, "cell1");
        this.testCluster.addNode(4, null, "cell2");
        this.testCluster.addNode(5, null, "cell2");
        this.testCluster.addNode(6, null, "cell2");
        this.testCluster.addNode(7, null, "cell2");
        this.testCluster.createPartitions(topicA, 0, Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 3), Arrays.asList(2, 3, 0), Arrays.asList(4, 5, 6), Arrays.asList(5, 6, 7), Arrays.asList(6, 7, 4)));
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topicB, 2, 3);
        Assert.assertEquals(Arrays.asList(3, 7), (Object)this.testCluster.cluster().leadersForTopic(topicB));
    }

    @Test
    public void testCellAndRackAwareAssignmentAddPartitionsCorrectly() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0, "rack1", "cell1");
        this.testCluster.addNode(1, "rack2", "cell1");
        this.testCluster.addNode(2, "rack3", "cell1");
        this.testCluster.addNode(3, "rack1", "cell2");
        this.testCluster.addNode(4, "rack2", "cell2");
        this.testCluster.addNode(5, "rack3", "cell2");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        this.createTopic(topic, 3, 3);
        ArrayList<List<Integer>> expectedAssignment = new ArrayList<List<Integer>>();
        expectedAssignment.addAll(Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 0), Arrays.asList(2, 0, 1)));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.addPartitions(topic, 3, 3);
        expectedAssignment.addAll(Arrays.asList(Arrays.asList(3, 4, 5), Arrays.asList(4, 5, 3), Arrays.asList(5, 3, 4)));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.addPartitions(topic, 6, 3);
        expectedAssignment.addAll(Arrays.asList(Arrays.asList(0, 1, 2), Arrays.asList(1, 2, 0), Arrays.asList(2, 0, 1)));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
        this.addPartitions(topic, 9, 3);
        expectedAssignment.addAll(Arrays.asList(Arrays.asList(3, 4, 5), Arrays.asList(4, 5, 3), Arrays.asList(5, 3, 4)));
        this.verifyAssignment(expectedAssignment, this.testCluster.cluster().partitionsForTopic(topic));
    }

    @Test
    public void testCellAndRackAwareAssignmentWithUnsatisfiableReplicationFactor() {
        String topic = "tenant1_topicA";
        this.testCluster.addNode(0, "rack1", "cell1");
        this.testCluster.addNode(1, "rack2", "cell1");
        this.testCluster.addNode(2, "rack3", "cell1");
        this.testCluster.addNode(3, "rack1", "cell2");
        this.testCluster.addNode(4, "rack2", "cell2");
        this.testCluster.addNode(5, "rack3", "cell2");
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        Map assignment = this.partitionAssignor.assignPartitionsForNewTopics("tenant1", Collections.singletonMap("topicA", new TenantPartitionAssignor.TopicInfo(3, 20, 0)));
        Assert.assertEquals(Collections.emptyList(), assignment.get("topicA"));
    }

    @Test
    public void testNodeOrderingByPartitionCount() {
        this.addNodes(3, 0);
        this.testCluster.setPartitionLeaders("tenant1_topicA", 0, 10, 1);
        this.testCluster.setPartitionLeaders("tenant1_topicB", 0, 2, 2);
        this.testCluster.setPartitionLeaders("tenant2_topicA", 0, 5, 2);
        this.testCluster.setPartitionLeaders("tenant2_topicB", 0, 2, 1);
        this.verifyTopicCreateNodeOrder("tenant1", 0, 2, 1);
        this.verifyTopicCreateNodeOrder("tenant2", 0, 1, 2);
        this.verifyPartitionAddNodeOrder("tenant1", "tenant1_topicA", 0, 2, 1);
        this.verifyPartitionAddNodeOrder("tenant1", "tenant1_topicB", 0, 1, 2);
        this.verifyPartitionAddNodeOrder("tenant2", "tenant2_topicA", 0, 1, 2);
        this.verifyPartitionAddNodeOrder("tenant2", "tenant2_topicB", 0, 2, 1);
        this.testCluster.addNode(3, null);
        this.testCluster.setPartitionLeaders("tenant1_topicC", 0, 7, 3);
        this.verifyTopicCreateNodeOrder("tenant1", 0, 2, 3, 1);
        this.verifyPartitionAddNodeOrder("tenant1", "tenant1_topicB", 0, 3, 1, 2);
    }

    private void verifyTopicCreateNodeOrder(String tenant, Integer ... expectedNodeIds) {
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        ClusterMetadata clusterMetadata = new ClusterMetadata(tenant, this.testCluster.cluster());
        Map replicaCounts = clusterMetadata.nodeReplicaCounts(Collections.emptyList(), true);
        List nodes = TenantPartitionAssignor.orderNodes((Map)replicaCounts);
        Assert.assertEquals(Arrays.asList(expectedNodeIds), (Object)nodes);
    }

    private void verifyPartitionAddNodeOrder(String tenant, String topic, Integer ... expectedNodeIds) {
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        ClusterMetadata clusterMetadata = new ClusterMetadata(tenant, this.testCluster.cluster());
        Map replicaCounts = clusterMetadata.nodeReplicaCounts(this.testCluster.cluster().partitionsForTopic(topic), true);
        List nodes = TenantPartitionAssignor.orderNodes((Map)replicaCounts);
        Assert.assertEquals(Arrays.asList(expectedNodeIds), (Object)nodes);
    }

    @Test
    public void testRackUnawareCreateTopicsBrokerMultiples() {
        this.verifyCreateTopicsBrokerMultiples(0);
    }

    @Test
    public void testRackAwareCreateTopicsBrokerMultiples() {
        this.verifyCreateTopicsBrokerMultiples(3);
    }

    private void verifyCreateTopicsBrokerMultiples(int racks) {
        int i;
        int numBrokers = 6;
        this.addNodes(numBrokers, racks);
        for (i = 1; i <= 5; ++i) {
            this.createTopic("tenant1_topicA" + i, numBrokers, 3);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
        }
        for (i = 1; i <= 5; ++i) {
            this.createTopic("tenant2_topicB" + i, numBrokers, 3);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
        }
        for (i = 1; i <= 10; ++i) {
            this.createTopic("tenant1_topicC" + i, numBrokers * i, 3);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
        }
        for (i = 1; i <= 10; ++i) {
            this.createTopic("tenant2_topicD" + i, numBrokers * i, 3);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
        }
    }

    @Test
    public void testRackUnawareAddPartitionsBrokerMultiples() {
        this.verifyAddPartitionsBrokerMultiples(0);
    }

    @Test
    public void testRackAwareAddPartitionsBrokerMultiples() {
        this.verifyAddPartitionsBrokerMultiples(3);
    }

    private void verifyAddPartitionsBrokerMultiples(int racks) {
        int i;
        int numBrokers = 6;
        this.addNodes(numBrokers, racks);
        this.createTopic("tenant1_topicA", numBrokers, 3);
        for (i = 1; i <= 5; ++i) {
            this.addPartitions("tenant1_topicA", numBrokers * i, numBrokers);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
        }
        this.createTopic("tenant2_topicB", numBrokers, 3);
        for (i = 1; i <= 5; ++i) {
            this.addPartitions("tenant2_topicB", numBrokers * i, numBrokers);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
        }
        this.createTopic("tenant1_topicC", numBrokers, 3);
        i = 1;
        int p = numBrokers;
        while (i <= 10) {
            this.addPartitions("tenant1_topicC", p, numBrokers * i);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
            p += numBrokers * i++;
        }
        this.createTopic("tenant2_topicD", numBrokers, 3);
        i = 1;
        p = numBrokers;
        while (i <= 10) {
            this.addPartitions("tenant2_topicD", p, numBrokers * i);
            this.verifyAssignmentsAreBalanced(0, 0, 0);
            p += numBrokers * i++;
        }
    }

    @Test
    public void testRackUnawareTopicCreate() {
        this.addNodes(3, 0);
        this.createTopic("tenant1_topicA", 3, 2);
        this.verifyPartitionCountsOnNodes(true, 1, 1, 1);
        this.verifyPartitionCountsOnNodes(false, 1, 1, 1);
        this.createTopic("tenant2_topicA", 5, 2);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(true, 2, 3, 3);
        this.verifyAssignmentsAreBalanced("tenant2", 2);
        this.createTopic("tenant2_topicB", 4, 2);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(true, 4, 4, 4);
        this.verifyAssignmentsAreBalanced("tenant2", 2);
    }

    @Test
    public void testRackAwareTopicCreate() {
        this.addNodes(6, 3);
        this.createTopic("tenant1_topicA", 6, 3);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(true, 1, 1, 1, 1, 1, 1);
        this.verifyPartitionCountsOnNodes(false, 2, 2, 2, 2, 2, 2);
        this.createTopic("tenant2_topicA", 9, 3);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(true, 2, 2, 2, 3, 3, 3);
        this.verifyPartitionCountsOnNodes(false, 5, 5, 5, 5, 5, 5);
        this.verifyAssignmentsAreBalanced(1, 1, 1);
        this.createTopic("tenant2_topicB", 3, 3);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(false, 6, 6, 6, 6, 6, 6);
        this.verifyAssignmentsAreBalanced(0, 2, 2);
    }

    @Test
    public void testRackUnawarePartitionAdd() {
        this.addNodes(3, 0);
        this.createTopic("tenant1_topicA", 5, 2);
        this.addPartitions("tenant1_topicA", 5, 4);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(true, 3, 3, 3);
        this.verifyAssignmentsAreBalanced("tenant1", 2);
    }

    @Test
    public void testRackAwarePartitionAdd() {
        this.addNodes(6, 3);
        this.createTopic("tenant1_topicA", 9, 3);
        this.addPartitions("tenant1_topicA", 9, 3);
        this.verifyAssignments();
        this.verifyPartitionCountsOnNodes(true, 2, 2, 2, 2, 2, 2);
        this.verifyAssignmentsAreBalanced(0, 2, 2);
    }

    @Test
    public void testRackUnawareAssignmentVariation() {
        this.addNodes(5, 0);
        this.createTopic("tenant1_topicA", 5, 3);
        this.verifyPartitionCountsOnNodes(true, 1, 1, 1, 1, 1);
        this.verifyPartitionCountsOnNodes(false, 2, 2, 2, 2, 2);
        this.createTopic("tenant1_topicB", 10, 3);
        this.verifyPartitionCountsOnNodes(true, 3, 3, 3, 3, 3);
        this.verifyPartitionCountsOnNodes(false, 6, 6, 6, 6, 6);
        this.verifyAssignmentVariation();
    }

    @Test
    public void testRackAwareAssignmentVariation() {
        this.addNodes(6, 3);
        this.createTopic("tenant1_topicA", 6, 3);
        this.verifyPartitionCountsOnNodes(true, 1, 1, 1, 1, 1, 1);
        this.verifyPartitionCountsOnNodes(false, 2, 2, 2, 2, 2, 2);
        this.createTopic("tenant1_topicB", 12, 3);
        this.verifyPartitionCountsOnNodes(true, 3, 3, 3, 3, 3, 3);
        this.verifyPartitionCountsOnNodes(false, 6, 6, 6, 6, 6, 6);
        this.verifyAssignmentVariation();
    }

    private void verifyAssignmentVariation() {
        Cluster cluster = this.testCluster.cluster();
        HashSet allReplicas = new HashSet();
        int numPartitions = 0;
        for (String topic : cluster.topics()) {
            for (PartitionInfo partitionInfo : cluster.partitionsForTopic(topic)) {
                ArrayList<Integer> partitionReplicas = new ArrayList<Integer>();
                for (Node replica : partitionInfo.replicas()) {
                    partitionReplicas.add(replica.id());
                }
                allReplicas.add(partitionReplicas);
                ++numPartitions;
            }
        }
        Assert.assertTrue((String)("Too few replica combinations " + allReplicas.size() + " for " + numPartitions), ((double)allReplicas.size() >= 0.5 * (double)numPartitions ? 1 : 0) != 0);
    }

    @Test
    public void testUnavailableBrokers() {
        this.addNodes(6, 3);
        for (int i = 0; i < 5; ++i) {
            this.testCluster.setPartitionLeaders("tenant1_topicA", 1, 5, 10 + i);
        }
        Assert.assertEquals((long)6L, (long)this.testCluster.cluster().nodes().size());
        this.createTopic("tenant1_topicB", 6, 3);
        this.verifyPartitionCountsOnNodes(true, 1, 1, 1, 1, 1, 1);
        this.verifyPartitionCountsOnNodes(false, 2, 2, 2, 2, 2, 2);
    }

    @Test
    public void testRackUnawareAssignmentsAreBalanced() {
        this.addNodes(3, 0);
        this.verifyTopicCreation("tenant1");
        this.verifyPartitionAddition("tenant1");
    }

    @Test
    public void testRackAwareAssignmentsAreBalanced() {
        this.addNodes(6, 3);
        this.verifyTopicCreation("tenant1");
        this.verifyPartitionAddition("tenant1");
    }

    private void verifyTopicCreation(String tenant) {
        this.testCluster.setPartitionLeaders("otherTenant_topicA", 0, 5, 2);
        this.testCluster.setPartitionLeaders("otherTenant_topicB", 0, 2, 1);
        TenantPartitionAssignor.TopicInfo topicInfo = new TenantPartitionAssignor.TopicInfo(10, 3, 0);
        Map<String, TenantPartitionAssignor.TopicInfo> newTopics = Collections.singletonMap(tenant + "_topicA", topicInfo);
        Map<String, List<List<Integer>>> assignment = this.assignNewTopics(tenant, newTopics);
        this.verifyAssignmentsAreBalanced(tenant, assignment, 3);
        topicInfo = new TenantPartitionAssignor.TopicInfo(10, 2, 0);
        newTopics = Collections.singletonMap(tenant + "_topicB", topicInfo);
        assignment = this.assignNewTopics(tenant, newTopics);
        this.verifyAssignmentsAreBalanced("tenant1", assignment, 3);
        newTopics = new HashMap<String, TenantPartitionAssignor.TopicInfo>();
        newTopics.put(tenant + "_topicC", new TenantPartitionAssignor.TopicInfo(10, 1, 0));
        newTopics.put(tenant + "_topicD", new TenantPartitionAssignor.TopicInfo(8, 2, 0));
        newTopics.put(tenant + "_topicE", new TenantPartitionAssignor.TopicInfo(7, 3, 0));
        assignment = this.assignNewTopics(tenant, newTopics);
        this.verifyAssignmentsAreBalanced(tenant, assignment, 3);
    }

    private void verifyPartitionAddition(String tenant) {
        HashMap<String, Integer> newPartitions = new HashMap<String, Integer>();
        int partitionsToAdd = 2;
        int maxReplicationFactor = 0;
        for (String topic : this.testCluster.cluster().topics()) {
            if (!topic.startsWith(tenant)) continue;
            List partitionInfos = this.testCluster.cluster().partitionsForTopic(topic);
            int firstNewPartition = partitionInfos.size();
            newPartitions.put(topic, firstNewPartition + partitionsToAdd);
            ++partitionsToAdd;
            int replicationFactor = ((PartitionInfo)partitionInfos.get(0)).replicas().length;
            if (replicationFactor <= maxReplicationFactor) continue;
            maxReplicationFactor = replicationFactor;
        }
        Assert.assertFalse((String)"No tenant topics", (boolean)newPartitions.isEmpty());
        Map<String, List<List<Integer>>> assignment = this.assignNewPartitions(newPartitions);
        this.verifyAssignmentsAreBalanced(tenant, assignment, maxReplicationFactor);
    }

    @Test
    public void testRackUnawareTopicCreateDelete() {
        this.addNodes(6, 0);
        this.verifyTopicCreateDelete();
    }

    @Test
    public void testRackAwareTopicCreateDelete() {
        this.addNodes(6, 3);
        this.verifyTopicCreateDelete();
    }

    private void verifyTopicCreateDelete() {
        Random random = new Random();
        List<String> tenants = Arrays.asList("tenant1", "tenant2", "tenant3");
        ArrayList<String> topics = new ArrayList<String>();
        for (int i = 0; i < 100; ++i) {
            String tenant = tenants.get(random.nextInt(tenants.size()));
            if (i > 5 && i % 5 == 0) {
                this.testCluster.deleteTopic((String)topics.remove(random.nextInt(topics.size())));
                continue;
            }
            String topic = tenant + "_Topic" + i;
            int partitions = random.nextInt(20);
            this.createTopic(topic, partitions == 0 ? 1 : partitions, 3);
            topics.add(topic);
        }
        block3: for (String tenant : tenants) {
            int maxRetries = 100;
            for (int i = 0; i < maxRetries; ++i) {
                try {
                    this.verifyAssignmentsAreBalanced(tenant, 3);
                    continue block3;
                }
                catch (Throwable t) {
                    if (i == maxRetries - 1) {
                        throw t;
                    }
                    this.createTopic(tenant + "_TopicA" + i, 2, 3);
                    continue;
                }
            }
        }
    }

    @Test
    public void testRackAwareOnlyIfAllBrokersHaveRack() {
        TestCluster cluster1 = new TestCluster();
        cluster1.addNode(1, null);
        Assert.assertFalse((boolean)new ClusterMetadata("tenant", cluster1.cluster()).rackAware());
        cluster1.addNode(2, null);
        Assert.assertFalse((boolean)new ClusterMetadata("tenant", cluster1.cluster()).rackAware());
        cluster1.addNode(3, "rack1");
        Assert.assertFalse((boolean)new ClusterMetadata("tenant", cluster1.cluster()).rackAware());
        TestCluster cluster2 = new TestCluster();
        cluster2.addNode(1, "rack1");
        Assert.assertTrue((boolean)new ClusterMetadata("tenant", cluster2.cluster()).rackAware());
        this.testCluster.addNode(2, "rack2");
        Assert.assertTrue((boolean)new ClusterMetadata("tenant", cluster2.cluster()).rackAware());
        cluster2.addNode(3, null);
        Assert.assertFalse((boolean)new ClusterMetadata("tenant", cluster2.cluster()).rackAware());
    }

    @Test
    public void testRackAlternatedBrokerList() {
        int i;
        int numRacks = 3;
        int numBrokers = numRacks * 4;
        HashMap<Integer, String> brokerRacks = new HashMap<Integer, String>();
        ArrayList<Integer> brokerList = new ArrayList<Integer>();
        for (i = 0; i < numBrokers; ++i) {
            brokerRacks.put(i, "rack" + i % numRacks);
            brokerList.add(i);
        }
        Assert.assertEquals(brokerList, (Object)TenantPartitionAssignor.rackAlternatedBrokerList(brokerList, brokerRacks, Collections.emptyList()));
        Assert.assertEquals(brokerList, (Object)TenantPartitionAssignor.rackAlternatedBrokerList(Arrays.asList(0, 3, 6, 9, 1, 4, 7, 10, 2, 5, 8, 11), brokerRacks, Collections.emptyList()));
        Collections.reverse(brokerList);
        Assert.assertEquals(brokerList, (Object)TenantPartitionAssignor.rackAlternatedBrokerList(brokerList, brokerRacks, Collections.emptyList()));
        for (i = 0; i < 5; ++i) {
            Collections.shuffle(brokerList);
            List alternateList = TenantPartitionAssignor.rackAlternatedBrokerList(brokerList, brokerRacks, Collections.emptyList());
            List rackOrder = alternateList.subList(0, numRacks).stream().map(b -> b % numRacks).collect(Collectors.toList());
            ArrayList rackAssignment = new ArrayList();
            for (int j = 0; j < numRacks; ++j) {
                rackAssignment.add(new ArrayList());
            }
            String errorMessage = "Unexpected assignment for " + brokerList + " : " + alternateList;
            for (int j = 0; j < numBrokers; ++j) {
                int expectedRack = (Integer)rackOrder.get(j % numRacks);
                int brokerId = (Integer)alternateList.get(j);
                Assert.assertEquals((String)errorMessage, (long)expectedRack, (long)(brokerId % numRacks));
                ((List)rackAssignment.get(expectedRack)).add(brokerId);
            }
            rackAssignment.forEach(list -> {
                int prevIndex = -1;
                Iterator iterator = list.iterator();
                while (iterator.hasNext()) {
                    int broker = (Integer)iterator.next();
                    int curIndex = brokerList.indexOf(broker);
                    Assert.assertTrue((String)errorMessage, (curIndex > prevIndex ? 1 : 0) != 0);
                    prevIndex = curIndex;
                }
            });
            ArrayList<Integer> expectedRackOrder = new ArrayList<Integer>();
            for (int j = 0; j < numBrokers; ++j) {
                int rack = (Integer)brokerList.get(j) % numRacks;
                if (expectedRackOrder.contains(rack)) continue;
                expectedRackOrder.add(rack);
                if (expectedRackOrder.size() == numRacks) break;
            }
            Assert.assertEquals((String)errorMessage, expectedRackOrder, rackOrder);
        }
    }

    private void createTopic(String topic, int partitions, int replicationFactor) {
        String tenant = topic.substring(0, topic.indexOf("_"));
        TenantPartitionAssignor.TopicInfo topicInfo = new TenantPartitionAssignor.TopicInfo(partitions, (short)replicationFactor, 0);
        Map<String, TenantPartitionAssignor.TopicInfo> topicInfos = Collections.singletonMap(topic, topicInfo);
        Map<String, List<List<Integer>>> assignment = this.assignNewTopics(tenant, topicInfos);
        this.testCluster.createPartitions(topic, 0, assignment.get(topic));
    }

    private void addPartitions(String topic, int firstNewPartition, int newPartitions) {
        int totalPartitions = firstNewPartition + newPartitions;
        Map<String, Integer> partitions = Collections.singletonMap(topic, totalPartitions);
        Map<String, List<List<Integer>>> assignment = this.assignNewPartitions(partitions);
        this.testCluster.createPartitions(topic, firstNewPartition, assignment.get(topic));
    }

    private void addNodes(int count, int racks) {
        for (int i = 0; i < count; ++i) {
            String rack = racks == 0 ? null : "rack" + i % racks;
            this.testCluster.addNode(i, rack);
        }
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        if (racks > 0) {
            Assert.assertTrue((boolean)this.testCluster.rackAware());
        }
    }

    private Map<String, List<List<Integer>>> assignNewTopics(String tenant, Map<String, TenantPartitionAssignor.TopicInfo> topicInfos) {
        Map assignments = this.partitionAssignor.assignPartitionsForNewTopics(tenant, topicInfos);
        Assert.assertEquals((long)topicInfos.size(), (long)assignments.size());
        for (Map.Entry<String, TenantPartitionAssignor.TopicInfo> entry : topicInfos.entrySet()) {
            String topic = entry.getKey();
            TenantPartitionAssignor.TopicInfo topicInfo = entry.getValue();
            List assignment = (List)assignments.get(topic);
            Assert.assertEquals((long)topicInfo.totalPartitions, (long)assignment.size());
            this.testCluster.createPartitions(topic, 0, assignment);
        }
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        return assignments;
    }

    private Map<String, List<List<Integer>>> assignNewPartitions(Map<String, Integer> partitionInfos) {
        Map assignments = this.partitionAssignor.assignPartitionsForExistingTopics("tenant1", partitionInfos);
        Assert.assertEquals((long)partitionInfos.size(), (long)assignments.size());
        for (Map.Entry<String, Integer> entry : partitionInfos.entrySet()) {
            String topic = entry.getKey();
            int totalPartitions = entry.getValue();
            int firstNewPartition = this.testCluster.cluster().partitionsForTopic(topic).size();
            int newPartitions = totalPartitions - firstNewPartition;
            List assignment = (List)assignments.get(topic);
            Assert.assertEquals((long)newPartitions, (long)assignment.size());
            this.testCluster.createPartitions(topic, firstNewPartition, assignment);
        }
        this.partitionAssignor.updateClusterMetadata(this.testCluster.cluster());
        return assignments;
    }

    private void verifyAssignmentsAreBalanced(String tenant, Map<String, List<List<Integer>>> assignments, int maxReplicationFactor) {
        Map<Node, Integer> leaders = this.testCluster.partitionCountByNode(tenant, true);
        Map<Node, Integer> followers = this.testCluster.partitionCountByNode(tenant, false);
        for (List<List<Integer>> assignment : assignments.values()) {
            for (List<Integer> replicas : assignment) {
                for (int i = 0; i < replicas.size(); ++i) {
                    Node node = this.testCluster.cluster().nodeById(replicas.get(i).intValue());
                    if (i == 0) {
                        leaders.put(node, leaders.get(node) + 1);
                        continue;
                    }
                    followers.put(node, leaders.get(node) + 1);
                }
            }
        }
        int maxReplicaDiff = maxReplicationFactor - 1;
        if (this.testCluster.rackAware()) {
            maxReplicaDiff *= this.testCluster.racks().size();
        }
        this.verifyAssignmentsAreBalanced(tenant, maxReplicaDiff);
    }

    private void verifyAssignmentsAreBalanced(String tenant, int maxReplicaDiff) {
        Collection<Integer> leaders = this.testCluster.partitionCountByNode(tenant, true).values();
        Collection<Integer> followers = this.testCluster.partitionCountByNode(tenant, false).values();
        int maxLeaderImbalance = this.testCluster.rackAware() ? this.testCluster.racks().size() : 1;
        Assert.assertTrue((String)("Leaders not balanced " + leaders), (Collections.max(leaders) - Collections.min(leaders) <= maxLeaderImbalance ? 1 : 0) != 0);
        Assert.assertTrue((String)("Follower replicas not balanced " + followers), (Collections.max(followers) - Collections.min(followers) <= maxReplicaDiff ? 1 : 0) != 0);
    }

    private void verifyAssignmentsAreBalanced(int ... maxDiff) {
        for (int i = 0; i < maxDiff.length; ++i) {
            List<Integer> replicaCounts = this.followerCountsByNode(Optional.of(i));
            Collections.sort(replicaCounts);
            Assert.assertTrue((String)("Replicas not balanced for replica #" + i + " : " + replicaCounts), (replicaCounts.get(replicaCounts.size() - 1) - replicaCounts.get(0) <= maxDiff[i] ? 1 : 0) != 0);
        }
    }

    private List<Integer> leaderCountsByNode() {
        Cluster cluster = this.testCluster.cluster();
        return cluster.nodes().stream().map(node -> cluster.partitionsForNode(node.id()).size()).collect(Collectors.toList());
    }

    private List<Integer> followerCountsByNode(Optional<Integer> replicaIndex) {
        Cluster cluster = this.testCluster.cluster();
        Map partitionCounts = cluster.nodes().stream().collect(Collectors.toMap(Function.identity(), n -> 0));
        for (String topic : cluster.topics()) {
            for (PartitionInfo partitionInfo : cluster.partitionsForTopic(topic)) {
                Node[] replicas = partitionInfo.replicas();
                replicaIndex.ifPresent(r -> partitionCounts.put(replicas[r], (Integer)partitionCounts.get(replicas[r]) + 1));
                if (replicaIndex.isPresent()) continue;
                for (int i = 1; i < replicas.length; ++i) {
                    partitionCounts.put(replicas[i], partitionCounts.get(replicas[i]) + 1);
                }
            }
        }
        return new ArrayList<Integer>(partitionCounts.values());
    }

    private void verifyPartitionCountsOnNodes(boolean leader, Integer ... sortedPartitionCounts) {
        List<Integer> expectedPartitionCounts = Arrays.asList(sortedPartitionCounts);
        List<Integer> actualPartitionCounts = leader ? this.leaderCountsByNode() : this.followerCountsByNode(Optional.empty());
        Collections.sort(actualPartitionCounts);
        Assert.assertEquals(expectedPartitionCounts, actualPartitionCounts);
    }

    private void verifyAssignments() {
        Cluster cluster = this.testCluster.cluster();
        for (String topic : cluster.topics()) {
            for (PartitionInfo partitionInfo : cluster.partitionsForTopic(topic)) {
                HashSet<Integer> replicas = new HashSet<Integer>();
                HashSet<String> replicaRacks = new HashSet<String>();
                for (Node node : partitionInfo.replicas()) {
                    replicas.add(node.id());
                    if (node.rack() == null) continue;
                    replicaRacks.add(node.rack());
                }
                Assert.assertEquals((long)partitionInfo.replicas().length, (long)replicas.size());
                if (!this.testCluster.rackAware()) continue;
                Assert.assertEquals((long)partitionInfo.replicas().length, (long)replicaRacks.size());
            }
        }
    }

    private void verifyAssignment(List<List<Integer>> expected, List<PartitionInfo> actual) {
        Assert.assertEquals(expected, this.partitionsToAssignment(actual));
    }

    private void verifyPartitionsAssignment(List<PartitionInfo> expected, List<PartitionInfo> actual) {
        Assert.assertEquals(this.partitionsToAssignment(expected), this.partitionsToAssignment(actual));
    }

    private List<List<Integer>> partitionsToAssignment(List<PartitionInfo> partitions) {
        ArrayList<List<Integer>> assignment = new ArrayList<List<Integer>>(partitions.size());
        for (int i = 0; i < partitions.size(); ++i) {
            assignment.add(null);
        }
        partitions.forEach(partitionInfo -> assignment.set(partitionInfo.partition(), Arrays.stream(partitionInfo.replicas()).map(Node::id).collect(Collectors.toList())));
        return assignment;
    }
}

