package io.cequence.openaiscala.gemini.domain

import io.cequence.openaiscala.OpenAIScalaClientException
import io.cequence.wsclient.domain.EnumValue

case class Content(
  parts: Seq[Part],
  role: Option[ChatRole]
)

object Content {
  def apply(parts: Part*): Content = Content(parts, None)
  def apply(
    role: ChatRole,
    parts: Part*
  ): Content = Content(parts, Some(role))
  def textPart(
    text: String,
    role: ChatRole
  ): Content = apply(role, Part.Text(text))
}

sealed trait PartPrefix extends EnumValue

object PartPrefix {
  case object text extends PartPrefix
  case object inlineData extends PartPrefix
  case object functionCall extends PartPrefix
  case object functionResponse extends PartPrefix
  case object fileData extends PartPrefix
  case object executableCode extends PartPrefix
  case object codeExecutionResult extends PartPrefix

  def values: Seq[PartPrefix] = Seq(
    text,
    inlineData,
    functionCall,
    functionResponse,
    fileData,
    executableCode,
    codeExecutionResult
  )

  def of(value: String): PartPrefix = values.find(_.toString() == value).getOrElse {
    throw new OpenAIScalaClientException(s"Unknown partPrefix: $value")
  }
}

sealed trait Part {
  val prefix: PartPrefix
}

object Part {

  /**
   * Prefix: none
   *
   * @param text
   */
  case class Text(text: String) extends Part {
    override val prefix: PartPrefix = PartPrefix.text
  }

  /**
   * Prefix: inlineData
   *
   * Blob of data. Inline media bytes.
   * @param mimeType
   *   The IANA standard MIME type of the source data. Examples: - image/png - image/jpeg If an
   *   unsupported MIME type is provided, an error will be returned. For a complete list of
   *   supported types, see Supported file formats.
   * @param data
   *   Raw bytes for media formats. A base64-encoded string.
   */
  case class InlineData(
    mimeType: String,
    data: String
  ) extends Part {
    override val prefix: PartPrefix = PartPrefix.inlineData
  }

  /**
   * Prefix: functionCall
   *
   * A predicted FunctionCall returned from the model that contains a string representing the
   * FunctionDeclaration.name with the arguments and their values.
   *
   * @param id
   *   The unique id of the function call. If populated, the client to execute the functionCall
   *   and return the response with the matching id.
   * @param name
   *   Required. The name of the function to call. Must be a-z, A-Z, 0-9, or contain
   *   underscores and dashes, with a maximum length of 63.
   * @param args
   *   Optional. The function parameters and values in JSON object format.
   */
  case class FunctionCall(
    id: Option[String],
    name: String,
    args: Map[String, Any] = Map.empty
  ) extends Part {
    override val prefix: PartPrefix = PartPrefix.functionCall
  }

  /**
   * Prefix: functionResponse
   *
   * The result output of a FunctionCall that contains a string representing the
   * FunctionDeclaration.name and a structured JSON object containing any output from the
   * function is used as context to the model.
   *
   * @param id
   *   The id of the function call this response is for. Populated by the client to match the
   *   corresponding function call id.
   * @param name
   *   The name of the function to call. Must be a-z, A-Z, 0-9, or contain underscores and
   *   dashes, with a maximum length of 63.
   * @param response
   *   The function response in JSON object format.
   */
  case class FunctionResponse(
    id: Option[String],
    name: String,
    response: Map[String, Any]
  ) extends Part {
    override val prefix: PartPrefix = PartPrefix.functionResponse
  }

  /**
   * Prefix: fileData
   *
   * URI based data.
   * @param mimeType
   *   Optional. The IANA standard MIME type of the source data.
   * @param fileUri
   *   Required. URI.
   */
  case class FileData(
    mimeType: Option[String],
    fileUri: String
  ) extends Part {
    override val prefix: PartPrefix = PartPrefix.fileData
  }

  /**
   * Prefix: executableCode
   *
   * Code generated by the model that is meant to be executed, and the result returned to the
   * model. Only generated when using the CodeExecution tool, in which the code will be
   * automatically executed, and a corresponding CodeExecutionResult will also be generated.
   *
   * @param language
   *   Required. Programming language of the code.
   * @param code
   *   Required. The code to be executed.
   */
  case class ExecutableCode(
    language: String, // TODO: enum
    code: String
  ) extends Part {
    override val prefix: PartPrefix = PartPrefix.executableCode
  }

  /**
   * Prefix: codeExecutionResult
   *
   * @param outcome
   *   Required. Outcome of the code execution.
   * @param output
   *   Optional. Contains stdout when code execution is successful, stderr or other description
   *   otherwise.
   */
  case class CodeExecutionResult(
    outcome: String, // TODO: enum
    output: Option[String]
  ) extends Part {
    override val prefix: PartPrefix = PartPrefix.codeExecutionResult
  }
}
