/*
 * Decompiled with CFR 0.152.
 */
package io.castled.warehouses.connectors.bigquery;

import com.google.api.gax.paging.Page;
import com.google.cloud.RetryOption;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.DatasetId;
import com.google.cloud.bigquery.DatasetInfo;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.Job;
import com.google.cloud.bigquery.LegacySQLTypeName;
import com.google.common.collect.Lists;
import io.castled.exceptions.CastledException;
import io.castled.exceptions.CastledRuntimeException;
import io.castled.schema.SchemaConstants;
import io.castled.schema.models.DecimalSchema;
import io.castled.schema.models.RecordSchema;
import io.castled.schema.models.Schema;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.threeten.bp.Duration;

public class BigQueryUtils {
    private static final Logger log = LoggerFactory.getLogger(BigQueryUtils.class);
    public static final int MAX_NUMERIC_SCALE = 9;
    public static final int MAX_NUMERIC_PRECISION = 38;
    public static final int MAX_BIG_NUMERIC_SCALE = 38;
    public static final int MAX_BIG_NUMERIC_PRECISION = 76;

    public static RecordSchema bqSchemaToConnectSchema(com.google.cloud.bigquery.Schema bqSchema) {
        RecordSchema.Builder schemaBuilder = RecordSchema.builder();
        for (Field field : bqSchema.getFields()) {
            Schema schema = BigQueryUtils.bqFieldToConnectSchema(field);
            if (schema == null) continue;
            schemaBuilder.put(field.getName(), schema);
        }
        return schemaBuilder.build();
    }

    public static List<String> fieldNames(com.google.cloud.bigquery.Schema schema) {
        return schema.getFields().stream().map(Field::getName).collect(Collectors.toList());
    }

    public static String parseFieldValue(Object fieldValue, LegacySQLTypeName sqlTypeName) {
        if (sqlTypeName.equals((Object)LegacySQLTypeName.TIMESTAMP)) {
            return BigQueryUtils.parseTimestampFieldValue((String)fieldValue);
        }
        return fieldValue.toString();
    }

    public static Dataset getOrCreateDataset(String datasetName, BigQuery bigQuery, String location) {
        Dataset dataset = bigQuery.getDataset(DatasetId.of((String)datasetName), new BigQuery.DatasetOption[0]);
        if (dataset == null) {
            return bigQuery.create(DatasetInfo.newBuilder((String)datasetName).setLocation(location).build(), new BigQuery.DatasetOption[0]);
        }
        return dataset;
    }

    public static void runJob(Job job) throws Exception {
        Job processedJob = job.waitFor(new RetryOption[]{RetryOption.initialRetryDelay((Duration)Duration.ofSeconds((long)1L)), RetryOption.totalTimeout((Duration)Duration.ofMinutes((long)40L))});
        if (processedJob.getStatus().getExecutionErrors() != null) {
            throw new CastledException(String.format("Job %s failed to run with error %s", processedJob.getJobId().getJob(), processedJob.getStatus().getExecutionErrors().toString()));
        }
    }

    public static List<String> listTables(String datasetName, BigQuery bigQuery) {
        ArrayList allTables = Lists.newArrayList();
        try {
            Page tablePage = bigQuery.listTables(DatasetId.of((String)datasetName), new BigQuery.TableListOption[0]);
            do {
                tablePage.iterateAll().forEach(table -> allTables.add(table.getTableId().getTable()));
            } while ((tablePage = tablePage.getNextPage()) != null);
            return allTables;
        }
        catch (BigQueryException e) {
            if (e.getCode() == 404) {
                return allTables;
            }
            throw e;
        }
    }

    public static String parseTimestampFieldValue(String fieldValue) {
        String[] valueTokens = fieldValue.split("\\.");
        if (valueTokens.length != 2) {
            throw new CastledRuntimeException(String.format("Invalid timestamp field value string %s", fieldValue));
        }
        long epochSeconds = Long.parseLong(valueTokens[0]);
        long nanos = Double.valueOf((double)Integer.parseInt(valueTokens[1]) * Math.pow(10.0, 9 - valueTokens[1].length())).longValue();
        ZonedDateTime zonedDateTime = ZonedDateTime.ofInstant(Instant.ofEpochSecond(epochSeconds, nanos), ZoneId.of("UTC"));
        return zonedDateTime.format(DateTimeFormatter.ISO_DATE_TIME);
    }

    public static Schema bqFieldToConnectSchema(Field field) {
        Schema schema = BigQueryUtils.buildSchema(field);
        if (schema != null && field.getMode() == Field.Mode.NULLABLE) {
            schema.setOptional(true);
        }
        return schema;
    }

    private static Schema buildSchema(Field field) {
        if (field.getType().name().equals("BIGNUMERIC")) {
            return DecimalSchema.builder().scale(38).precision(76).build();
        }
        switch (field.getType().getStandardType()) {
            case STRING: {
                return SchemaConstants.STRING_SCHEMA;
            }
            case TIME: {
                return SchemaConstants.TIME_SCHEMA;
            }
            case BOOL: {
                return SchemaConstants.BOOL_SCHEMA;
            }
            case BYTES: {
                return SchemaConstants.BYTES_SCHEMA;
            }
            case INT64: {
                return SchemaConstants.LONG_SCHEMA;
            }
            case FLOAT64: {
                return SchemaConstants.DOUBLE_SCHEMA;
            }
            case NUMERIC: {
                return DecimalSchema.builder().scale(9).precision(38).build();
            }
            case DATE: {
                return SchemaConstants.DATE_SCHEMA;
            }
            case DATETIME: {
                return SchemaConstants.TIMESTAMP_SCHEMA;
            }
            case TIMESTAMP: {
                return SchemaConstants.ZONED_TIMESTAMP_SCHEMA;
            }
        }
        log.error("Unsupported BigQuery field type {}", (Object)field.getType().getStandardType());
        return null;
    }
}

