/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.scheduler.ConcurrencyControl;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.future.CompletableActorFuture;
import io.camunda.zeebe.snapshots.CRC32CChecksumProvider;
import io.camunda.zeebe.snapshots.ImmutableChecksumsSFV;
import io.camunda.zeebe.snapshots.MutableChecksumsSFV;
import io.camunda.zeebe.snapshots.PersistableSnapshot;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.PersistedSnapshotListener;
import io.camunda.zeebe.snapshots.SnapshotException;
import io.camunda.zeebe.snapshots.SnapshotId;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedReceivedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotId;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotMetadata;
import io.camunda.zeebe.snapshots.impl.FileBasedTransientSnapshot;
import io.camunda.zeebe.snapshots.impl.SnapshotChecksum;
import io.camunda.zeebe.snapshots.impl.SnapshotMetrics;
import io.camunda.zeebe.util.CloseableSilently;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileBasedSnapshotStoreImpl {
    public static final int VERSION = 1;
    public static final String SNAPSHOTS_DIRECTORY = "snapshots";
    public static final String METADATA_FILE_NAME = "zeebe.metadata";
    public static final String SNAPSHOTS_BOOTSTRAP_DIRECTORY = "bootstrap-snapshots";
    public static final String CHECKSUM_SUFFIX = ".checksum";
    private static final Logger LOGGER = LoggerFactory.getLogger(FileBasedSnapshotStoreImpl.class);
    private final int brokerId;
    private final Path snapshotsDirectory;
    private final Path bootstrapSnapshotsDirectory;
    private final Set<PersistedSnapshotListener> listeners = new CopyOnWriteArraySet<PersistedSnapshotListener>();
    private final SnapshotMetrics metrics;
    private final CRC32CChecksumProvider checksumProvider;
    private final ConcurrencyControl actor;
    private final AtomicReference<FileBasedSnapshot> currentSnapshot = new AtomicReference();
    private final AtomicReference<FileBasedSnapshot> bootstrapSnapshot = new AtomicReference();
    private final Set<PersistableSnapshot> pendingSnapshots = new HashSet<PersistableSnapshot>();
    private final Set<FileBasedSnapshot> availableSnapshots = new HashSet<FileBasedSnapshot>();

    public FileBasedSnapshotStoreImpl(int brokerId, Path root, CRC32CChecksumProvider checksumProvider, ConcurrencyControl actor, SnapshotMetrics metrics) {
        this.brokerId = brokerId;
        this.actor = Objects.requireNonNull(actor);
        this.metrics = Objects.requireNonNull(metrics);
        this.checksumProvider = Objects.requireNonNull(checksumProvider);
        this.snapshotsDirectory = root.resolve(SNAPSHOTS_DIRECTORY);
        this.bootstrapSnapshotsDirectory = root.resolve(SNAPSHOTS_BOOTSTRAP_DIRECTORY);
        try {
            FileUtil.ensureDirectoryExists((Path)this.snapshotsDirectory);
            FileUtil.ensureDirectoryExists((Path)this.bootstrapSnapshotsDirectory);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to create snapshot directories", e);
        }
    }

    public void start() {
        this.setLatestSnapshot(this.loadLatestSnapshot(this.snapshotsDirectory));
    }

    public void close() {
        LOGGER.debug("Closing snapshot store {}", (Object)this.snapshotsDirectory);
        this.listeners.clear();
        this.deleteBootstrapSnapshotsInternal();
    }

    private FileBasedSnapshot loadLatestSnapshot(Path snapshotDirectory) {
        FileBasedSnapshot latestPersistedSnapshot = null;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(snapshotDirectory, x$0 -> Files.isDirectory(x$0, new LinkOption[0]));){
            for (Path path : stream) {
                FileBasedSnapshot snapshot = this.collectSnapshot(path);
                if (snapshot == null || latestPersistedSnapshot != null && snapshot.getSnapshotId().compareTo(latestPersistedSnapshot.getSnapshotId()) < 0) continue;
                latestPersistedSnapshot = snapshot;
            }
            if (latestPersistedSnapshot != null) {
                this.cleanupSnapshotDirectory(snapshotDirectory, latestPersistedSnapshot);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return latestPersistedSnapshot;
    }

    private void cleanupSnapshotDirectory(Path snapshotDirectory, FileBasedSnapshot latestPersistedSnapshot) throws IOException {
        Path latestChecksumFile = latestPersistedSnapshot.getChecksumPath();
        Path latestDirectory = latestPersistedSnapshot.getDirectory();
        try (DirectoryStream<Path> paths = Files.newDirectoryStream(snapshotDirectory, p -> !p.equals(latestDirectory) && !p.equals(latestChecksumFile));){
            LOGGER.debug("Deleting snapshots other than {}", (Object)latestPersistedSnapshot.getId());
            paths.forEach(p -> {
                try {
                    LOGGER.debug("Deleting {}", p);
                    FileUtil.deleteFolderIfExists((Path)p);
                }
                catch (IOException e) {
                    LOGGER.warn("Unable to delete {}", p, (Object)e);
                }
            });
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private FileBasedSnapshot collectSnapshot(Path path) throws IOException {
        FileBasedSnapshotId.SnapshotParseResult optionalMeta = FileBasedSnapshotId.ofPath(path);
        if (optionalMeta instanceof FileBasedSnapshotId.SnapshotParseResult.Invalid) {
            Exception cause;
            FileBasedSnapshotId.SnapshotParseResult.Invalid invalid = (FileBasedSnapshotId.SnapshotParseResult.Invalid)optionalMeta;
            try {
                Exception exception;
                cause = exception = invalid.exception();
            }
            catch (Throwable throwable) {
                throw new MatchException(throwable.toString(), throwable);
            }
            LOGGER.warn("Failed to parse snapshot id", (Throwable)cause);
            return null;
        }
        FileBasedSnapshotId snapshotId = optionalMeta.getOrThrow();
        Path checksumPath = this.buildSnapshotsChecksumPath(path, snapshotId);
        if (!Files.exists(checksumPath, new LinkOption[0])) {
            LOGGER.debug("Snapshot {} does not have a checksum file, which most likely indicates a partial write (e.g. crash during move), and will be deleted", (Object)path);
            try {
                FileUtil.deleteFolder((Path)path);
                return null;
            }
            catch (Exception e) {
                LOGGER.debug("Failed to delete partial snapshot {}", (Object)path, (Object)e);
            }
            return null;
        }
        try {
            ImmutableChecksumsSFV expectedChecksum = SnapshotChecksum.read(checksumPath);
            MutableChecksumsSFV actualChecksum = SnapshotChecksum.calculateWithProvidedChecksums(path, this.checksumProvider);
            if (!actualChecksum.sameChecksums(expectedChecksum)) {
                LOGGER.warn("Expected snapshot {} to have checksums {}, but the actual checksums are {}; the snapshot is most likely corrupted. The startup will fail if there is no other valid snapshot and the log has been compacted.", new Object[]{path, expectedChecksum.getChecksums(), actualChecksum.getChecksums()});
                return null;
            }
            FileBasedSnapshotMetadata metadata = this.collectMetadata(path, snapshotId);
            return new FileBasedSnapshot(path, checksumPath, actualChecksum, snapshotId, metadata, this::onSnapshotDeleted, this.actor);
        }
        catch (Exception e) {
            LOGGER.warn("Could not load snapshot in {}", (Object)path, (Object)e);
            return null;
        }
    }

    private FileBasedSnapshotMetadata collectMetadata(Path path, FileBasedSnapshotId snapshotId) throws IOException {
        Path metadataPath = path.resolve(METADATA_FILE_NAME);
        if (metadataPath.toFile().exists()) {
            byte[] encodedMetadata = Files.readAllBytes(metadataPath);
            return FileBasedSnapshotMetadata.decode(encodedMetadata);
        }
        return new FileBasedSnapshotMetadata(1, snapshotId.getProcessedPosition(), snapshotId.getExportedPosition(), Long.MAX_VALUE);
    }

    public boolean hasSnapshotId(String id) {
        Optional<PersistedSnapshot> optLatestSnapshot = this.getLatestSnapshot();
        if (optLatestSnapshot.isPresent()) {
            PersistedSnapshot snapshot = optLatestSnapshot.get();
            return snapshot.getPath().getFileName().toString().equals(id);
        }
        return false;
    }

    public Optional<PersistedSnapshot> getLatestSnapshot() {
        return Optional.ofNullable((PersistedSnapshot)this.currentSnapshot.get());
    }

    private void setLatestSnapshot(FileBasedSnapshot snapshot) {
        this.currentSnapshot.set(snapshot);
        if (snapshot != null) {
            this.availableSnapshots.add(snapshot);
        }
    }

    public ActorFuture<Set<PersistedSnapshot>> getAvailableSnapshots() {
        return this.actor.call(() -> Collections.unmodifiableSet(this.availableSnapshots));
    }

    public ActorFuture<Long> getCompactionBound() {
        return this.actor.call(() -> this.availableSnapshots.stream().map(PersistedSnapshot::getCompactionBound).min(Long::compareTo).orElse(0L));
    }

    public ActorFuture<Void> abortPendingSnapshots() {
        CompletableActorFuture abortFuture = new CompletableActorFuture();
        this.actor.run(() -> {
            List<ActorFuture> abortedAll = this.pendingSnapshots.stream().map(PersistableSnapshot::abort).toList();
            this.actor.runOnCompletion(abortedAll, error -> {
                if (error == null) {
                    abortFuture.complete(null);
                } else {
                    abortFuture.completeExceptionally(error);
                }
            });
        });
        return abortFuture;
    }

    public ActorFuture<Boolean> addSnapshotListener(PersistedSnapshotListener listener) {
        return this.actor.call(() -> this.listeners.add(listener));
    }

    public ActorFuture<Boolean> removeSnapshotListener(PersistedSnapshotListener listener) {
        return this.actor.call(() -> this.listeners.remove(listener));
    }

    public long getCurrentSnapshotIndex() {
        return this.getLatestSnapshot().map(PersistedSnapshot::getIndex).orElse(0L);
    }

    public ActorFuture<Void> delete() {
        return this.actor.call(() -> {
            this.currentSnapshot.set(null);
            try {
                LOGGER.debug("DELETE FOLDER {}", (Object)this.snapshotsDirectory);
                FileUtil.deleteFolder((Path)this.snapshotsDirectory);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            return null;
        });
    }

    public Path getPath() {
        return this.snapshotsDirectory;
    }

    public ActorFuture<FileBasedReceivedSnapshot> newReceivedSnapshot(String snapshotId) {
        CompletableActorFuture newSnapshotFuture = new CompletableActorFuture();
        FileBasedSnapshotId parsedSnapshotId = FileBasedSnapshotId.ofFileName(snapshotId).getOrThrow();
        this.actor.run(() -> {
            Path directory = this.buildSnapshotDirectory(parsedSnapshotId, false);
            try {
                this.checkAndCleanupExistingDirectory(snapshotId, parsedSnapshotId, directory);
                this.createReceivedSnapshot(parsedSnapshotId, directory, (CompletableActorFuture<FileBasedReceivedSnapshot>)newSnapshotFuture);
            }
            catch (Exception e) {
                newSnapshotFuture.completeExceptionally((Throwable)e);
            }
        });
        return newSnapshotFuture;
    }

    private void createReceivedSnapshot(FileBasedSnapshotId parsedSnapshotId, Path directory, CompletableActorFuture<FileBasedReceivedSnapshot> newSnapshotFuture) {
        FileBasedReceivedSnapshot newPendingSnapshot = new FileBasedReceivedSnapshot(parsedSnapshotId, directory, this, this.actor);
        this.addPendingSnapshot(newPendingSnapshot);
        newSnapshotFuture.complete((Object)newPendingSnapshot);
    }

    private void checkAndCleanupExistingDirectory(String snapshotId, FileBasedSnapshotId parsedSnapshotId, Path directory) {
        if (directory.toFile().exists()) {
            if (!this.buildSnapshotsChecksumPath(directory, parsedSnapshotId).toFile().exists()) {
                try {
                    FileUtil.deleteFolderIfExists((Path)directory);
                }
                catch (IOException e) {
                    throw new IllegalStateException("Expected to delete pending received snapshot, but failed.", e);
                }
            } else {
                throw new SnapshotException.SnapshotAlreadyExistsException(String.format("Expected to receive snapshot with id %s, but was already persisted. This shouldn't happen.", snapshotId));
            }
        }
    }

    public Either<SnapshotException, TransientSnapshot> newTransientSnapshot(long index, long term, long processedPosition, long exportedPosition, boolean forceSnapshot) {
        Path directory;
        FileBasedSnapshotId newSnapshotId = new FileBasedSnapshotId(index, term, processedPosition, exportedPosition, this.brokerId);
        FileBasedSnapshot currentSnapshot = this.currentSnapshot.get();
        if (!forceSnapshot && currentSnapshot != null && currentSnapshot.getSnapshotId().compareTo(newSnapshotId) >= 0) {
            String error = String.format("Previous snapshot was taken for the same processed position %d and exported position %d.", processedPosition, exportedPosition);
            return Either.left((Object)new SnapshotException.SnapshotAlreadyExistsException(error));
        }
        while (Files.exists(directory = this.snapshotsDirectory.resolve("transient-" + Long.toHexString(ThreadLocalRandom.current().nextLong())), new LinkOption[0])) {
        }
        FileBasedTransientSnapshot newPendingSnapshot = new FileBasedTransientSnapshot(newSnapshotId, directory, this, this.actor, this.checksumProvider, false);
        this.addPendingSnapshot(newPendingSnapshot);
        return Either.right((Object)newPendingSnapshot);
    }

    private void addPendingSnapshot(PersistableSnapshot pendingSnapshot) {
        Runnable action = () -> this.pendingSnapshots.add(pendingSnapshot);
        this.actor.run(action);
    }

    void removePendingSnapshot(PersistableSnapshot pendingSnapshot) {
        this.pendingSnapshots.remove(pendingSnapshot);
    }

    private void observeSnapshotSize(FileBasedSnapshot persistedSnapshot, boolean isBootstrap) {
        try (DirectoryStream<Path> contents = Files.newDirectoryStream(persistedSnapshot.getPath());){
            long totalSize = 0L;
            long totalCount = 0L;
            for (Path path : contents) {
                if (!Files.isRegularFile(path, new LinkOption[0])) continue;
                long size = Files.size(path);
                this.metrics.observeSnapshotFileSize(size, isBootstrap);
                totalSize += size;
                ++totalCount;
            }
            this.metrics.observeSnapshotSize(totalSize, isBootstrap);
            this.metrics.observeSnapshotChunkCount(totalCount, isBootstrap);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to observe size for snapshot {}", (Object)persistedSnapshot, (Object)e);
        }
    }

    private void abortPendingSnapshots(SnapshotId cutoffSnapshot) {
        LOGGER.trace("Search for orphaned snapshots below oldest valid snapshot with index {}", (Object)cutoffSnapshot.getSnapshotIdAsString());
        this.pendingSnapshots.stream().filter(pendingSnapshot -> pendingSnapshot.snapshotId().compareTo(cutoffSnapshot) < 0).forEach(PersistableSnapshot::abort);
    }

    private boolean isCurrentSnapshotNewer(FileBasedSnapshotId snapshotId) {
        FileBasedSnapshot persistedSnapshot = this.currentSnapshot.get();
        return persistedSnapshot != null && persistedSnapshot.getSnapshotId().compareTo(snapshotId) > 0;
    }

    FileBasedSnapshot persistNewSnapshot(Path destination, FileBasedSnapshotId snapshotId, ImmutableChecksumsSFV immutableChecksumsSFV, FileBasedSnapshotMetadata metadata) {
        boolean isBootstrap = metadata.isBootstrap();
        FileBasedSnapshot currentPersistedSnapshot = this.currentSnapshot.get();
        if (!isBootstrap && this.isCurrentSnapshotNewer(snapshotId)) {
            FileBasedSnapshotId currentPersistedSnapshotId = currentPersistedSnapshot.getSnapshotId();
            LOGGER.debug("Snapshot is older than the latest snapshot {}. Snapshot {} won't be committed.", (Object)currentPersistedSnapshotId, (Object)snapshotId);
            this.abortPendingSnapshots(currentPersistedSnapshotId);
            return currentPersistedSnapshot;
        }
        try (CloseableSilently ignored = this.metrics.startPersistTimer(isBootstrap);){
            boolean failed;
            Path checksumPath = this.computeChecksum(destination, snapshotId, immutableChecksumsSFV, destination);
            FileBasedSnapshot newPersistedSnapshot = new FileBasedSnapshot(destination, checksumPath, immutableChecksumsSFV, snapshotId, metadata, this::onSnapshotDeleted, this.actor);
            boolean bl = failed = !this.currentSnapshot.compareAndSet(currentPersistedSnapshot, newPersistedSnapshot);
            if (failed) {
                String errorMessage = "Expected that last snapshot is '%s', which should be replace with '%s', but last snapshot was '%s'.";
                throw new ConcurrentModificationException(String.format("Expected that last snapshot is '%s', which should be replace with '%s', but last snapshot was '%s'.", currentPersistedSnapshot, newPersistedSnapshot.getSnapshotId(), this.currentSnapshot.get()));
            }
            if (!isBootstrap) {
                this.availableSnapshots.add(newPersistedSnapshot);
            }
            LOGGER.info("Committed new snapshot {}, isBoostrap: {}", (Object)newPersistedSnapshot.getId(), (Object)newPersistedSnapshot.isBootstrap());
            this.metrics.incrementSnapshotCount(isBootstrap);
            this.observeSnapshotSize(newPersistedSnapshot, isBootstrap);
            if (!isBootstrap) {
                this.deleteOlderSnapshots(newPersistedSnapshot);
                this.listeners.forEach(listener -> listener.onNewSnapshot(newPersistedSnapshot));
            }
            FileBasedSnapshot fileBasedSnapshot = newPersistedSnapshot;
            return fileBasedSnapshot;
        }
    }

    private Path computeChecksum(Path source, FileBasedSnapshotId snapshotId, ImmutableChecksumsSFV immutableChecksumsSFV, Path destination) {
        Path checksumPath = this.buildSnapshotsChecksumPath(source, snapshotId);
        Path tmpChecksumPath = checksumPath.resolveSibling(checksumPath.getFileName().toString() + ".tmp");
        try {
            SnapshotChecksum.persist(tmpChecksumPath, immutableChecksumsSFV);
            FileUtil.moveDurably((Path)tmpChecksumPath, (Path)checksumPath, (CopyOption[])new CopyOption[0]);
            return checksumPath;
        }
        catch (IOException e) {
            this.rollbackPartialSnapshot(destination);
            throw new UncheckedIOException(e);
        }
    }

    private void deleteOlderSnapshots(FileBasedSnapshot newPersistedSnapshot) {
        LOGGER.trace("Purging snapshots older than {}", (Object)newPersistedSnapshot.getSnapshotId().getSnapshotIdAsString());
        List<FileBasedSnapshot> snapshotsToDelete = this.availableSnapshots.stream().filter(s -> !s.getId().equals(newPersistedSnapshot.getId())).filter(s -> !s.isReserved()).toList();
        snapshotsToDelete.forEach(previousSnapshot -> {
            LOGGER.debug("Deleting previous snapshot {}", (Object)previousSnapshot.getId());
            previousSnapshot.delete();
        });
        this.abortPendingSnapshots(newPersistedSnapshot.getSnapshotId());
    }

    private void rollbackPartialSnapshot(Path destination) {
        try {
            FileUtil.deleteFolderIfExists((Path)destination);
        }
        catch (IOException ioException) {
            LOGGER.debug("Pending snapshot {} could not be deleted on rollback, but will be safely ignored as a partial snapshot", (Object)destination, (Object)ioException);
        }
    }

    private Path buildSnapshotDirectory(FileBasedSnapshotId snapshotId, boolean isBootstrap) {
        Path directory = isBootstrap ? this.bootstrapSnapshotsDirectory : this.snapshotsDirectory;
        return directory.resolve(snapshotId.getSnapshotIdAsString());
    }

    private Path buildSnapshotsChecksumPath(Path snapshotPath, SnapshotId snapshotId) {
        return snapshotPath.getParent().resolve(snapshotId.getSnapshotIdAsString() + CHECKSUM_SUFFIX);
    }

    private boolean isChecksumFile(String name) {
        return name.endsWith(CHECKSUM_SUFFIX);
    }

    SnapshotMetrics getMetrics() {
        return this.metrics;
    }

    void onSnapshotDeleted(FileBasedSnapshot snapshot) {
        this.availableSnapshots.remove(snapshot);
    }

    public String toString() {
        return "FileBasedSnapshotStore{snapshotsDirectory=" + String.valueOf(this.snapshotsDirectory) + ", listeners=" + String.valueOf(this.listeners) + ", currentSnapshot=" + String.valueOf(this.currentSnapshot) + ", pendingSnapshots=" + String.valueOf(this.pendingSnapshots) + ", availableSnapshots=" + String.valueOf(this.availableSnapshots) + "}";
    }

    public void restore(String snapshotId, Map<String, Path> snapshotFiles) throws IOException {
        FileBasedSnapshotId parsedSnapshotId = FileBasedSnapshotId.ofFileName(snapshotId).getOrThrow();
        Path snapshotPath = this.buildSnapshotDirectory(parsedSnapshotId, false);
        FileUtil.ensureDirectoryExists((Path)snapshotPath);
        LOGGER.info("Moving snapshot {} to {}", (Object)snapshotId, (Object)snapshotPath);
        Set<String> snapshotFileNames = snapshotFiles.keySet();
        snapshotFileNames.stream().filter(name -> !this.isChecksumFile((String)name)).forEach(name -> this.moveNamedFileToDirectory((String)name, (Path)snapshotFiles.get(name), snapshotPath));
        Path checksumFile = snapshotFileNames.stream().filter(this::isChecksumFile).findFirst().map(snapshotFiles::get).orElseThrow();
        this.moveNamedFileToDirectory(checksumFile.getFileName().toString(), checksumFile, this.snapshotsDirectory);
        FileUtil.flushDirectory((Path)snapshotPath);
        FileUtil.flushDirectory((Path)this.snapshotsDirectory);
        LOGGER.info("Moved snapshot {} to {}", (Object)snapshotId, (Object)snapshotPath);
        FileBasedSnapshot snapshot = this.collectSnapshot(snapshotPath);
        if (snapshot == null) {
            throw new SnapshotException.CorruptedSnapshotException("Failed to open restored snapshot in %s".formatted(snapshotPath));
        }
        this.setLatestSnapshot(snapshot);
    }

    public ActorFuture<Void> restore(PersistedSnapshot snapshot) {
        return this.actor.call(() -> {
            this.restore(snapshot.getId(), snapshot.files());
            return null;
        });
    }

    private void moveNamedFileToDirectory(String name, Path source, Path targetDirectory) {
        Path targetFilePath = targetDirectory.resolve(name);
        try {
            try {
                Files.move(source, targetFilePath, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (AtomicMoveNotSupportedException e) {
                Files.move(source, targetFilePath, new CopyOption[0]);
                FileUtil.flush((Path)targetFilePath);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public ActorFuture<PersistedSnapshot> copyForBootstrap(PersistedSnapshot persistedSnapshot, BiConsumer<Path, Path> copySnapshot) {
        Path snapshotPath = persistedSnapshot.getPath();
        FileBasedSnapshotId zeroedSnapshotId = FileBasedSnapshotId.forBoostrap(this.brokerId);
        Path destinationFolder = this.buildSnapshotDirectory(zeroedSnapshotId, true);
        try {
            return this.actor.call(() -> {
                if (Files.exists(destinationFolder, new LinkOption[0]) || this.bootstrapSnapshot.get() != null) {
                    return CompletableActorFuture.completedExceptionally((Throwable)new SnapshotException.SnapshotAlreadyExistsException(String.format("Destination folder already exists: %s. Only one bootstrap snapshot can be taken at a time.If the previous scaling operation has terminated successfully, please delete the folder manually and try again.If the previous operation has not terminated successfully, please wait for it to complete before trying again.", destinationFolder)));
                }
                FileUtil.ensureDirectoryExists((Path)destinationFolder);
                return CompletableActorFuture.completed();
            }).andThen(fut -> fut, (Executor)this.actor).andThen(ignored -> {
                FileBasedTransientSnapshot transientSnapshot = new FileBasedTransientSnapshot(zeroedSnapshotId, destinationFolder, this, this.actor, this.checksumProvider, true);
                return transientSnapshot.take(toPath -> copySnapshot.accept(snapshotPath, (Path)toPath)).andThen(ignore -> transientSnapshot.persistInternal(), (Executor)this.actor);
            }, (Executor)this.actor).thenApply(persisted -> {
                this.bootstrapSnapshot.set((FileBasedSnapshot)persisted);
                return persisted;
            }, (Executor)this.actor);
        }
        catch (Exception e) {
            throw new SnapshotException.SnapshotCopyForBootstrapException(String.format("Failed to copy snapshot %s to new location: sourcePath=%s, destinationPath=%s", zeroedSnapshotId, snapshotPath, destinationFolder), e);
        }
    }

    ActorFuture<Void> deleteBootstrapSnapshots() {
        return this.actor.call(() -> {
            this.deleteBootstrapSnapshotsInternal();
            return null;
        });
    }

    private void deleteBootstrapSnapshotsInternal() {
        FileBasedSnapshot deletableSnapshot = this.bootstrapSnapshot.getAndSet(null);
        if (deletableSnapshot != null) {
            deletableSnapshot.delete();
        }
    }

    public Optional<PersistedSnapshot> getBootstrapSnapshot() {
        return Optional.ofNullable((PersistedSnapshot)this.bootstrapSnapshot.get());
    }
}

