/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.snapshots.ImmutableChecksumsSFV;
import io.camunda.zeebe.snapshots.MutableChecksumsSFV;
import io.camunda.zeebe.snapshots.impl.SnapshotChecksum;
import io.camunda.zeebe.test.util.STracer;
import io.camunda.zeebe.test.util.asserts.strace.FSyncTraceAssert;
import io.camunda.zeebe.test.util.asserts.strace.STracerAssert;
import io.camunda.zeebe.test.util.junit.StraceTest;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.zip.CRC32C;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.awaitility.Awaitility;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

public final class SnapshotChecksumTest {
    @TempDir
    private Path temporaryFolder;
    private Path singleFileSnapshot;
    private Path multipleFileSnapshot;
    private Path corruptedSnapshot;

    @BeforeEach
    void setup() throws Exception {
        this.singleFileSnapshot = this.createTempDir("single");
        this.multipleFileSnapshot = this.createTempDir("multi");
        this.corruptedSnapshot = this.createTempDir("corrupted");
        SnapshotChecksumTest.createChunk(this.singleFileSnapshot, "file1.txt");
        SnapshotChecksumTest.createChunk(this.multipleFileSnapshot, "file1.txt");
        SnapshotChecksumTest.createChunk(this.multipleFileSnapshot, "file2.txt");
        SnapshotChecksumTest.createChunk(this.multipleFileSnapshot, "file3.txt");
        SnapshotChecksumTest.createChunk(this.corruptedSnapshot, "file1.txt");
        SnapshotChecksumTest.createChunk(this.corruptedSnapshot, "file2.txt");
        SnapshotChecksumTest.createChunk(this.corruptedSnapshot, "file3.txt");
    }

    static void createChunk(Path snapshot, String chunkName) throws IOException {
        Files.writeString(snapshot.resolve(chunkName), (CharSequence)chunkName, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
    }

    @Test
    void shouldGenerateTheSameChecksumForOneFile() throws Exception {
        MutableChecksumsSFV expectedChecksums = SnapshotChecksum.calculate((Path)this.singleFileSnapshot);
        MutableChecksumsSFV actual = SnapshotChecksum.calculate((Path)this.singleFileSnapshot);
        Assertions.assertThat((boolean)expectedChecksums.sameChecksums((ImmutableChecksumsSFV)actual)).isTrue();
    }

    @Test
    void shouldGenerateDifferentChecksumWhenFileNameIsDifferent() throws Exception {
        MutableChecksumsSFV expectedChecksum = SnapshotChecksum.calculate((Path)this.singleFileSnapshot);
        Files.move(this.singleFileSnapshot.resolve("file1.txt"), this.singleFileSnapshot.resolve("renamed"), new CopyOption[0]);
        MutableChecksumsSFV actual = SnapshotChecksum.calculate((Path)this.singleFileSnapshot);
        Assertions.assertThat((boolean)expectedChecksum.sameChecksums((ImmutableChecksumsSFV)actual)).isFalse();
    }

    @Test
    void shouldGenerateTheSameChecksumForMultipleFiles() throws Exception {
        MutableChecksumsSFV expectedChecksum = SnapshotChecksum.calculate((Path)this.multipleFileSnapshot);
        MutableChecksumsSFV actual = SnapshotChecksum.calculate((Path)this.multipleFileSnapshot);
        Assertions.assertThat((boolean)expectedChecksum.sameChecksums((ImmutableChecksumsSFV)actual)).isTrue();
    }

    @Test
    void shouldGenerateDifferentChecksumForDifferentFiles() throws Exception {
        MutableChecksumsSFV expectedChecksum = SnapshotChecksum.calculate((Path)this.singleFileSnapshot);
        MutableChecksumsSFV actual = SnapshotChecksum.calculate((Path)this.multipleFileSnapshot);
        Assertions.assertThat((boolean)expectedChecksum.sameChecksums((ImmutableChecksumsSFV)actual)).isFalse();
    }

    @Test
    void shouldPersistChecksum() throws Exception {
        MutableChecksumsSFV expectedChecksum = SnapshotChecksum.calculate((Path)this.multipleFileSnapshot);
        Path checksumPath = this.multipleFileSnapshot.resolveSibling("checksum");
        SnapshotChecksum.persist((Path)checksumPath, (ImmutableChecksumsSFV)expectedChecksum);
        ImmutableChecksumsSFV actual = SnapshotChecksum.read((Path)checksumPath);
        Assertions.assertThat((boolean)expectedChecksum.sameChecksums(actual)).isTrue();
    }

    @StraceTest
    void shouldFlushOnPersist() throws Exception {
        STracer tracer;
        Path traceFile = this.temporaryFolder.resolve("traceFile");
        MutableChecksumsSFV expectedChecksum = SnapshotChecksum.calculate((Path)this.multipleFileSnapshot);
        Path checksumPath = this.multipleFileSnapshot.resolveSibling("checksum");
        try (STracer sTracer = tracer = STracer.tracerFor((STracer.Syscall)STracer.Syscall.FSYNC, (Path)traceFile);){
            SnapshotChecksum.persist((Path)checksumPath, (ImmutableChecksumsSFV)expectedChecksum);
            Awaitility.await((String)"until fsync was called for our checksum file").pollInSameThread().pollInterval(Duration.ofMillis(50L)).untilAsserted(() -> ((FSyncTraceAssert)((ListAssert)STracerAssert.assertThat((STracer)tracer).fsyncTraces().hasSize(1)).first(FSyncTraceAssert.factory())).hasPath(checksumPath));
        }
    }

    @Test
    void shouldDetectCorruptedSnapshot() throws IOException {
        MutableChecksumsSFV expectedChecksum = SnapshotChecksum.calculate((Path)this.corruptedSnapshot);
        Path checksumPath = this.corruptedSnapshot.resolveSibling("checksum");
        SnapshotChecksum.persist((Path)checksumPath, (ImmutableChecksumsSFV)expectedChecksum);
        Files.delete(this.corruptedSnapshot.resolve("file1.txt"));
        MutableChecksumsSFV actualChecksum = SnapshotChecksum.calculate((Path)this.corruptedSnapshot);
        Assertions.assertThat((boolean)expectedChecksum.sameChecksums((ImmutableChecksumsSFV)actualChecksum)).isFalse();
    }

    @Test
    void shouldCalculateSameChecksumOfLargeFile() throws IOException {
        Path largeSnapshot = this.createTempDir("large");
        Path file = largeSnapshot.resolve("file");
        String largeData = "a".repeat(16484);
        Files.writeString(file, (CharSequence)largeData, StandardOpenOption.CREATE, StandardOpenOption.WRITE);
        CRC32C checksum = new CRC32C();
        checksum.update(Files.readAllBytes(file));
        long expected = checksum.getValue();
        MutableChecksumsSFV actual = SnapshotChecksum.calculate((Path)largeSnapshot);
        Assertions.assertThat((Long)((Long)actual.getChecksums().get("file"))).isEqualTo(expected);
    }

    @Test
    void shouldAddChecksumOfMetadataAtTheEnd() throws IOException {
        Path folder = this.createTempDir("folder");
        SnapshotChecksumTest.createChunk(folder, "file1.txt");
        SnapshotChecksumTest.createChunk(folder, "file2.txt");
        SnapshotChecksumTest.createChunk(folder, "file3.txt");
        MutableChecksumsSFV checksumCalculatedInSteps = SnapshotChecksum.calculate((Path)folder);
        SnapshotChecksumTest.createChunk(folder, "zeebe.metadata");
        checksumCalculatedInSteps.updateFromFile(folder.resolve("zeebe.metadata"));
        MutableChecksumsSFV checksumCalculatedAtOnce = SnapshotChecksum.calculate((Path)folder);
        Assertions.assertThat((boolean)checksumCalculatedInSteps.sameChecksums((ImmutableChecksumsSFV)checksumCalculatedAtOnce)).isTrue();
    }

    private Path createTempDir(String name) throws IOException {
        Path path = this.temporaryFolder.resolve(name);
        FileUtil.ensureDirectoryExists((Path)path);
        return path;
    }
}

