/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.protocol.Protocol;
import io.camunda.zeebe.snapshots.SnapshotChunk;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotChunkReader;
import io.camunda.zeebe.snapshots.impl.SnapshotChunkUtil;
import io.camunda.zeebe.util.FileUtil;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import org.assertj.core.api.AbstractComparableAssert;
import org.assertj.core.api.Assertions;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public final class FileBasedSnapshotChunkReaderTest {
    private static final long SNAPSHOT_CHECKSUM = 1L;
    private static final Map<String, String> SNAPSHOT_CHUNK = Map.of("file3", "content", "file1", "this", "file2", "is");
    private static final List<Map.Entry<String, String>> SORTED_CHUNKS = SNAPSHOT_CHUNK.entrySet().stream().sorted(Map.Entry.comparingByKey()).collect(Collectors.toUnmodifiableList());
    @Rule
    public final TemporaryFolder temporaryFolder = new TemporaryFolder();
    private Path snapshotDirectory;

    @Test
    public void shouldAssignChunkIdsFromFileNames() throws IOException {
        FileBasedSnapshotChunkReader reader = this.newReader();
        Assertions.assertThat((String)reader.next().getChunkName()).isEqualTo("file1");
        Assertions.assertThat((String)reader.next().getChunkName()).isEqualTo("file2");
        Assertions.assertThat((String)reader.next().getChunkName()).isEqualTo("file3");
    }

    @Test
    public void shouldThrowExceptionWhenChunkFileDoesNotExist() throws IOException {
        FileBasedSnapshotChunkReader reader = this.newReader();
        Files.delete(this.snapshotDirectory.resolve("file1"));
        Assertions.assertThatThrownBy(() -> ((FileBasedSnapshotChunkReader)reader).next()).hasCauseInstanceOf(FileNotFoundException.class);
    }

    @Test
    public void shouldThrowExceptionWhenNoDirectoryExist() throws IOException {
        FileBasedSnapshotChunkReader reader = this.newReader();
        FileUtil.deleteFolder((Path)this.snapshotDirectory);
        Assertions.assertThatThrownBy(() -> ((FileBasedSnapshotChunkReader)reader).next()).hasCauseInstanceOf(FileNotFoundException.class);
    }

    @Test
    public void shouldReadSnapshotChunks() throws IOException {
        try (FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();){
            for (int i = 0; i < SNAPSHOT_CHUNK.size(); ++i) {
                Assertions.assertThat((boolean)snapshotChunkReader.hasNext()).isTrue();
                ByteBuffer nextId = snapshotChunkReader.nextId();
                SnapshotChunk chunk = snapshotChunkReader.next();
                String chunkId = chunk.getChunkName() + "__" + chunk.getFileBlockPosition();
                ((AbstractComparableAssert)Assertions.assertThat((Comparable)ByteBuffer.wrap(chunkId.getBytes(StandardCharsets.UTF_8))).isNotNull()).isEqualTo((Object)nextId);
                Assertions.assertThat((String)chunk.getSnapshotId()).isEqualTo(this.snapshotDirectory.getFileName().toString());
                Assertions.assertThat((int)chunk.getTotalCount()).isEqualTo(SNAPSHOT_CHUNK.size());
                Assertions.assertThat((long)chunk.getChecksum()).isEqualTo(SnapshotChunkUtil.createChecksum((byte[])chunk.getContent()));
                Assertions.assertThat((Path)this.snapshotDirectory.resolve(chunk.getChunkName())).hasBinaryContent(chunk.getContent());
            }
        }
    }

    @Test
    public void shouldReadSnapshotChunksInOrder() throws IOException {
        ArrayList<SnapshotChunk> snapshotChunks = new ArrayList<SnapshotChunk>();
        ArrayList<ByteBuffer> snapshotChunkIds = new ArrayList<ByteBuffer>();
        try (FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();){
            while (snapshotChunkReader.hasNext()) {
                snapshotChunkIds.add(snapshotChunkReader.nextId());
                snapshotChunks.add(snapshotChunkReader.next());
            }
        }
        Assertions.assertThat(snapshotChunkIds).containsExactly((Object[])new ByteBuffer[]{this.asByteBuffer("file1__0"), this.asByteBuffer("file2__0"), this.asByteBuffer("file3__0")});
        Assertions.assertThat(snapshotChunks).extracting(SnapshotChunk::getContent).extracting(String::new).containsExactly((Object[])new String[]{"this", "is", "content"});
    }

    @Test
    public void shouldSeekToChunk() throws IOException {
        ArrayList<String> snapshotChunkIds = new ArrayList<String>();
        try (FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();){
            snapshotChunkReader.seek(this.asByteBuffer("file2__0"));
            while (snapshotChunkReader.hasNext()) {
                snapshotChunkIds.add(snapshotChunkReader.next().getChunkName());
            }
        }
        Assertions.assertThat(snapshotChunkIds).containsExactly((Object[])new String[]{"file2", "file3"});
    }

    @Test
    public void shouldResetToInitialChunk() throws IOException {
        ArrayList<String> snapshotChunkIds = new ArrayList<String>();
        try (FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();){
            snapshotChunkReader.seek(this.asByteBuffer("file2__0"));
            snapshotChunkReader.next();
            snapshotChunkReader.reset();
            while (snapshotChunkReader.hasNext()) {
                snapshotChunkIds.add(snapshotChunkReader.next().getChunkName());
            }
        }
        Assertions.assertThat(snapshotChunkIds).containsExactly((Object[])new String[]{"file1", "file2", "file3"});
    }

    @Test
    public void shouldThrowExceptionOnReachingLimit() throws IOException {
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();
        while (snapshotChunkReader.hasNext()) {
            snapshotChunkReader.next();
        }
        Assertions.assertThat((Comparable)snapshotChunkReader.nextId()).isNull();
        Assertions.assertThatThrownBy(() -> ((FileBasedSnapshotChunkReader)snapshotChunkReader).next()).isInstanceOf(NoSuchElementException.class);
    }

    @Test
    public void shouldSplitFileContentsIntoChunks() throws IOException {
        byte[] contentBytes;
        int maxChunkSize = 3;
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader(3L);
        List<SnapshotChunk> snapshotChunks = this.getAllChunks(snapshotChunkReader);
        HashMap<String, ByteBuffer> fileNameBytesMap = new HashMap<String, ByteBuffer>();
        for (Map.Entry<String, String> entry : SNAPSHOT_CHUNK.entrySet()) {
            contentBytes = entry.getValue().getBytes(StandardCharsets.UTF_8);
            fileNameBytesMap.put(entry.getKey(), ByteBuffer.allocate(contentBytes.length));
        }
        for (SnapshotChunk chunk : snapshotChunks) {
            ((ByteBuffer)fileNameBytesMap.get(chunk.getChunkName())).put(chunk.getContent());
        }
        for (Map.Entry<String, String> entry : SNAPSHOT_CHUNK.entrySet()) {
            contentBytes = entry.getValue().getBytes(StandardCharsets.UTF_8);
            Assertions.assertThat((byte[])((ByteBuffer)fileNameBytesMap.get(entry.getKey())).array()).isEqualTo((Object)contentBytes);
        }
    }

    @Test
    public void shouldHaveCorrectFileBlockFieldsForSplitFiles() throws IOException {
        int maxChunkSize = 3;
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader(3L);
        List<SnapshotChunk> snapshotChunks = this.getAllChunks(snapshotChunkReader);
        Map<String, List<SnapshotChunk>> fileChunksGroupedByFileName = snapshotChunks.stream().collect(Collectors.groupingBy(SnapshotChunk::getChunkName));
        for (Map.Entry<String, List<SnapshotChunk>> entry : fileChunksGroupedByFileName.entrySet()) {
            String fileName = entry.getKey();
            int fileContentsBytesLength = SNAPSHOT_CHUNK.get(fileName).getBytes(StandardCharsets.UTF_8).length;
            int expectedFileBlocksCount = Math.ceilDiv(fileContentsBytesLength, 3);
            List<SnapshotChunk> fileBlocks = entry.getValue();
            Assertions.assertThat((int)fileBlocks.size()).isEqualTo(expectedFileBlocksCount);
            Assertions.assertThat(fileBlocks.stream().map(SnapshotChunk::getFileBlockPosition).toList()).isEqualTo(LongStream.range(0L, expectedFileBlocksCount).map(l -> l * 3L).boxed().toList());
        }
    }

    @Test
    public void shouldHaveCorrectNextChunkIdForSplitFiles() throws IOException {
        int maxChunkSize = 3;
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader(3L);
        ArrayList<ByteBuffer> snapshotChunkIds = new ArrayList<ByteBuffer>();
        while (snapshotChunkReader.hasNext()) {
            snapshotChunkIds.add(snapshotChunkReader.nextId());
            snapshotChunkReader.next();
        }
        ArrayList expectedChunkIds = new ArrayList();
        Collections.addAll(expectedChunkIds, ByteBuffer.wrap("file1__0".getBytes()), ByteBuffer.wrap("file1__3".getBytes()), ByteBuffer.wrap("file2__0".getBytes()), ByteBuffer.wrap("file3__0".getBytes()), ByteBuffer.wrap("file3__3".getBytes()), ByteBuffer.wrap("file3__6".getBytes()));
        Assertions.assertThat(snapshotChunkIds).isEqualTo(expectedChunkIds);
    }

    @Test
    public void shouldStartSplittingFilesOnceChunkSizeIsSet() throws IOException {
        int maxChunkSize = 3;
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();
        ArrayList<SnapshotChunk> snapshotChunks = new ArrayList<SnapshotChunk>();
        for (int i = 0; i < 2; ++i) {
            snapshotChunks.add(snapshotChunkReader.next());
        }
        snapshotChunkReader.setMaximumChunkSize(3);
        while (snapshotChunkReader.hasNext()) {
            snapshotChunks.add(snapshotChunkReader.next());
        }
        Map<String, List<SnapshotChunk>> fileChunksGroupedByFileName = snapshotChunks.stream().collect(Collectors.groupingBy(SnapshotChunk::getChunkName));
        Assertions.assertThat((int)fileChunksGroupedByFileName.get("file1").size()).isEqualTo(1);
        Assertions.assertThat((int)fileChunksGroupedByFileName.get("file2").size()).isEqualTo(1);
        Assertions.assertThat((int)fileChunksGroupedByFileName.get("file3").size()).isEqualTo(3);
    }

    @Test
    public void shouldSeekCorrectlyWhenFileChunkingEnabled() throws IOException {
        int maxChunkSize = 3;
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader(3L);
        ArrayList<SnapshotChunk> snapshotChunks = new ArrayList<SnapshotChunk>();
        ByteBuffer firstId = snapshotChunkReader.nextId();
        while (snapshotChunkReader.hasNext()) {
            snapshotChunks.add(snapshotChunkReader.next());
        }
        snapshotChunkReader.seek(firstId);
        ArrayList<SnapshotChunk> snapshotChunksAfterSeek = new ArrayList<SnapshotChunk>();
        while (snapshotChunkReader.hasNext()) {
            snapshotChunksAfterSeek.add(snapshotChunkReader.next());
        }
        Assertions.assertThat(snapshotChunksAfterSeek.stream().map(SnapshotChunk::getContent).collect(Collectors.toUnmodifiableList())).containsExactlyElementsOf((Iterable)snapshotChunks.stream().map(SnapshotChunk::getContent).collect(Collectors.toUnmodifiableList()));
    }

    @Test
    public void shouldSeekSameByteBufferTwiceWithNoError() throws IOException {
        ByteBuffer buffer = ByteBuffer.wrap("file3__0".getBytes(StandardCharsets.UTF_8));
        FileBasedSnapshotChunkReader snapshotChunkReader = this.newReader();
        snapshotChunkReader.seek(buffer);
        SnapshotChunk chunkFromFirstSeek = snapshotChunkReader.next();
        snapshotChunkReader.seek(buffer);
        SnapshotChunk chunkFromSecondSeek = snapshotChunkReader.next();
        Assertions.assertThat((String)chunkFromFirstSeek.getChunkName()).isEqualTo(chunkFromSecondSeek.getChunkName());
        Assertions.assertThat((byte[])chunkFromFirstSeek.getContent()).isEqualTo((Object)chunkFromSecondSeek.getContent());
        Assertions.assertThat((long)chunkFromFirstSeek.getChecksum()).isEqualTo(chunkFromSecondSeek.getChecksum());
    }

    private List<SnapshotChunk> getAllChunks(FileBasedSnapshotChunkReader reader) {
        ArrayList<SnapshotChunk> snapshotChunks = new ArrayList<SnapshotChunk>();
        while (reader.hasNext()) {
            snapshotChunks.add(reader.next());
        }
        return snapshotChunks;
    }

    private ByteBuffer asByteBuffer(String string) {
        return ByteBuffer.wrap(string.getBytes()).order(Protocol.ENDIANNESS);
    }

    private FileBasedSnapshotChunkReader newReader(long chunkSize) throws IOException {
        this.snapshotDirectory = this.temporaryFolder.getRoot().toPath();
        for (String chunk : SNAPSHOT_CHUNK.keySet()) {
            Path path = this.snapshotDirectory.resolve(chunk);
            Files.createFile(path, new FileAttribute[0]);
            Files.writeString(path, (CharSequence)SNAPSHOT_CHUNK.get(chunk), new OpenOption[0]);
        }
        return new FileBasedSnapshotChunkReader(this.snapshotDirectory, chunkSize);
    }

    private FileBasedSnapshotChunkReader newReader() throws IOException {
        return this.newReader(Long.MAX_VALUE);
    }
}

