/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.testing.ActorSchedulerRule;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.SnapshotMetadata;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotId;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotStore;
import io.camunda.zeebe.test.util.asserts.DirectoryAssert;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractPathAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ObjectAssert;
import org.assertj.core.api.OptionalAssert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class FileBasedTransientSnapshotTest {
    private static final String SNAPSHOT_DIRECTORY = "snapshots";
    private static final String PENDING_DIRECTORY = "pending";
    private static final Map<String, String> SNAPSHOT_FILE_CONTENTS = Map.of("file1", "file1 contents", "file2", "file2 contents");
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public ActorSchedulerRule scheduler = new ActorSchedulerRule();
    private FileBasedSnapshotStore snapshotStore;
    private Path snapshotsDir;
    private Path pendingDir;

    @Before
    public void beforeEach() throws IOException {
        Path root = this.temporaryFolder.getRoot().toPath();
        this.pendingDir = root.resolve(PENDING_DIRECTORY);
        this.snapshotsDir = root.resolve(SNAPSHOT_DIRECTORY);
        this.snapshotStore = this.createStore(root);
    }

    @Test
    public void shouldNotCreateTransientDirectoryIfNothingWritten() {
        this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L);
        ((AbstractPathAssert)Assertions.assertThat((Path)this.pendingDir).as("the pending directory is empty as nothing was written", new Object[0])).isEmptyDirectory();
        ((AbstractPathAssert)Assertions.assertThat((Path)this.snapshotsDir).as("the snapshots directory is empty as nothing was written", new Object[0])).isEmptyDirectory();
    }

    @Test
    public void shouldEncodeSnapshotIdInPath() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        FileBasedSnapshotId pathId = (FileBasedSnapshotId)FileBasedSnapshotId.ofPath((Path)transientSnapshot.getPath()).orElseThrow();
        Assertions.assertThat((Comparable)pathId).isEqualTo((Object)transientSnapshot.snapshotId());
    }

    @Test
    public void shouldNotCommitUntilPersisted() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        Assertions.assertThat((Path)this.snapshotsDir).isDirectoryNotContaining(p -> p.getFileName().toFile().getName().equals("1-2-3-4.checksum"));
    }

    @Test
    public void shouldTakeTransientSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        ((AbstractPathAssert)Assertions.assertThat((Path)transientSnapshot.getPath()).as("the transient snapshot directory was written in the pending directory", new Object[0])).hasParent(this.snapshotsDir).isNotEmptyDirectory();
    }

    @Test
    public void shouldDeleteTransientDirectoryOnAbort() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        transientSnapshot.abort().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)transientSnapshot.getPath()).as("the transient directory should not exist after abort", new Object[0])).doesNotExist();
        ((AbstractPathAssert)Assertions.assertThat((Path)this.pendingDir).as("the pending directory is empty after abort", new Object[0])).isEmptyDirectory();
    }

    @Test
    public void shouldNotDeletePersistedSnapshotOnPurge() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        this.snapshotStore.purgePendingSnapshots().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)persistedSnapshot.getPath()).as("the persisted snapshot still exists in the committed snapshots directory", new Object[0])).isNotEmptyDirectory().hasParent(this.snapshotsDir);
        ((OptionalAssert)Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).as("the latest snapshot was not changed after purge", new Object[0])).hasValue((Object)persistedSnapshot);
    }

    @Test
    public void shouldPersistSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        transientSnapshot.take(this::writeSnapshot);
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)persistedSnapshot.getPath()).as("the snapshot was persisted under the snapshots directory", new Object[0])).hasParent(this.snapshotsDir);
        for (Map.Entry<String, String> expectedChunk : SNAPSHOT_FILE_CONTENTS.entrySet()) {
            String expectedFileName = expectedChunk.getKey();
            String expectedFileContents = expectedChunk.getValue();
            ((AbstractPathAssert)Assertions.assertThat((Path)persistedSnapshot.getPath().resolve(expectedFileName)).as("chunk %s was persisted with the expected contents", new Object[]{expectedFileName})).isRegularFile().hasBinaryContent(expectedFileContents.getBytes(StandardCharsets.UTF_8));
        }
    }

    @Test
    public void shouldReplacePreviousSnapshotOnPersist() {
        TransientSnapshot oldSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        oldSnapshot.take(this::writeSnapshot);
        oldSnapshot.persist().join();
        TransientSnapshot newSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(2L, 0L, 1L, 0L).get();
        newSnapshot.take(this::writeSnapshot);
        FileBasedSnapshot persistedSnapshot = (FileBasedSnapshot)newSnapshot.persist().join();
        ((DirectoryAssert)((DirectoryAssert)Assertions.assertThat((Path)this.snapshotsDir).asInstanceOf(DirectoryAssert.factory())).as("the committed snapshots directory only contains the latest snapshot", new Object[0])).isDirectoryContainingExactly(new Path[]{persistedSnapshot.getPath(), persistedSnapshot.getChecksumPath()});
    }

    @Test
    public void shouldRemoveTransientSnapshotOnPersist() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        transientSnapshot.take(this::writeSnapshot);
        TransientSnapshot newSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(2L, 0L, 1L, 0L).get();
        newSnapshot.take(this::writeSnapshot);
        newSnapshot.persist().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)this.pendingDir).as("there are no more transient snapshots after persisting a snapshot with higher index", new Object[0])).isEmptyDirectory();
    }

    @Test
    public void shouldRemoveTransientWhenCurrentSnapshotIsNewer() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(10L, 10L, 10L, 10L).get();
        transientSnapshot.take(this::writeSnapshot);
        transientSnapshot.persist().join();
        TransientSnapshot oldSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 1L, 1L, 1L).get();
        oldSnapshot.take(this::writeSnapshot);
        oldSnapshot.persist().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)this.pendingDir).as("transient and outdated snapshot has been deleted", new Object[0])).isEmptyDirectory();
    }

    @Test
    public void shouldNotRemoveTransientSnapshotWithGreaterIdOnPersist() {
        TransientSnapshot newerTransientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(2L, 0L, 1L, 0L).get();
        newerTransientSnapshot.take(this::writeSnapshot);
        TransientSnapshot newSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        newSnapshot.take(this::writeSnapshot);
        newSnapshot.persist().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)newerTransientSnapshot.getPath()).as("the newer transient snapshot still exists since it has a greater ID", new Object[0])).isNotEmptyDirectory().hasParent(this.snapshotsDir);
    }

    @Test
    public void shouldRemoveTransientDirectoryOnTakeException() {
        TransientSnapshot snapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        ActorFuture didTakeSnapshot = snapshot.take(path -> {
            throw new RuntimeException("expected");
        });
        Assertions.assertThatThrownBy(() -> ((ActorFuture)didTakeSnapshot).join()).hasCauseInstanceOf(RuntimeException.class);
        ((AbstractPathAssert)Assertions.assertThat((Path)this.pendingDir).as("there is no leftover transient directory", new Object[0])).isEmptyDirectory();
        ((AbstractPathAssert)Assertions.assertThat((Path)this.snapshotsDir).as("there is no committed snapshot", new Object[0])).isEmptyDirectory();
    }

    @Test
    public void shouldNotPersistNonExistentTransientSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 2L, 3L).get();
        transientSnapshot.take(p -> {});
        ActorFuture persisted = transientSnapshot.persist();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ActorFuture)persisted).join()).as("did not persist a non existent transient snapshot", new Object[0])).hasCauseInstanceOf(IllegalStateException.class);
    }

    @Test
    public void shouldNotPersistEmptyTransientSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 2L, 3L).get();
        transientSnapshot.take(p -> p.toFile().mkdir());
        ActorFuture persisted = transientSnapshot.persist();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ActorFuture)persisted).join()).as("did not persist an empty transient snapshot directory", new Object[0])).hasCauseInstanceOf(IllegalStateException.class);
    }

    @Test
    public void shouldPersistIdempotently() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        FileBasedSnapshot firstSnapshot = (FileBasedSnapshot)transientSnapshot.persist().join();
        PersistedSnapshot secondSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        ((ObjectAssert)Assertions.assertThat((Object)secondSnapshot).as("did persist the same snapshot twice", new Object[0])).isEqualTo((Object)firstSnapshot);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)secondSnapshot.getChecksums().sameChecksums(firstSnapshot.getChecksums())).as("the content of the snapshot remains unchanged", new Object[0])).isTrue();
        ((DirectoryAssert)((DirectoryAssert)Assertions.assertThat((Path)this.snapshotsDir).asInstanceOf(DirectoryAssert.factory())).as("snapshots directory only contains snapshot %s", new Object[]{firstSnapshot.getId()})).isDirectoryContainingExactly(new Path[]{firstSnapshot.getPath(), firstSnapshot.getChecksumPath()});
    }

    @Test
    public void shouldAddMetadataToPersistedSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.withLastFollowupEventPosition(100L).persist().join();
        SnapshotMetadata metadata = persistedSnapshot.getMetadata();
        Assertions.assertThat((Object)metadata).extracting(new Function[]{SnapshotMetadata::processedPosition, SnapshotMetadata::exportedPosition, SnapshotMetadata::lastFollowupEventPosition, SnapshotMetadata::version}).containsExactly(new Object[]{3L, 4L, 100L, 1});
    }

    @Test
    public void shouldPersistMetadata() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.withLastFollowupEventPosition(100L).persist().join();
        ((AbstractPathAssert)Assertions.assertThat((Path)persistedSnapshot.getPath()).describedAs("Metadata file is persisted in snapshot path", new Object[0])).isDirectoryContaining(path -> path.getFileName().toString().equals("zeebe.metadata"));
    }

    private boolean writeSnapshot(Path path) {
        try {
            FileUtil.ensureDirectoryExists((Path)path);
            for (Map.Entry<String, String> entry : SNAPSHOT_FILE_CONTENTS.entrySet()) {
                Path fileName = path.resolve(entry.getKey());
                byte[] fileContent = entry.getValue().getBytes(StandardCharsets.UTF_8);
                Files.write(fileName, fileContent, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return true;
    }

    private FileBasedSnapshotStore createStore(Path root) throws IOException {
        FileBasedSnapshotStore store = new FileBasedSnapshotStore(0, 1, root, snapshotPath -> Map.of());
        this.scheduler.submitActor((Actor)store);
        return store;
    }
}

