/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.testing.ActorSchedulerRule;
import io.camunda.zeebe.snapshots.CRC32CChecksumProvider;
import io.camunda.zeebe.snapshots.ImmutableChecksumsSFV;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.SnapshotChunk;
import io.camunda.zeebe.snapshots.SnapshotChunkReader;
import io.camunda.zeebe.snapshots.SnapshotException;
import io.camunda.zeebe.snapshots.SnapshotReservation;
import io.camunda.zeebe.snapshots.TestChecksumProvider;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedReceivedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotStore;
import io.camunda.zeebe.snapshots.impl.SfvChecksumImpl;
import io.camunda.zeebe.snapshots.impl.SnapshotChecksum;
import io.camunda.zeebe.test.util.asserts.DirectoryAssert;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.zip.CRC32C;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AssertionsForClassTypes;
import org.assertj.core.api.OptionalAssert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class FileBasedSnapshotStoreTest {
    private static final String SNAPSHOT_DIRECTORY = "snapshots";
    private static final String PENDING_DIRECTORY = "pending";
    private static final String SNAPSHOT_CONTENT_FILE_NAME = "file1.txt";
    private static final String SNAPSHOT_CONTENT = "this is the content";
    private static final Integer PARTITION_ID = 1;
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public ActorSchedulerRule scheduler = new ActorSchedulerRule();
    private Path snapshotsDir;
    private Path pendingSnapshotsDir;
    private FileBasedSnapshotStore snapshotStore;
    private Path rootDirectory;

    @Before
    public void before() throws IOException {
        this.rootDirectory = this.temporaryFolder.newFolder(SNAPSHOT_DIRECTORY).toPath();
        this.snapshotsDir = this.rootDirectory.resolve(SNAPSHOT_DIRECTORY);
        this.pendingSnapshotsDir = this.rootDirectory.resolve(PENDING_DIRECTORY);
        this.snapshotStore = this.createStore(this.rootDirectory);
    }

    @Test
    public void shouldCreateDirectoriesIfNotExist() {
        Path root = this.temporaryFolder.getRoot().toPath();
        FileBasedSnapshotStore store = new FileBasedSnapshotStore(0, 1, root, snapshotPath -> Map.of());
        Assertions.assertThat((Path)root.resolve(SNAPSHOT_DIRECTORY)).exists().isDirectory();
        Assertions.assertThat((Path)root.resolve(PENDING_DIRECTORY)).exists().isDirectory();
        Assertions.assertThat((Optional)store.getLatestSnapshot()).isEmpty();
    }

    @Test
    public void shouldDeleteStore() {
        this.snapshotStore.delete().join();
        Assertions.assertThat((Path)this.pendingSnapshotsDir).doesNotExist();
        Assertions.assertThat((Path)this.snapshotsDir).doesNotExist();
    }

    @Test
    public void shouldLoadExistingSnapshot() throws IOException {
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)this.takeTransientSnapshot().persist().join();
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((long)this.snapshotStore.getCurrentSnapshotIndex()).isEqualTo(1L);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).hasValue((Object)persistedSnapshot);
    }

    @Test
    public void shouldLoadExistingSnapshotWithMetadata() throws IOException {
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)this.takeTransientSnapshot().withLastFollowupEventPosition(100L).persist().join();
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((long)this.snapshotStore.getCurrentSnapshotIndex()).isEqualTo(1L);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).hasValue((Object)persistedSnapshot);
        PersistedSnapshot latestSnapshot = (PersistedSnapshot)this.snapshotStore.getLatestSnapshot().orElseThrow();
        Assertions.assertThat((Object)latestSnapshot.getMetadata()).isEqualTo((Object)persistedSnapshot.getMetadata());
    }

    @Test
    public void shouldLoadLatestSnapshotWhenMoreThanOneExistsAndDeleteOlder() throws IOException {
        FileBasedSnapshotStore otherStore = this.createStore(this.rootDirectory);
        PersistedSnapshot olderSnapshot = (PersistedSnapshot)this.takeTransientSnapshot(1L, otherStore).persist().join();
        FileBasedSnapshot newerSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(2L, this.snapshotStore).persist().join();
        ((DirectoryAssert)((DirectoryAssert)Assertions.assertThat((Path)this.snapshotsDir).asInstanceOf(DirectoryAssert.factory())).as("ensure both the older and newer snapshots exist", new Object[0])).isDirectoryContainingAllOf(new Path[]{olderSnapshot.getPath(), newerSnapshot.getPath()});
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).hasValue((Object)newerSnapshot);
        ((DirectoryAssert)((DirectoryAssert)Assertions.assertThat((Path)this.snapshotsDir).asInstanceOf(DirectoryAssert.factory())).as("the older snapshots should have been deleted", new Object[0])).isDirectoryContainingExactly(new Path[]{newerSnapshot.getPath(), newerSnapshot.getChecksumPath()});
    }

    @Test
    public void shouldNotLoadCorruptedSnapshot() throws Exception {
        FileBasedSnapshot persistedSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot().persist().join();
        SnapshotChecksum.persist((Path)persistedSnapshot.getChecksumPath(), (ImmutableChecksumsSFV)new SfvChecksumImpl());
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).isEmpty();
    }

    @Test
    public void shouldDeleteSnapshotWithoutChecksumFile() throws IOException {
        FileBasedSnapshot persistedSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot().persist().join();
        Files.delete(persistedSnapshot.getChecksumPath());
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).isEmpty();
        Assertions.assertThat((Path)persistedSnapshot.getDirectory()).doesNotExist();
    }

    @Test
    public void shouldDeleteOlderSnapshotsWithCorruptChecksums() throws IOException {
        FileBasedSnapshotStore otherStore = this.createStore(this.rootDirectory);
        FileBasedSnapshot corruptOlderSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(1L, otherStore).persist().join();
        SnapshotChecksum.persist((Path)corruptOlderSnapshot.getChecksumPath(), (ImmutableChecksumsSFV)new SfvChecksumImpl());
        FileBasedSnapshot newerSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(2L, this.snapshotStore).persist().join();
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).hasValue((Object)newerSnapshot);
        Assertions.assertThat((Path)newerSnapshot.getDirectory()).exists();
        Assertions.assertThat((Path)newerSnapshot.getChecksumPath()).exists();
        Assertions.assertThat((Path)corruptOlderSnapshot.getDirectory()).doesNotExist();
        Assertions.assertThat((Path)corruptOlderSnapshot.getChecksumPath()).doesNotExist();
    }

    @Test
    public void shouldDeleteOlderSnapshotsWithMissingChecksums() throws IOException {
        FileBasedSnapshotStore otherStore = this.createStore(this.rootDirectory);
        FileBasedSnapshot corruptOlderSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(1L, otherStore).persist().join();
        Files.delete(corruptOlderSnapshot.getChecksumPath());
        FileBasedSnapshot newerSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(2L, this.snapshotStore).persist().join();
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).get().isEqualTo((Object)newerSnapshot);
        Assertions.assertThat((Path)newerSnapshot.getDirectory()).exists();
        Assertions.assertThat((Path)newerSnapshot.getChecksumPath()).exists();
        Assertions.assertThat((Path)corruptOlderSnapshot.getDirectory()).doesNotExist();
        Assertions.assertThat((Path)corruptOlderSnapshot.getChecksumPath()).doesNotExist();
    }

    @Test
    public void shouldDeleteCorruptSnapshotsWhenAddingNewSnapshot() throws IOException {
        FileBasedSnapshot olderSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(1L, this.snapshotStore).persist().join();
        FileBasedSnapshotStore otherStore = this.createStore(this.rootDirectory);
        SnapshotChecksum.persist((Path)olderSnapshot.getChecksumPath(), (ImmutableChecksumsSFV)new SfvChecksumImpl());
        FileBasedSnapshot newerSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(2L, otherStore).persist().join();
        Assertions.assertThat((Optional)otherStore.getLatestSnapshot()).get().isEqualTo((Object)newerSnapshot);
        Assertions.assertThat((Path)newerSnapshot.getDirectory()).exists();
        Assertions.assertThat((Path)newerSnapshot.getChecksumPath()).exists();
        Assertions.assertThat((Path)olderSnapshot.getDirectory()).doesNotExist();
        Assertions.assertThat((Path)olderSnapshot.getChecksumPath()).doesNotExist();
    }

    @Test
    public void shouldNotDeleteCorruptSnapshotWithoutValidSnapshot() throws IOException {
        FileBasedSnapshotStore otherStore = this.createStore(this.rootDirectory);
        FileBasedSnapshot corruptSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(1L, otherStore).persist().join();
        SnapshotChecksum.persist((Path)corruptSnapshot.getChecksumPath(), (ImmutableChecksumsSFV)new SfvChecksumImpl());
        this.snapshotStore.close();
        this.snapshotStore = this.createStore(this.rootDirectory);
        Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).isEmpty();
        Assertions.assertThat((Path)corruptSnapshot.getDirectory()).exists();
        Assertions.assertThat((Path)corruptSnapshot.getChecksumPath()).exists();
    }

    @Test
    public void shouldUseChecksumProviderForChecksumsIfSupplied() throws IOException {
        HashMap<String, Long> badChecksums = new HashMap<String, Long>();
        badChecksums.put(SNAPSHOT_CONTENT_FILE_NAME, 123L);
        TestChecksumProvider testChecksumProvider = new TestChecksumProvider(badChecksums);
        FileBasedSnapshotStore store = new FileBasedSnapshotStore(0, 1, this.rootDirectory, (CRC32CChecksumProvider)testChecksumProvider);
        this.scheduler.submitActor((Actor)store).join();
        FileBasedSnapshot takenSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot(1L, store).persist().join();
        ImmutableChecksumsSFV persistedChecksums = SnapshotChecksum.read((Path)takenSnapshot.getChecksumPath());
        Assertions.assertThat((Long)((Long)persistedChecksums.getChecksums().get(SNAPSHOT_CONTENT_FILE_NAME))).isEqualTo(123L);
    }

    @Test
    public void shouldPurgePendingSnapshots() {
        this.takeTransientSnapshot();
        this.snapshotStore.purgePendingSnapshots().join();
        Assertions.assertThat((Path)this.pendingSnapshotsDir).isEmptyDirectory();
    }

    @Test
    public void shouldDeleteSnapshot() {
        FileBasedSnapshot persistedSnapshot = (FileBasedSnapshot)this.takeTransientSnapshot().persist().join();
        persistedSnapshot.delete();
        Assertions.assertThat((Path)persistedSnapshot.getPath()).doesNotExist();
    }

    @Test
    public void shouldNotDeleteReservedSnapshot() {
        PersistedSnapshot reservedSnapshot = (PersistedSnapshot)this.takeTransientSnapshot(1L, this.snapshotStore).persist().join();
        reservedSnapshot.reserve().join();
        PersistedSnapshot newSnapshot = (PersistedSnapshot)this.takeTransientSnapshot(2L, this.snapshotStore).persist().join();
        Assertions.assertThat((Collection)((Collection)this.snapshotStore.getAvailableSnapshots().join())).containsExactlyInAnyOrder((Object[])new PersistedSnapshot[]{reservedSnapshot, newSnapshot});
        Assertions.assertThat((Path)reservedSnapshot.getPath()).exists();
        Assertions.assertThat((Path)newSnapshot.getPath()).exists();
    }

    @Test
    public void shouldNotReserveDeletedSnapshot() throws IOException {
        PersistedSnapshot snapshotToReserve = (PersistedSnapshot)this.takeTransientSnapshot(1L, this.snapshotStore).persist().join();
        this.takeTransientSnapshot(2L, this.snapshotStore).persist().join();
        AssertionsForClassTypes.assertThatThrownBy(() -> snapshotToReserve.reserve().join()).hasCauseInstanceOf(SnapshotException.SnapshotNotFoundException.class);
    }

    @Test
    public void shouldDeleteReleasedSnapshot() {
        PersistedSnapshot reservedSnapshot = (PersistedSnapshot)this.takeTransientSnapshot(1L, this.snapshotStore).persist().join();
        SnapshotReservation reservation = (SnapshotReservation)reservedSnapshot.reserve().join();
        reservation.release().join();
        PersistedSnapshot newSnapshot = (PersistedSnapshot)this.takeTransientSnapshot(3L, this.snapshotStore).persist().join();
        Assertions.assertThat((Collection)((Collection)this.snapshotStore.getAvailableSnapshots().join())).containsExactly((Object[])new PersistedSnapshot[]{newSnapshot});
        Assertions.assertThat((Path)reservedSnapshot.getPath()).doesNotExist();
        Assertions.assertThat((Path)newSnapshot.getPath()).exists();
    }

    @Test
    public void shouldNotDeleteReservedSnapshotUntilAllReservationsAreReleased() {
        PersistedSnapshot reservedSnashot = (PersistedSnapshot)this.takeTransientSnapshot(1L, this.snapshotStore).persist().join();
        SnapshotReservation reservation = (SnapshotReservation)reservedSnashot.reserve().join();
        reservedSnashot.reserve().join();
        reservation.release();
        PersistedSnapshot newSnapshot = (PersistedSnapshot)this.takeTransientSnapshot(2L, this.snapshotStore).persist().join();
        Assertions.assertThat((Collection)((Collection)this.snapshotStore.getAvailableSnapshots().join())).containsExactlyInAnyOrder((Object[])new PersistedSnapshot[]{reservedSnashot, newSnapshot});
        Assertions.assertThat((Path)reservedSnashot.getPath()).exists();
        Assertions.assertThat((Path)newSnapshot.getPath()).exists();
    }

    @Test
    public void shouldRestartWithAReceivedSnapshot() throws IOException {
        HashMap<String, Long> fileChecksums = new HashMap<String, Long>();
        CRC32C fileContentChecksum = new CRC32C();
        fileContentChecksum.update(SNAPSHOT_CONTENT.getBytes(StandardCharsets.UTF_8));
        fileChecksums.put(SNAPSHOT_CONTENT_FILE_NAME, fileContentChecksum.getValue());
        Path receiverStorePath = this.temporaryFolder.newFolder("receiver").toPath();
        FileBasedSnapshotStore store = new FileBasedSnapshotStore(0, PARTITION_ID.intValue(), receiverStorePath, (CRC32CChecksumProvider)new TestChecksumProvider(fileChecksums));
        this.scheduler.submitActor((Actor)store);
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)this.takeTransientSnapshot().persist().join();
        FileBasedReceivedSnapshot receivedSnapshot = (FileBasedReceivedSnapshot)store.newReceivedSnapshot(persistedSnapshot.getId()).join();
        try (SnapshotChunkReader reader = persistedSnapshot.newChunkReader();){
            while (reader.hasNext()) {
                receivedSnapshot.apply((SnapshotChunk)reader.next()).join();
            }
        }
        receivedSnapshot.persist().join();
        store.close();
        FileBasedSnapshotStore restartedStore = new FileBasedSnapshotStore(0, PARTITION_ID.intValue(), receiverStorePath, (CRC32CChecksumProvider)new TestChecksumProvider(fileChecksums));
        this.scheduler.submitActor((Actor)restartedStore).join();
        ((OptionalAssert)Assertions.assertThat((Optional)restartedStore.getLatestSnapshot()).describedAs("The latest snapshot is not detected as corrupted and should be loaded after restart", new Object[0])).hasValueSatisfying(s -> Assertions.assertThat((String)s.getId()).isEqualTo(persistedSnapshot.getId()));
    }

    private boolean createSnapshotDir(Path path) {
        try {
            FileUtil.ensureDirectoryExists((Path)path);
            Files.write(path.resolve(SNAPSHOT_CONTENT_FILE_NAME), SNAPSHOT_CONTENT.getBytes(), StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return true;
    }

    private TransientSnapshot takeTransientSnapshot() {
        return this.takeTransientSnapshot(1L, this.snapshotStore);
    }

    private TransientSnapshot takeTransientSnapshot(long index, FileBasedSnapshotStore store) {
        TransientSnapshot transientSnapshot = (TransientSnapshot)store.newTransientSnapshot(index, 1L, 1L, 0L).get();
        transientSnapshot.take(this::createSnapshotDir);
        return transientSnapshot;
    }

    private FileBasedSnapshotStore createStore(Path root) {
        FileBasedSnapshotStore store = new FileBasedSnapshotStore(0, 1, root, snapshotPath -> Map.of());
        this.scheduler.submitActor((Actor)store).join();
        return store;
    }
}

