/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots;

import io.camunda.zeebe.scheduler.Actor;
import io.camunda.zeebe.scheduler.future.ActorFuture;
import io.camunda.zeebe.scheduler.testing.ActorSchedulerRule;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.PersistedSnapshotListener;
import io.camunda.zeebe.snapshots.SnapshotException;
import io.camunda.zeebe.snapshots.SnapshotId;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotStore;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.FileUtil;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicReference;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.AtomicReferenceAssert;
import org.assertj.core.api.FutureAssert;
import org.assertj.core.api.NotThrownAssert;
import org.assertj.core.api.ObjectAssert;
import org.assertj.core.api.OptionalAssert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class TransientSnapshotTest {
    private static final Map<String, String> SNAPSHOT_FILE_CONTENTS = Map.of("file1", "file1 contents", "file2", "file2 contents");
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public ActorSchedulerRule scheduler = new ActorSchedulerRule();
    private FileBasedSnapshotStore snapshotStore;

    @Before
    public void beforeEach() {
        boolean partitionId = true;
        File root = this.temporaryFolder.getRoot();
        this.snapshotStore = new FileBasedSnapshotStore(0, 1, root.toPath(), snapshotPath -> Map.of());
        this.scheduler.submitActor((Actor)this.snapshotStore).join();
    }

    @Test
    public void shouldHaveCorrectSnapshotId() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        SnapshotId snapshotId = transientSnapshot.snapshotId();
        ((AbstractLongAssert)Assertions.assertThat((long)snapshotId.getIndex()).as("the ID has the right index", new Object[0])).isEqualTo(1L);
        ((AbstractLongAssert)Assertions.assertThat((long)snapshotId.getTerm()).as("the ID has the right term", new Object[0])).isEqualTo(2L);
        ((AbstractLongAssert)Assertions.assertThat((long)snapshotId.getProcessedPosition()).as("the ID has the right processed position", new Object[0])).isEqualTo(3L);
        ((AbstractLongAssert)Assertions.assertThat((long)snapshotId.getExportedPosition()).as("the ID has the right exported position", new Object[0])).isEqualTo(4L);
    }

    @Test
    public void shouldAbortSuccessfullyEvenIfNothingWasWritten() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        ActorFuture didAbort = transientSnapshot.abort();
        ((FutureAssert)Assertions.assertThat((Future)didAbort).as("the transient snapshot was aborted successfully", new Object[0])).succeedsWithin(Duration.ofSeconds(5L));
    }

    @Test
    public void shouldNotCommitUntilPersisted() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        ((OptionalAssert)Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).as("there should be no persisted snapshot", new Object[0])).isEmpty();
    }

    @Test
    public void shouldTakeTransientSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        ActorFuture didWriteSnapshot = transientSnapshot.take(this::writeSnapshot);
        ((NotThrownAssert)Assertions.assertThatNoException().as("the transient snapshot was successfully written", new Object[0])).isThrownBy(() -> ((ActorFuture)didWriteSnapshot).join());
    }

    @Test
    public void shouldCompleteExceptionallyOnExceptionInTake() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        ActorFuture didTakeSnapshot = transientSnapshot.take(p -> {
            throw new RuntimeException("EXPECTED");
        });
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ActorFuture)didTakeSnapshot).join()).as("did not take snapshot due to exception thrown in callback", new Object[0])).hasCauseInstanceOf(RuntimeException.class).hasMessageContaining("EXPECTED");
    }

    @Test
    public void shouldBeAbleToAbortNotStartedSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 3L, 4L).get();
        ActorFuture didAbort = transientSnapshot.abort();
        ((FutureAssert)Assertions.assertThat((Future)didAbort).as("did abort even if nothing was written", new Object[0])).succeedsWithin(Duration.ofSeconds(5L));
    }

    @Test
    public void shouldPersistSnapshotWithCorrectId() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        ((AbstractStringAssert)Assertions.assertThat((String)persistedSnapshot.getId()).as("the persisted snapshot as the same ID as the transient snapshot", new Object[0])).isEqualTo(transientSnapshot.snapshotId().getSnapshotIdAsString());
    }

    @Test
    public void shouldPersistSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        transientSnapshot.take(this::writeSnapshot);
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        ((ObjectAssert)Assertions.assertThat((Object)persistedSnapshot).as("the persisted snapshot should be the latest snapshot", new Object[0])).isEqualTo(this.snapshotStore.getLatestSnapshot().orElseThrow());
    }

    @Test
    public void shouldReplacePreviousSnapshotOnPersist() {
        TransientSnapshot oldSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        oldSnapshot.take(this::writeSnapshot);
        oldSnapshot.persist().join();
        TransientSnapshot newSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(2L, 0L, 1L, 0L).get();
        newSnapshot.take(this::writeSnapshot);
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)newSnapshot.persist().join();
        ((OptionalAssert)Assertions.assertThat((Optional)this.snapshotStore.getLatestSnapshot()).as("the latest snapshot is the last persisted snapshot", new Object[0])).hasValue((Object)persistedSnapshot);
    }

    @Test
    public void shouldNotPersistSnapshotIfIdIsLessThanTheLatestSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(2L, 2L, 3L, 4L).get();
        transientSnapshot.take(this::writeSnapshot);
        PersistedSnapshot previousSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        TransientSnapshot newTransientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 2L, 4L, 5L).get();
        newTransientSnapshot.take(this::writeSnapshot);
        ActorFuture didPersist = newTransientSnapshot.persist();
        ((FutureAssert)Assertions.assertThat((Future)didPersist).as("did not persist snapshot %s with ID less than %s and returns the previous snapshot", new Object[]{previousSnapshot.getId(), newTransientSnapshot.snapshotId().getSnapshotIdAsString()})).succeedsWithin(Duration.ofSeconds(5L)).isEqualTo((Object)previousSnapshot);
    }

    @Test
    public void shouldNotRemoveTransientSnapshotWithGreaterIdOnPersist() {
        TransientSnapshot newerTransientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(2L, 0L, 1L, 0L).get();
        newerTransientSnapshot.take(this::writeSnapshot);
        TransientSnapshot newSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        newSnapshot.take(this::writeSnapshot);
        newSnapshot.persist().join();
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)newerTransientSnapshot.persist().join();
        ((ObjectAssert)Assertions.assertThat((Object)persistedSnapshot).as("the first transient snapshot with greater ID was persisted after the second one", new Object[0])).isEqualTo(this.snapshotStore.getLatestSnapshot().orElseThrow());
    }

    @Test
    public void shouldNotPersistOnTakeException() {
        TransientSnapshot snapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        ActorFuture didTakeSnapshot = snapshot.take(path -> {
            throw new RuntimeException("expected");
        });
        Assertions.assertThatThrownBy(() -> ((ActorFuture)didTakeSnapshot).join()).hasCauseInstanceOf(RuntimeException.class);
        ((FutureAssert)Assertions.assertThat((Future)snapshot.persist()).as("did not persist snapshot as it failed to be taken", new Object[0])).failsWithin(Duration.ofSeconds(5L));
    }

    @Test
    public void shouldNotifyListenersOnNewSnapshot() {
        AtomicReference snapshotRef = new AtomicReference();
        PersistedSnapshotListener listener = snapshotRef::set;
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        this.snapshotStore.addSnapshotListener(listener);
        transientSnapshot.take(this::writeSnapshot).join();
        PersistedSnapshot persistedSnapshot = (PersistedSnapshot)transientSnapshot.persist().join();
        Assertions.assertThat(snapshotRef).hasValue((Object)persistedSnapshot);
    }

    @Test
    public void shouldNotNotifyListenersOnNewSnapshotWhenRemoved() {
        AtomicReference snapshotRef = new AtomicReference();
        PersistedSnapshotListener listener = snapshotRef::set;
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 1L, 0L).get();
        this.snapshotStore.addSnapshotListener(listener);
        this.snapshotStore.removeSnapshotListener(listener);
        transientSnapshot.take(this::writeSnapshot).join();
        transientSnapshot.persist().join();
        ((AtomicReferenceAssert)Assertions.assertThat(snapshotRef).as("the listener was not called and did not record any new snapshot", new Object[0])).hasValue(null);
    }

    @Test
    public void shouldNotTakeSnapshotIfIdAlreadyExists() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 2L, 3L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        transientSnapshot.persist().join();
        Either secondTransientSnapshot = this.snapshotStore.newTransientSnapshot(1L, 0L, 2L, 3L);
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)((SnapshotException)secondTransientSnapshot.getLeft())).as("should have no value since there already exists a transient snapshot with the same ID", new Object[0])).isInstanceOf(SnapshotException.SnapshotAlreadyExistsException.class);
    }

    @Test
    public void shouldNotPersistDeletedTransientSnapshot() {
        TransientSnapshot transientSnapshot = (TransientSnapshot)this.snapshotStore.newTransientSnapshot(1L, 0L, 2L, 3L).get();
        transientSnapshot.take(this::writeSnapshot).join();
        this.snapshotStore.purgePendingSnapshots().join();
        ActorFuture persisted = transientSnapshot.persist();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ActorFuture)persisted).join()).as("did not persist a deleted transient snapshot", new Object[0])).hasCauseInstanceOf(SnapshotException.SnapshotNotFoundException.class);
    }

    private void writeSnapshot(Path path) {
        try {
            FileUtil.ensureDirectoryExists((Path)path);
            for (Map.Entry<String, String> entry : SNAPSHOT_FILE_CONTENTS.entrySet()) {
                Path fileName = path.resolve(entry.getKey());
                Files.writeString(fileName, (CharSequence)entry.getValue(), StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}

