/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.exporter.opensearch;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.search.connect.plugin.PluginRepository;
import io.camunda.zeebe.exporter.api.Exporter;
import io.camunda.zeebe.exporter.api.ExporterException;
import io.camunda.zeebe.exporter.api.context.Context;
import io.camunda.zeebe.exporter.api.context.Controller;
import io.camunda.zeebe.exporter.opensearch.OpensearchClient;
import io.camunda.zeebe.exporter.opensearch.OpensearchExporterConfiguration;
import io.camunda.zeebe.exporter.opensearch.OpensearchExporterException;
import io.camunda.zeebe.exporter.opensearch.OpensearchExporterMetadata;
import io.camunda.zeebe.exporter.opensearch.OpensearchRecordCounters;
import io.camunda.zeebe.exporter.opensearch.RecordSequence;
import io.camunda.zeebe.exporter.opensearch.RestClientFactory;
import io.camunda.zeebe.exporter.opensearch.dto.GetIndexStateManagementPolicyResponse;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.IOException;
import java.time.Duration;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import org.apache.http.HttpRequestInterceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpensearchExporter
implements Exporter {
    private static final int RECOMMENDED_MAX_BULK_MEMORY_LIMIT = 0x6400000;
    private Logger log = LoggerFactory.getLogger((String)this.getClass().getPackageName());
    private final ObjectMapper exporterMetadataObjectMapper = new ObjectMapper();
    private final OpensearchExporterMetadata exporterMetadata = new OpensearchExporterMetadata();
    private final PluginRepository pluginRepository = new PluginRepository();
    private Controller controller;
    private OpensearchExporterConfiguration configuration;
    private OpensearchClient client;
    private OpensearchRecordCounters recordCounters;
    private MeterRegistry meterRegistry;
    private long lastPosition = -1L;
    private Set<String> indexTemplatesCreated;

    public void configure(Context context) {
        this.log = context.getLogger();
        this.configuration = (OpensearchExporterConfiguration)context.getConfiguration().instantiate(OpensearchExporterConfiguration.class);
        this.log.debug("Exporter configured with {}", (Object)this.configuration);
        this.validate(this.configuration);
        this.pluginRepository.load(this.configuration.getInterceptorPlugins());
        context.setFilter((Context.RecordFilter)new OpensearchRecordFilter(this.configuration));
        this.indexTemplatesCreated = new HashSet<String>();
        this.meterRegistry = context.getMeterRegistry();
    }

    public void open(Controller controller) {
        this.controller = controller;
        this.client = this.createClient();
        this.recordCounters = controller.readMetadata().map(this::deserializeExporterMetadata).map(OpensearchExporterMetadata::getRecordCountersByValueType).filter(counters -> !counters.isEmpty()).map(OpensearchRecordCounters::new).orElse(new OpensearchRecordCounters());
        this.scheduleDelayedFlush();
        this.log.info("Exporter opened");
    }

    public void close() {
        if (this.client != null) {
            try {
                this.flush();
                this.updateLastExportedPosition();
            }
            catch (Exception e) {
                this.log.warn("Failed to flush records before closing exporter.", (Throwable)e);
            }
            try {
                this.client.close();
            }
            catch (Exception e) {
                this.log.warn("Failed to close opensearch client", (Throwable)e);
            }
        }
        try {
            this.pluginRepository.close();
        }
        catch (Exception e) {
            this.log.warn("Failed to close plugin repository", (Throwable)e);
        }
        this.log.info("Exporter closed");
    }

    public void export(Record<?> record) {
        RecordSequence recordSequence;
        boolean isRecordIndexedToBatch;
        if (!this.indexTemplatesCreated.contains(record.getBrokerVersion())) {
            this.createIndexTemplates(record.getBrokerVersion());
            this.updateRetentionPolicyForExistingIndices();
        }
        if (isRecordIndexedToBatch = this.client.index(record, recordSequence = this.recordCounters.getNextRecordSequence(record))) {
            this.recordCounters.updateRecordCounters(record, recordSequence);
        }
        this.lastPosition = record.getPosition();
        if (this.client.shouldFlush()) {
            this.flush();
            this.updateLastExportedPosition();
        }
    }

    private void validate(OpensearchExporterConfiguration configuration) {
        Integer numberOfShards;
        if (configuration.index.prefix != null && configuration.index.prefix.contains("_")) {
            throw new ExporterException(String.format("Opensearch prefix must not contain underscore. Current value: %s", configuration.index.prefix));
        }
        if (configuration.bulk.memoryLimit > 0x6400000) {
            this.log.warn("The bulk memory limit is set to more than {} bytes. It is recommended to set the limit between 5 to 15 MB.", (Object)0x6400000);
        }
        if ((numberOfShards = configuration.index.getNumberOfShards()) != null && numberOfShards < 1) {
            throw new ExporterException(String.format("Opensearch numberOfShards must be >= 1. Current value: %d", numberOfShards));
        }
        Integer numberOfReplicas = configuration.index.getNumberOfReplicas();
        if (numberOfReplicas != null && numberOfReplicas < 0) {
            throw new ExporterException(String.format("Opensearch numberOfReplicas must be >= 0. Current value: %d", numberOfReplicas));
        }
    }

    protected OpensearchClient createClient() {
        return new OpensearchClient(this.configuration, this.meterRegistry, RestClientFactory.of(this.configuration, new HttpRequestInterceptor[]{this.pluginRepository.asRequestInterceptor()}));
    }

    private void flushAndReschedule() {
        try {
            this.flush();
            this.updateLastExportedPosition();
        }
        catch (Exception e) {
            this.log.warn("Unexpected exception occurred on periodically flushing bulk, will retry later.", (Throwable)e);
        }
        this.scheduleDelayedFlush();
    }

    private void scheduleDelayedFlush() {
        this.controller.scheduleCancellableTask(Duration.ofSeconds(this.configuration.bulk.delay), this::flushAndReschedule);
    }

    private void flush() {
        this.client.flush();
    }

    private void updateLastExportedPosition() {
        this.exporterMetadata.setRecordCountersByValueType(this.recordCounters.getRecordCounters());
        byte[] serializeExporterMetadata = this.serializeExporterMetadata(this.exporterMetadata);
        this.controller.updateLastExportedRecordPosition(this.lastPosition, serializeExporterMetadata);
    }

    private byte[] serializeExporterMetadata(OpensearchExporterMetadata metadata) {
        try {
            return this.exporterMetadataObjectMapper.writeValueAsBytes((Object)metadata);
        }
        catch (JsonProcessingException e) {
            throw new OpensearchExporterException("Failed to serialize exporter metadata", e);
        }
    }

    private OpensearchExporterMetadata deserializeExporterMetadata(byte[] metadata) {
        try {
            return (OpensearchExporterMetadata)this.exporterMetadataObjectMapper.readValue(metadata, OpensearchExporterMetadata.class);
        }
        catch (IOException e) {
            throw new OpensearchExporterException("Failed to deserialize exporter metadata", e);
        }
    }

    private void createIndexTemplates(String version) {
        if (this.configuration.retention.isEnabled()) {
            this.createIndexStateManagementPolicy();
        } else {
            this.deleteIndexStateManagementPolicy();
        }
        OpensearchExporterConfiguration.IndexConfiguration index = this.configuration.index;
        if (index.createTemplate) {
            this.createComponentTemplate();
            if (index.deployment) {
                this.createValueIndexTemplate(ValueType.DEPLOYMENT, version);
            }
            if (index.process) {
                this.createValueIndexTemplate(ValueType.PROCESS, version);
            }
            if (index.error) {
                this.createValueIndexTemplate(ValueType.ERROR, version);
            }
            if (index.incident) {
                this.createValueIndexTemplate(ValueType.INCIDENT, version);
            }
            if (index.job) {
                this.createValueIndexTemplate(ValueType.JOB, version);
            }
            if (index.jobBatch) {
                this.createValueIndexTemplate(ValueType.JOB_BATCH, version);
            }
            if (index.message) {
                this.createValueIndexTemplate(ValueType.MESSAGE, version);
            }
            if (index.messageBatch) {
                this.createValueIndexTemplate(ValueType.MESSAGE_BATCH, version);
            }
            if (index.messageSubscription) {
                this.createValueIndexTemplate(ValueType.MESSAGE_SUBSCRIPTION, version);
            }
            if (index.variable) {
                this.createValueIndexTemplate(ValueType.VARIABLE, version);
            }
            if (index.variableDocument) {
                this.createValueIndexTemplate(ValueType.VARIABLE_DOCUMENT, version);
            }
            if (index.processInstance) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE, version);
            }
            if (index.processInstanceBatch) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_BATCH, version);
            }
            if (index.processInstanceCreation) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_CREATION, version);
            }
            if (index.processInstanceMigration) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_MIGRATION, version);
            }
            if (index.processInstanceModification) {
                this.createValueIndexTemplate(ValueType.PROCESS_INSTANCE_MODIFICATION, version);
            }
            if (index.processMessageSubscription) {
                this.createValueIndexTemplate(ValueType.PROCESS_MESSAGE_SUBSCRIPTION, version);
            }
            if (index.decisionRequirements) {
                this.createValueIndexTemplate(ValueType.DECISION_REQUIREMENTS, version);
            }
            if (index.decision) {
                this.createValueIndexTemplate(ValueType.DECISION, version);
            }
            if (index.decisionEvaluation) {
                this.createValueIndexTemplate(ValueType.DECISION_EVALUATION, version);
            }
            if (index.checkpoint) {
                this.createValueIndexTemplate(ValueType.CHECKPOINT, version);
            }
            if (index.timer) {
                this.createValueIndexTemplate(ValueType.TIMER, version);
            }
            if (index.messageStartEventSubscription) {
                this.createValueIndexTemplate(ValueType.MESSAGE_START_EVENT_SUBSCRIPTION, version);
            }
            if (index.processEvent) {
                this.createValueIndexTemplate(ValueType.PROCESS_EVENT, version);
            }
            if (index.deploymentDistribution) {
                this.createValueIndexTemplate(ValueType.DEPLOYMENT_DISTRIBUTION, version);
            }
            if (index.escalation) {
                this.createValueIndexTemplate(ValueType.ESCALATION, version);
            }
            if (index.signal) {
                this.createValueIndexTemplate(ValueType.SIGNAL, version);
            }
            if (index.signalSubscription) {
                this.createValueIndexTemplate(ValueType.SIGNAL_SUBSCRIPTION, version);
            }
            if (index.resourceDeletion) {
                this.createValueIndexTemplate(ValueType.RESOURCE_DELETION, version);
            }
            if (index.commandDistribution) {
                this.createValueIndexTemplate(ValueType.COMMAND_DISTRIBUTION, version);
            }
            if (index.form) {
                this.createValueIndexTemplate(ValueType.FORM, version);
            }
            if (index.userTask) {
                this.createValueIndexTemplate(ValueType.USER_TASK, version);
            }
            if (index.compensationSubscription) {
                this.createValueIndexTemplate(ValueType.COMPENSATION_SUBSCRIPTION, version);
            }
            if (index.messageCorrelation) {
                this.createValueIndexTemplate(ValueType.MESSAGE_CORRELATION, version);
            }
            if (index.user) {
                this.createValueIndexTemplate(ValueType.USER, version);
            }
            if (index.authorization) {
                this.createValueIndexTemplate(ValueType.AUTHORIZATION, version);
            }
        }
        this.indexTemplatesCreated.add(version);
    }

    private void createIndexStateManagementPolicy() {
        Optional<GetIndexStateManagementPolicyResponse> policyOptional = this.client.getIndexStateManagementPolicy();
        if (policyOptional.isEmpty()) {
            if (!this.client.createIndexStateManagementPolicy()) {
                this.log.warn("Failed to acknowledge the creation of the Index State Management Policy");
            }
            return;
        }
        GetIndexStateManagementPolicyResponse policy = policyOptional.get();
        if (!policy.equalsConfiguration(this.configuration) && !this.client.updateIndexStateManagementPolicy(policy.seqNo(), policy.primaryTerm())) {
            this.log.warn("Failed to acknowledge the update of the Index State Management Policy");
        }
    }

    private void deleteIndexStateManagementPolicy() {
        Optional<GetIndexStateManagementPolicyResponse> policyOptional = this.client.getIndexStateManagementPolicy();
        if (policyOptional.isEmpty()) {
            return;
        }
        if (!this.client.deleteIndexStateManagementPolicy()) {
            this.log.warn("Failed to acknowledge the deletion of the Index State Management Policy");
        }
    }

    private void createComponentTemplate() {
        if (!this.client.putComponentTemplate()) {
            this.log.warn("Failed to acknowledge the creation or update of the component template");
        }
    }

    private void createValueIndexTemplate(ValueType valueType, String version) {
        if (!this.client.putIndexTemplate(valueType, version)) {
            this.log.warn("Failed to acknowledge the creation or update of the index template for value type {}", (Object)valueType);
        }
    }

    private void updateRetentionPolicyForExistingIndices() {
        boolean successful = this.configuration.retention.isEnabled() ? this.client.bulkAddISMPolicyToAllZeebeIndices() : this.client.bulkRemoveISMPolicyToAllZeebeIndices();
        if (!successful) {
            this.log.warn("Failed to acknowledge the the update of retention policy for existing indices");
        }
    }

    private static class OpensearchRecordFilter
    implements Context.RecordFilter {
        private final OpensearchExporterConfiguration configuration;

        OpensearchRecordFilter(OpensearchExporterConfiguration configuration) {
            this.configuration = configuration;
        }

        public boolean acceptType(RecordType recordType) {
            return this.configuration.shouldIndexRecordType(recordType);
        }

        public boolean acceptValue(ValueType valueType) {
            return this.configuration.shouldIndexValueType(valueType);
        }
    }
}

