/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.logstreams.impl.log;

import io.camunda.zeebe.logstreams.log.LogAppendEntry;
import io.camunda.zeebe.logstreams.log.LogStreamReader;
import io.camunda.zeebe.logstreams.log.LogStreamWriter;
import io.camunda.zeebe.logstreams.log.LoggedEvent;
import io.camunda.zeebe.logstreams.log.WriteContext;
import io.camunda.zeebe.logstreams.util.LogStreamReaderRule;
import io.camunda.zeebe.logstreams.util.LogStreamRule;
import io.camunda.zeebe.logstreams.util.TestEntry;
import io.camunda.zeebe.protocol.impl.record.RecordMetadata;
import io.camunda.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.camunda.zeebe.util.ByteValue;
import io.camunda.zeebe.util.buffer.BufferReader;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.assertj.core.api.Assertions;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.rules.TestRule;

public final class LogStreamReaderTest {
    private static final int LOG_SEGMENT_SIZE = (int)ByteValue.ofMegabytes((long)4L);
    private final LogStreamRule logStreamRule = LogStreamRule.startByDefault(builder -> builder.withMaxFragmentSize(LOG_SEGMENT_SIZE));
    private final LogStreamReaderRule readerRule = new LogStreamReaderRule(this.logStreamRule);
    @Rule
    public final RuleChain ruleChain = RuleChain.outerRule((TestRule)this.logStreamRule).around((TestRule)this.readerRule);
    private LogStreamReader reader;
    private LogStreamWriter writer;

    @Before
    public void setUp() {
        this.reader = this.readerRule.getLogStreamReader();
        this.writer = this.logStreamRule.getLogStream().newBlockingLogStreamWriter();
    }

    @Test
    public void shouldNotHaveNextIfReaderIsClosed() {
        LogStreamReader reader = this.logStreamRule.getLogStreamReader();
        reader.close();
        Assertions.assertThat((boolean)reader.hasNext()).isFalse();
    }

    @Test
    public void shouldThrowExceptionIfReaderClosedOnNext() {
        LogStreamReader reader = this.logStreamRule.getLogStreamReader();
        reader.close();
        Assertions.assertThatCode(() -> reader.next()).isInstanceOf(NoSuchElementException.class);
    }

    @Test
    public void shouldNotHaveNext() {
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldHaveNext() {
        LogAppendEntry entry = TestEntry.ofKey(5L);
        long position = (Long)this.writer.tryWrite(WriteContext.internal(), entry).get();
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        LoggedEvent next = (LoggedEvent)this.reader.next();
        TestEntry.TestEntryAssert.assertThatEntry(entry).matchesLoggedEvent(next);
        Assertions.assertThat((long)next.getKey()).isEqualTo(entry.key());
        Assertions.assertThat((long)next.getPosition()).isEqualTo(position);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldThrowNoSuchElementExceptionOnNextCall() {
        Assertions.assertThatCode(() -> this.reader.next()).isInstanceOf(NoSuchElementException.class);
    }

    @Test
    public void shouldReturnPositionOfCurrentLoggedEvent() {
        long position = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        this.reader.seekToFirstEvent();
        Assertions.assertThat((long)this.reader.getPosition()).isEqualTo(position);
    }

    @Test
    public void shouldReturnNoPositionIfNotActiveOrInitialized() {
        this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults());
        Assertions.assertThat((long)this.reader.getPosition()).isEqualTo(-1L);
    }

    @Test
    public void shouldReopenAndReturnLoggedEvent() {
        this.reader.close();
        LogAppendEntry entry = TestEntry.ofKey(5L);
        long position = (Long)this.writer.tryWrite(WriteContext.internal(), entry).get();
        this.reader = this.readerRule.resetReader();
        LoggedEvent loggedEvent = this.readerRule.nextEvent();
        Assertions.assertThat((long)loggedEvent.getKey()).isEqualTo(entry.key());
        Assertions.assertThat((long)loggedEvent.getPosition()).isEqualTo(position);
    }

    @Test
    public void shouldWrapAndSeekToEvent() {
        this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults());
        LogAppendEntry entry = TestEntry.ofKey(5L);
        long secondPos = (Long)this.writer.tryWrite(WriteContext.internal(), entry).get();
        this.reader = this.logStreamRule.newLogStreamReader();
        this.reader.seek(secondPos);
        LoggedEvent loggedEvent = (LoggedEvent)this.reader.next();
        TestEntry.TestEntryAssert.assertThatEntry(entry).matchesLoggedEvent(loggedEvent);
        Assertions.assertThat((long)loggedEvent.getKey()).isEqualTo(entry.key());
        Assertions.assertThat((long)loggedEvent.getPosition()).isEqualTo(secondPos);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldReturnLastEventAfterSeekToLastEvent() {
        int eventCount = 10;
        long lastPosition = this.writeEvents(10);
        long seekedPosition = this.reader.seekToEnd();
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
        Assertions.assertThat((long)lastPosition).isEqualTo(seekedPosition);
    }

    @Test
    public void shouldReturnNextAfterSeekToEnd() {
        int eventCount = 10;
        long lastEventPosition = this.writeEvents(10);
        long seekedPosition = this.reader.seekToEnd();
        long newLastPosition = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        Assertions.assertThat((long)lastEventPosition).isEqualTo(seekedPosition);
        Assertions.assertThat((long)newLastPosition).isGreaterThan(seekedPosition);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        LoggedEvent loggedEvent = (LoggedEvent)this.reader.next();
        Assertions.assertThat((long)loggedEvent.getPosition()).isEqualTo(newLastPosition);
    }

    @Test
    public void shouldSeekToEnd() {
        int eventCount = 1000;
        long lastPosition = this.writeEvents(1000);
        long seekedPosition = this.reader.seekToEnd();
        Assertions.assertThat((long)lastPosition).isEqualTo(seekedPosition);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldIterateOverManyEventsInOrder() {
        int eventCount = 10000;
        List<LogAppendEntry> entries = IntStream.range(0, 10000).mapToObj(TestEntry::ofKey).toList();
        this.writer.tryWrite(WriteContext.internal(), entries);
        this.assertReaderHasEntries(entries);
    }

    @Test
    public void shouldSeekToMiddleOfBatch() {
        long firstBatchLastPosition = this.writeEvents(4);
        this.writeEvents(8);
        this.reader.seekToNextEvent(firstBatchLastPosition + 1L);
        Assertions.assertThat((Iterator)this.reader).hasNext();
        Assertions.assertThat((long)((LoggedEvent)this.reader.next()).getPosition()).isEqualTo(firstBatchLastPosition + 2L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
    }

    @Test
    public void shouldIterateMultipleTimes() {
        int eventCount = 500;
        List<LogAppendEntry> entries = IntStream.range(0, 500).mapToObj(TestEntry::ofKey).toList();
        this.writer.tryWrite(WriteContext.internal(), entries);
        this.assertReaderHasEntries(entries);
        this.assertReaderHasEntries(entries);
        this.assertReaderHasEntries(entries);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    private void assertReaderHasEntries(List<LogAppendEntry> entries) {
        long lastPosition = -1L;
        this.reader.seekToFirstEvent();
        for (int i = 0; i < entries.size(); ++i) {
            LoggedEvent loggedEvent = this.readerRule.nextEvent();
            Assertions.assertThat((long)loggedEvent.getPosition()).isGreaterThan(lastPosition);
            Assertions.assertThat((long)loggedEvent.getKey()).isEqualTo((long)i);
            TestEntry.TestEntryAssert.assertThatEntry(entries.get(i)).matchesLoggedEvent(loggedEvent);
            lastPosition = loggedEvent.getPosition();
        }
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldSeekToFirstEvent() {
        long firstPosition = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        this.writeEvents(2);
        this.reader.seekToFirstEvent();
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((long)((LoggedEvent)this.reader.next()).getPosition()).isEqualTo(firstPosition);
    }

    @Test
    public void shouldSeekToFirstPositionWhenPositionBeforeFirstEvent() {
        long firstPosition = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        this.writeEvents(2);
        this.reader.seek(firstPosition - 1L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((long)((LoggedEvent)this.reader.next()).getPosition()).isEqualTo(firstPosition);
    }

    @Test
    public void shouldNotSeekToEventBeyondLastEvent() {
        long lastEventPosition = this.writeEvents(100);
        this.reader.seek(lastEventPosition + 1L);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldReturnNegativeOnSeekToEndOfEmptyLog() {
        LogStreamReader reader = this.logStreamRule.getLogStreamReader();
        long result = reader.seekToEnd();
        Assertions.assertThat((long)result).isNegative();
    }

    @Test
    public void shouldSeekToNextEventWhenThereIsNone() {
        long lastEventPosition = this.writeEvents(10);
        boolean positionExists = this.reader.seekToNextEvent(lastEventPosition);
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
        Assertions.assertThat((boolean)positionExists).isTrue();
        Assertions.assertThat((long)this.reader.getPosition()).isEqualTo(lastEventPosition);
    }

    @Test
    public void shouldSeekToNextEvent() {
        long lastEventPosition = this.writeEvents(10);
        boolean positionExists = this.reader.seekToNextEvent(lastEventPosition - 1L);
        Assertions.assertThat((boolean)positionExists).isTrue();
        Assertions.assertThat((Iterator)this.reader).hasNext();
        Assertions.assertThat((long)((LoggedEvent)this.reader.next()).getPosition()).isEqualTo(lastEventPosition);
    }

    @Test
    public void shouldNotSeekToNextEvent() {
        long lastEventPosition = this.writeEvents(10);
        boolean positionExists = this.reader.seekToNextEvent(lastEventPosition + 1L);
        Assertions.assertThat((boolean)positionExists).isFalse();
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
    }

    @Test
    public void shouldSeekToFirstEventWhenNextIsNegative() {
        long firstEventPosition = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        this.writeEvents(10);
        this.reader.seekToEnd();
        boolean positionExists = this.reader.seekToNextEvent(-1L);
        Assertions.assertThat((boolean)positionExists).isTrue();
        Assertions.assertThat((Iterator)this.reader).hasNext();
        Assertions.assertThat((long)((LoggedEvent)this.reader.next()).getPosition()).isEqualTo(firstEventPosition);
    }

    @Test
    public void shouldPeekFirstEvent() {
        Long eventPosition1 = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults());
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        LoggedEvent nextEvent = this.reader.peekNext();
        Assertions.assertThat((long)nextEvent.getPosition()).isEqualTo((Object)eventPosition1);
    }

    @Test
    public void shouldPeekNextEvent() {
        Long eventPosition1 = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        Long eventPosition2 = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        Assertions.assertThat((long)((LoggedEvent)this.reader.next()).getPosition()).isEqualTo((Object)eventPosition1);
        LoggedEvent nextEvent = this.reader.peekNext();
        Assertions.assertThat((long)nextEvent.getPosition()).isEqualTo((Object)eventPosition2);
    }

    @Test
    public void shouldPeekAndReadNextEvent() {
        Long eventPosition1 = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        Long eventPosition2 = (Long)this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        LoggedEvent event = (LoggedEvent)this.reader.next();
        Assertions.assertThat((long)event.getPosition()).isEqualTo((Object)eventPosition1);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        LoggedEvent peekedEvent = this.reader.peekNext();
        Assertions.assertThat((long)peekedEvent.getPosition()).isEqualTo((Object)eventPosition2);
        Assertions.assertThat((boolean)this.reader.hasNext()).isTrue();
        LoggedEvent nextEvent = (LoggedEvent)this.reader.next();
        Assertions.assertThat((long)nextEvent.getPosition()).isEqualTo((Object)eventPosition2);
    }

    @Test
    public void shouldThrowNoSuchElementExceptionOnPeek() {
        Assertions.assertThat((boolean)this.reader.hasNext()).isFalse();
        Assertions.assertThatThrownBy(() -> ((LogStreamReader)this.reader).peekNext()).isInstanceOf(NoSuchElementException.class);
    }

    @Test
    public void shouldNotInvalidateEventsOnClose() {
        this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        this.writer.tryWrite(WriteContext.internal(), TestEntry.ofDefaults()).get();
        LoggedEvent event = (LoggedEvent)this.reader.next();
        LoggedEvent nextEvent = this.reader.peekNext();
        this.reader.close();
        Assertions.assertThatCode(() -> event.readValue((BufferReader)new UnifiedRecordValue(1))).doesNotThrowAnyException();
        Assertions.assertThatCode(() -> event.readMetadata((BufferReader)new RecordMetadata())).doesNotThrowAnyException();
        Assertions.assertThatCode(() -> nextEvent.readValue((BufferReader)new UnifiedRecordValue(1))).doesNotThrowAnyException();
        Assertions.assertThatCode(() -> nextEvent.readMetadata((BufferReader)new RecordMetadata())).doesNotThrowAnyException();
    }

    private long writeEvents(int eventCount) {
        List entries = IntStream.rangeClosed(1, eventCount).mapToObj(TestEntry::ofKey).collect(Collectors.toList());
        return (Long)this.writer.tryWrite(WriteContext.internal(), entries).get();
    }
}

