/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.exporter;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.search.connect.plugin.PluginRepository;
import io.camunda.zeebe.exporter.ElasticsearchClient;
import io.camunda.zeebe.exporter.ElasticsearchExporterConfiguration;
import io.camunda.zeebe.exporter.ElasticsearchExporterException;
import io.camunda.zeebe.exporter.ElasticsearchExporterMetadata;
import io.camunda.zeebe.exporter.ElasticsearchExporterSchemaManager;
import io.camunda.zeebe.exporter.ElasticsearchRecordCounters;
import io.camunda.zeebe.exporter.RecordSequence;
import io.camunda.zeebe.exporter.RestClientFactory;
import io.camunda.zeebe.exporter.api.Exporter;
import io.camunda.zeebe.exporter.api.ExporterException;
import io.camunda.zeebe.exporter.api.context.Context;
import io.camunda.zeebe.exporter.api.context.Controller;
import io.camunda.zeebe.protocol.record.Record;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.ValueType;
import io.camunda.zeebe.util.SemanticVersion;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.IOException;
import java.time.Duration;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.apache.http.HttpRequestInterceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticsearchExporter
implements Exporter {
    private static final String PATTERN_MIN_AGE_FORMAT = "^[0-9]+[dhms]$";
    private static final Predicate<String> CHECKER_MIN_AGE = Pattern.compile("^[0-9]+[dhms]$").asPredicate();
    private static final int RECOMMENDED_MAX_BULK_MEMORY_LIMIT = 0x6400000;
    private Logger log = LoggerFactory.getLogger((String)this.getClass().getPackageName());
    private final ObjectMapper exporterMetadataObjectMapper = new ObjectMapper();
    private final ElasticsearchExporterMetadata exporterMetadata = new ElasticsearchExporterMetadata();
    private final PluginRepository pluginRepository = new PluginRepository();
    private Controller controller;
    private ElasticsearchExporterConfiguration configuration;
    private ElasticsearchClient client;
    private ElasticsearchRecordCounters recordCounters;
    private MeterRegistry registry;
    private ElasticsearchExporterSchemaManager schemaManager;
    private long lastPosition = -1L;

    public void configure(Context context) {
        this.log = context.getLogger();
        this.configuration = (ElasticsearchExporterConfiguration)context.getConfiguration().instantiate(ElasticsearchExporterConfiguration.class);
        this.log.debug("Exporter configured with {}", (Object)this.configuration);
        this.validate(this.configuration);
        this.pluginRepository.load(this.configuration.getInterceptorPlugins());
        context.setFilter((Context.RecordFilter)new ElasticsearchRecordFilter(this.configuration));
        this.schemaManager = new ElasticsearchExporterSchemaManager(this.client, this.configuration);
        this.registry = context.getMeterRegistry();
    }

    public void open(Controller controller) {
        this.controller = controller;
        this.client = this.createClient();
        this.recordCounters = controller.readMetadata().map(this::deserializeExporterMetadata).map(ElasticsearchExporterMetadata::getRecordCountersByValueType).map(ElasticsearchRecordCounters::new).orElse(new ElasticsearchRecordCounters());
        this.scheduleDelayedFlush();
        this.schemaManager = new ElasticsearchExporterSchemaManager(this.client, this.configuration);
        this.log.info("Exporter opened");
    }

    public void close() {
        if (this.client != null) {
            try {
                this.flush();
                this.updateLastExportedPosition();
            }
            catch (Exception e) {
                this.log.warn("Failed to flush records before closing exporter.", (Throwable)e);
            }
            try {
                this.client.close();
            }
            catch (Exception e) {
                this.log.warn("Failed to close elasticsearch client", (Throwable)e);
            }
        }
        try {
            this.pluginRepository.close();
        }
        catch (Exception e) {
            this.log.warn("Failed to close plugin repository", (Throwable)e);
        }
        this.log.info("Exporter closed");
    }

    public void export(Record<?> record) {
        if (!this.shouldExportRecord(record)) {
            this.lastPosition = record.getPosition();
            return;
        }
        this.schemaManager.createSchema(record.getBrokerVersion());
        RecordSequence recordSequence = this.recordCounters.getNextRecordSequence(record);
        boolean isRecordIndexedToBatch = this.client.index(record, recordSequence);
        if (isRecordIndexedToBatch) {
            this.recordCounters.updateRecordCounters(record, recordSequence);
        }
        this.lastPosition = record.getPosition();
        if (this.client.shouldFlush()) {
            this.flush();
            this.updateLastExportedPosition();
        }
    }

    private void validate(ElasticsearchExporterConfiguration configuration) {
        Integer numberOfShards;
        if (configuration.index.prefix != null && configuration.index.prefix.contains("_")) {
            throw new ExporterException(String.format("Elasticsearch prefix must not contain underscore. Current value: %s", configuration.index.prefix));
        }
        if (configuration.bulk.memoryLimit > 0x6400000) {
            this.log.warn("The bulk memory limit is set to more than {} bytes. It is recommended to set the limit between 5 to 15 MB.", (Object)0x6400000);
        }
        if ((numberOfShards = configuration.index.getNumberOfShards()) != null && numberOfShards < 1) {
            throw new ExporterException(String.format("Elasticsearch numberOfShards must be >= 1. Current value: %d", numberOfShards));
        }
        Integer numberOfReplicas = configuration.index.getNumberOfReplicas();
        if (numberOfReplicas != null && numberOfReplicas < 0) {
            throw new ExporterException(String.format("Elasticsearch numberOfReplicas must be >= 0. Current value: %d", numberOfReplicas));
        }
        String minimumAge = configuration.retention.getMinimumAge();
        if (minimumAge != null && !CHECKER_MIN_AGE.test(minimumAge)) {
            throw new ExporterException(String.format("Elasticsearch minimumAge '%s' must match pattern '%s', but didn't.", minimumAge, PATTERN_MIN_AGE_FORMAT));
        }
        String indexSuffixDatePattern = configuration.index.indexSuffixDatePattern;
        try {
            DateTimeFormatter.ofPattern(indexSuffixDatePattern).withZone(ZoneOffset.UTC);
        }
        catch (IllegalArgumentException iae) {
            throw new ExporterException(String.format("Expected a valid date format pattern for the given elasticsearch indexSuffixDatePattern, but '%s' was not. Examples are: 'yyyy-MM-dd' or 'yyyy-MM-dd_HH'", indexSuffixDatePattern), (Throwable)iae);
        }
    }

    protected ElasticsearchClient createClient() {
        return new ElasticsearchClient(this.configuration, this.registry, RestClientFactory.of(this.configuration, new HttpRequestInterceptor[]{this.pluginRepository.asRequestInterceptor()}));
    }

    private void flushAndReschedule() {
        try {
            this.flush();
            this.updateLastExportedPosition();
        }
        catch (Exception e) {
            this.log.warn("Unexpected exception occurred on periodically flushing bulk, will retry later.", (Throwable)e);
        }
        this.scheduleDelayedFlush();
    }

    private void scheduleDelayedFlush() {
        this.controller.scheduleCancellableTask(Duration.ofSeconds(this.configuration.bulk.delay), this::flushAndReschedule);
    }

    private void flush() {
        this.client.flush();
    }

    private void updateLastExportedPosition() {
        this.exporterMetadata.setRecordCountersByValueType(this.recordCounters.getRecordCounters());
        byte[] serializeExporterMetadata = this.serializeExporterMetadata(this.exporterMetadata);
        this.controller.updateLastExportedRecordPosition(this.lastPosition, serializeExporterMetadata);
    }

    private byte[] serializeExporterMetadata(ElasticsearchExporterMetadata metadata) {
        try {
            return this.exporterMetadataObjectMapper.writeValueAsBytes((Object)metadata);
        }
        catch (JsonProcessingException e) {
            throw new ElasticsearchExporterException("Failed to serialize exporter metadata", e);
        }
    }

    private ElasticsearchExporterMetadata deserializeExporterMetadata(byte[] metadata) {
        try {
            return (ElasticsearchExporterMetadata)this.exporterMetadataObjectMapper.readValue(metadata, ElasticsearchExporterMetadata.class);
        }
        catch (IOException e) {
            throw new ElasticsearchExporterException("Failed to deserialize exporter metadata", e);
        }
    }

    private boolean shouldExportRecord(Record<?> record) {
        SemanticVersion recordVersion = this.getVersion(record.getBrokerVersion());
        if (this.configuration.getIsIncludeEnabledRecords() || recordVersion.major() == 8 && recordVersion.minor() < 8) {
            return true;
        }
        return this.configuration.shouldIndexRequiredValueType(record.getValueType());
    }

    private SemanticVersion getVersion(String version) {
        return (SemanticVersion)SemanticVersion.parse((String)version).orElseThrow(() -> new IllegalArgumentException("Unsupported record broker version: [" + version + "] Must be a semantic version."));
    }

    private static class ElasticsearchRecordFilter
    implements Context.RecordFilter {
        private final ElasticsearchExporterConfiguration configuration;

        ElasticsearchRecordFilter(ElasticsearchExporterConfiguration configuration) {
            this.configuration = configuration;
        }

        public boolean acceptType(RecordType recordType) {
            return this.configuration.shouldIndexRecordType(recordType);
        }

        public boolean acceptValue(ValueType valueType) {
            return this.configuration.shouldIndexValueType(valueType);
        }
    }
}

