/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.gcs;

import com.google.cloud.storage.BucketInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageOptions;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupDescriptor;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.api.NamedFileSet;
import io.camunda.zeebe.backup.common.BackupImpl;
import io.camunda.zeebe.backup.common.BackupStatusImpl;
import io.camunda.zeebe.backup.common.Manifest;
import io.camunda.zeebe.backup.gcs.FileSetManager;
import io.camunda.zeebe.backup.gcs.GcsBackupConfig;
import io.camunda.zeebe.backup.gcs.GcsBackupStoreException;
import io.camunda.zeebe.backup.gcs.GcsConnectionConfig;
import io.camunda.zeebe.backup.gcs.ManifestManager;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public final class GcsBackupStore
implements BackupStore {
    public static final String ERROR_MSG_BACKUP_NOT_FOUND = "Expected to restore from backup with id '%s', but does not exist.";
    public static final String ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE = "Expected to restore from completed backup with id '%s', but was in state '%s'";
    public static final String ERROR_VALIDATION_FAILED = "Invalid configuration for GcsBackupStore: %s";
    public static final String SNAPSHOT_FILESET_NAME = "snapshot";
    public static final String SEGMENTS_FILESET_NAME = "segments";
    private static final Logger LOG = LoggerFactory.getLogger(GcsBackupStore.class);
    private final ExecutorService executor;
    private final ManifestManager manifestManager;
    private final FileSetManager fileSetManager;
    private final Storage client;

    GcsBackupStore(GcsBackupConfig config) {
        this(config, GcsBackupStore.buildClient(config));
    }

    GcsBackupStore(GcsBackupConfig config, Storage client) {
        BucketInfo bucketInfo = BucketInfo.of((String)config.bucketName());
        String basePath = Optional.ofNullable(config.basePath()).map(s -> s + "/").orElse("");
        this.client = client;
        this.executor = Executors.newWorkStealingPool(4);
        this.manifestManager = new ManifestManager(client, bucketInfo, basePath);
        this.fileSetManager = new FileSetManager(client, bucketInfo, basePath);
    }

    public static BackupStore of(GcsBackupConfig config) {
        return new GcsBackupStore(config).logging(LOG, Level.INFO);
    }

    public CompletableFuture<Void> save(Backup backup) {
        return CompletableFuture.runAsync(() -> {
            ManifestManager.PersistedManifest persistedManifest = this.manifestManager.createInitialManifest(backup);
            try {
                this.fileSetManager.save(backup.id(), SNAPSHOT_FILESET_NAME, backup.snapshot());
                this.fileSetManager.save(backup.id(), SEGMENTS_FILESET_NAME, backup.segments());
                this.manifestManager.completeManifest(persistedManifest);
            }
            catch (Exception e) {
                this.manifestManager.markAsFailed((Manifest)persistedManifest.manifest(), e.getMessage());
                throw e;
            }
        }, this.executor);
    }

    public CompletableFuture<BackupStatus> getStatus(BackupIdentifier id) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                return BackupStatusImpl.doesNotExist((BackupIdentifier)id);
            }
            return Manifest.toStatus((Manifest)manifest);
        }, this.executor);
    }

    public CompletableFuture<Collection<BackupStatus>> list(BackupIdentifierWildcard wildcard) {
        return CompletableFuture.supplyAsync(() -> this.manifestManager.listManifests(wildcard).stream().map(Manifest::toStatus).toList(), this.executor);
    }

    public CompletableFuture<Void> delete(BackupIdentifier id) {
        return CompletableFuture.runAsync(() -> {
            this.manifestManager.deleteManifest(id);
            this.fileSetManager.delete(id, SNAPSHOT_FILESET_NAME);
            this.fileSetManager.delete(id, SEGMENTS_FILESET_NAME);
        }, this.executor);
    }

    public CompletableFuture<Backup> restore(BackupIdentifier id, Path targetFolder) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                throw new RuntimeException(ERROR_MSG_BACKUP_NOT_FOUND.formatted(id));
            }
            switch (manifest.statusCode()) {
                default: {
                    throw new MatchException(null, null);
                }
                case FAILED: 
                case IN_PROGRESS: {
                    throw new RuntimeException(ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE.formatted(id, manifest.statusCode()));
                }
                case COMPLETED: 
            }
            Manifest.CompletedManifest completed = manifest.asCompleted();
            NamedFileSet snapshot = this.fileSetManager.restore(id, SNAPSHOT_FILESET_NAME, completed.snapshot(), targetFolder);
            NamedFileSet segments = this.fileSetManager.restore(id, SEGMENTS_FILESET_NAME, completed.segments(), targetFolder);
            return new BackupImpl(id, (BackupDescriptor)manifest.descriptor(), snapshot, segments);
        }, this.executor);
    }

    public CompletableFuture<BackupStatusCode> markFailed(BackupIdentifier id, String failureReason) {
        return CompletableFuture.supplyAsync(() -> {
            this.manifestManager.markAsFailed(id, failureReason);
            return BackupStatusCode.FAILED;
        }, this.executor);
    }

    public CompletableFuture<Void> closeAsync() {
        return CompletableFuture.runAsync(() -> {
            try {
                this.executor.shutdown();
                boolean closed = this.executor.awaitTermination(1L, TimeUnit.MINUTES);
                if (!closed) {
                    this.executor.shutdownNow();
                }
                this.client.close();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static Storage buildClient(GcsBackupConfig config) {
        String projectId;
        StorageOptions.Builder builder = (StorageOptions.Builder)((StorageOptions.Builder)StorageOptions.newBuilder().setHost(config.connection().host())).setCredentials(config.connection().auth().credentials());
        GcsConnectionConfig.Authentication authentication = config.connection().auth();
        if (!(authentication instanceof GcsConnectionConfig.Authentication.None)) return (Storage)builder.build().getService();
        GcsConnectionConfig.Authentication.None none = (GcsConnectionConfig.Authentication.None)authentication;
        try {
            String string;
            projectId = string = none.projectId();
        }
        catch (Throwable throwable) {
            throw new MatchException(throwable.toString(), throwable);
        }
        builder.setProjectId(projectId);
        return (Storage)builder.build().getService();
    }

    public static void validateConfig(GcsBackupConfig config) {
        try (Storage storage = GcsBackupStore.buildClient(config);){
            try {
                storage.list(config.bucketName(), new Storage.BlobListOption[]{Storage.BlobListOption.pageSize((long)1L)});
            }
            catch (Exception e) {
                LOG.warn("Unable to verify that the bucket %s exists, initialization will continue as it can be a transient network issue".formatted(config.bucketName()), (Throwable)e);
            }
        }
        catch (Exception e) {
            throw new GcsBackupStoreException.ConfigurationException(ERROR_VALIDATION_FAILED.formatted(config), e);
        }
    }
}

