/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.filesystem;

import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupDescriptor;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.api.NamedFileSet;
import io.camunda.zeebe.backup.common.BackupImpl;
import io.camunda.zeebe.backup.common.BackupStatusImpl;
import io.camunda.zeebe.backup.common.BackupStoreException;
import io.camunda.zeebe.backup.common.Manifest;
import io.camunda.zeebe.backup.filesystem.FileSetManager;
import io.camunda.zeebe.backup.filesystem.FilesystemBackupConfig;
import io.camunda.zeebe.backup.filesystem.ManifestManager;
import java.nio.file.Path;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public final class FilesystemBackupStore
implements BackupStore {
    public static final String ERROR_MSG_BACKUP_NOT_FOUND = "Expected to restore from backup with id '%s', but does not exist.";
    public static final String ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE = "Expected to restore from completed backup with id '%s', but was in state '%s'";
    public static final String SNAPSHOT_FILESET_NAME = "snapshot";
    public static final String SEGMENTS_FILESET_NAME = "segments";
    private static final Logger LOG = LoggerFactory.getLogger(FilesystemBackupStore.class);
    private final ExecutorService executor;
    private final FileSetManager fileSetManager;
    private final ManifestManager manifestManager;

    FilesystemBackupStore(FilesystemBackupConfig config) {
        this(config, Executors.newVirtualThreadPerTaskExecutor());
    }

    FilesystemBackupStore(FilesystemBackupConfig config, ExecutorService executor) {
        FilesystemBackupStore.validateConfig(config);
        this.executor = executor;
        this.fileSetManager = new FileSetManager(config.basePath());
        this.manifestManager = new ManifestManager(config.basePath());
    }

    public CompletableFuture<Void> save(Backup backup) {
        LOG.debug("Saving {}", (Object)backup.id());
        return CompletableFuture.runAsync(() -> {
            Manifest.InProgressManifest manifest = this.manifestManager.createInitialManifest(backup);
            try {
                this.fileSetManager.save(backup.id(), SNAPSHOT_FILESET_NAME, backup.snapshot());
                this.fileSetManager.save(backup.id(), SEGMENTS_FILESET_NAME, backup.segments());
                this.manifestManager.completeManifest(manifest);
            }
            catch (Exception e) {
                this.manifestManager.markAsFailed((BackupIdentifier)manifest.id(), e.getMessage());
                throw e;
            }
        }, this.executor);
    }

    public CompletableFuture<BackupStatus> getStatus(BackupIdentifier id) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                return BackupStatusImpl.doesNotExist((BackupIdentifier)id);
            }
            return Manifest.toStatus((Manifest)manifest);
        }, this.executor);
    }

    public CompletableFuture<Collection<BackupStatus>> list(BackupIdentifierWildcard wildcard) {
        return CompletableFuture.supplyAsync(() -> this.manifestManager.listManifests(wildcard).stream().map(Manifest::toStatus).toList(), this.executor);
    }

    public CompletableFuture<Void> delete(BackupIdentifier id) {
        return CompletableFuture.runAsync(() -> {
            this.manifestManager.deleteManifest(id);
            this.fileSetManager.delete(id, SNAPSHOT_FILESET_NAME);
            this.fileSetManager.delete(id, SEGMENTS_FILESET_NAME);
        }, this.executor);
    }

    public CompletableFuture<Backup> restore(BackupIdentifier id, Path targetFolder) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                throw new BackupStoreException.UnexpectedManifestState(ERROR_MSG_BACKUP_NOT_FOUND.formatted(id));
            }
            switch (manifest.statusCode()) {
                default: {
                    throw new MatchException(null, null);
                }
                case FAILED: 
                case IN_PROGRESS: {
                    throw new BackupStoreException.UnexpectedManifestState(ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE.formatted(id, manifest.statusCode()));
                }
                case COMPLETED: 
            }
            Manifest.CompletedManifest completed = manifest.asCompleted();
            NamedFileSet snapshot = this.fileSetManager.restore(id, SNAPSHOT_FILESET_NAME, completed.snapshot(), targetFolder);
            NamedFileSet segments = this.fileSetManager.restore(id, SEGMENTS_FILESET_NAME, completed.segments(), targetFolder);
            return new BackupImpl(id, (BackupDescriptor)manifest.descriptor(), snapshot, segments);
        }, this.executor);
    }

    public CompletableFuture<BackupStatusCode> markFailed(BackupIdentifier id, String failureReason) {
        return CompletableFuture.supplyAsync(() -> {
            this.manifestManager.markAsFailed(id, failureReason);
            return BackupStatusCode.FAILED;
        }, this.executor);
    }

    public CompletableFuture<Void> closeAsync() {
        return CompletableFuture.runAsync(() -> {
            try {
                this.executor.shutdown();
                boolean closed = this.executor.awaitTermination(1L, TimeUnit.MINUTES);
                if (!closed) {
                    LOG.debug("Expected file system backup store executor to shutdown within a minute, but one task is hanging; will forcefully shutdown, but some backup may not be written properly");
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOG.debug("Interrupted while awaiting shutdown of file system store executor, possible resource leak", (Throwable)e);
            }
        });
    }

    public static void validateConfig(FilesystemBackupConfig config) {
        if (config.basePath() == null || config.basePath().isBlank()) {
            throw new IllegalArgumentException("Expected a basePath to be provided, but got [%s]".formatted(config.basePath()));
        }
    }

    public static BackupStore of(FilesystemBackupConfig storeConfig) {
        return new FilesystemBackupStore(storeConfig).logging(LOG, Level.INFO);
    }
}

