/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.filesystem;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.common.BackupStoreException;
import io.camunda.zeebe.backup.common.Manifest;
import io.camunda.zeebe.util.FileUtil;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collection;
import java.util.List;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ManifestManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ManifestManager.class);
    private static final ObjectMapper MAPPER = new ObjectMapper().registerModule((Module)new Jdk8Module()).registerModule((Module)new JavaTimeModule()).disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS).setSerializationInclusion(JsonInclude.Include.NON_ABSENT);
    private static final String MANIFEST_FILENAME = "manifest.json";
    private final Path manifestsPath;

    ManifestManager(Path manifestsPath) {
        this.manifestsPath = manifestsPath;
    }

    Manifest.InProgressManifest createInitialManifest(Backup backup) {
        Manifest.InProgressManifest manifest = Manifest.createInProgress((Backup)backup);
        Path path = this.manifestPath((Manifest)manifest);
        try {
            FileUtil.ensureDirectoryExists((Path)path.getParent());
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to create directories for manifest: " + String.valueOf(path.getParent()), e);
        }
        try {
            byte[] serializedManifest = MAPPER.writeValueAsBytes((Object)manifest);
            Files.write(path, serializedManifest, StandardOpenOption.CREATE_NEW, StandardOpenOption.SYNC);
            FileUtil.flushDirectory((Path)path.getParent());
            return manifest;
        }
        catch (FileAlreadyExistsException e) {
            throw new BackupStoreException.UnexpectedManifestState("Manifest already exists.");
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to write manifest to " + String.valueOf(path), e);
        }
    }

    void completeManifest(Manifest.InProgressManifest inProgressManifest) {
        Manifest.CompletedManifest completed = inProgressManifest.complete();
        try {
            byte[] serializedManifest = MAPPER.writeValueAsBytes((Object)completed);
            Manifest existingManifest = this.getManifest((BackupIdentifier)inProgressManifest.id());
            if (existingManifest == null) {
                throw new BackupStoreException.UnexpectedManifestState("Manifest does not exist.");
            }
            if (existingManifest.statusCode() != Manifest.StatusCode.IN_PROGRESS) {
                throw new BackupStoreException.UnexpectedManifestState("Expected manifest to be in progress but was in %s".formatted(existingManifest.statusCode().name()));
            }
            Path path = this.manifestPath((Manifest)inProgressManifest);
            Files.write(path, serializedManifest, StandardOpenOption.CREATE, StandardOpenOption.SYNC);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to write updated manifest", e);
        }
    }

    void markAsFailed(BackupIdentifier manifestId, String failureReason) {
        Manifest.FailedManifest updatedManifest;
        Manifest manifest = this.getManifest(manifestId);
        if (manifest == null) {
            manifest = Manifest.createFailed((BackupIdentifier)manifestId);
        }
        switch (manifest.statusCode()) {
            default: {
                throw new MatchException(null, null);
            }
            case FAILED: {
                Manifest.FailedManifest failedManifest = manifest.asFailed();
                break;
            }
            case COMPLETED: {
                Manifest.FailedManifest failedManifest = manifest.asCompleted().fail(failureReason);
                break;
            }
            case IN_PROGRESS: {
                Manifest.FailedManifest failedManifest = updatedManifest = manifest.asInProgress().fail(failureReason);
            }
        }
        if (manifest != updatedManifest) {
            try {
                byte[] serializedManifest = MAPPER.writeValueAsBytes((Object)updatedManifest);
                Path path = this.manifestPath(manifest);
                Files.write(path, serializedManifest, StandardOpenOption.SYNC);
            }
            catch (IOException e) {
                throw new UncheckedIOException("Unable to write updated manifest", e);
            }
        }
    }

    void deleteManifest(BackupIdentifier id) {
        Manifest manifest = this.getManifest(id);
        if (manifest == null) {
            return;
        }
        if (manifest.statusCode() == Manifest.StatusCode.IN_PROGRESS) {
            throw new BackupStoreException.UnexpectedManifestState("Cannot delete Backup with id '%s' while saving is in progress.".formatted(id.toString()));
        }
        try {
            Path path = this.manifestPath(manifest);
            Files.delete(path);
            FileUtil.flushDirectory((Path)path.getParent());
        }
        catch (NoSuchFileException e) {
            LOGGER.warn("Try to remove unknown manifest with id {}", (Object)id);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to delete manifest", e);
        }
    }

    Manifest getManifest(BackupIdentifier id) {
        return this.getManifestWithPath(this.getManifestPath(id));
    }

    Collection<Manifest> listManifests(BackupIdentifierWildcard wildcard) {
        List<Manifest> list;
        block8: {
            Stream<Path> files = Files.walk(this.manifestsPath, new FileVisitOption[0]);
            try {
                list = files.filter(filePath -> this.filterBlobsByWildcard(wildcard, filePath.toString())).map(this::getManifestWithPath).toList();
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException("Unable to list manifests from " + String.valueOf(this.manifestsPath), e);
                }
            }
            files.close();
        }
        return list;
    }

    private Manifest getManifestWithPath(Path path) {
        if (!Files.exists(path, new LinkOption[0])) {
            return null;
        }
        try {
            return (Manifest)MAPPER.readValue(path.toFile(), Manifest.class);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Unable to read manifest from path " + String.valueOf(path), e);
        }
    }

    Path manifestPath(Manifest manifest) {
        return this.getManifestPath((BackupIdentifier)manifest.id());
    }

    private Path getManifestPath(BackupIdentifier id) {
        return this.getManifestPath(String.valueOf(id.partitionId()), String.valueOf(id.checkpointId()), String.valueOf(id.nodeId()));
    }

    private boolean filterBlobsByWildcard(BackupIdentifierWildcard wildcard, String path) {
        Predicate<String> pattern = Pattern.compile(this.getManifestPath(wildcard.partitionId().map(Object::toString).orElse("\\d+"), wildcard.checkpointPattern().asRegex(), wildcard.nodeId().map(Object::toString).orElse("\\d+")).toString()).asMatchPredicate();
        return pattern.test(path);
    }

    private Path getManifestPath(String partitionId, String checkpointId, String nodeId) {
        return this.manifestsPath.resolve(partitionId).resolve(checkpointId).resolve(nodeId).resolve(MANIFEST_FILENAME);
    }
}

