/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.azure;

import com.azure.core.credential.TokenCredential;
import com.azure.identity.DefaultAzureCredentialBuilder;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.BlobServiceClientBuilder;
import com.azure.storage.common.StorageSharedKeyCredential;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupDescriptor;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.api.BackupStatus;
import io.camunda.zeebe.backup.api.BackupStatusCode;
import io.camunda.zeebe.backup.api.BackupStore;
import io.camunda.zeebe.backup.api.NamedFileSet;
import io.camunda.zeebe.backup.azure.AzureBackupConfig;
import io.camunda.zeebe.backup.azure.AzureBackupStoreException;
import io.camunda.zeebe.backup.azure.FileSetManager;
import io.camunda.zeebe.backup.azure.ManifestManager;
import io.camunda.zeebe.backup.common.BackupImpl;
import io.camunda.zeebe.backup.common.BackupStatusImpl;
import io.camunda.zeebe.backup.common.BackupStoreException;
import io.camunda.zeebe.backup.common.Manifest;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AzureBackupStore
implements BackupStore {
    public static final String ERROR_MSG_BACKUP_NOT_FOUND = "Expected to restore from backup with id '%s', but does not exist.";
    public static final String ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE = "Expected to restore from completed backup with id '%s', but was in state '%s'";
    public static final String SNAPSHOT_FILESET_NAME = "snapshot";
    public static final String SEGMENTS_FILESET_NAME = "segments";
    private static final Logger LOG = LoggerFactory.getLogger(AzureBackupStore.class);
    private final ExecutorService executor = Executors.newVirtualThreadPerTaskExecutor();
    private final FileSetManager fileSetManager;
    private final ManifestManager manifestManager;

    public AzureBackupStore(AzureBackupConfig config) {
        this(config, AzureBackupStore.buildClient(config));
    }

    public AzureBackupStore(AzureBackupConfig config, BlobServiceClient client) {
        BlobContainerClient blobContainerClient = this.getContainerClient(client, config);
        this.fileSetManager = new FileSetManager(blobContainerClient, this.isCreateContainer(config));
        this.manifestManager = new ManifestManager(blobContainerClient, this.isCreateContainer(config));
    }

    public static BlobServiceClient buildClient(AzureBackupConfig config) {
        if (config.sasTokenConfig() != null) {
            return new BlobServiceClientBuilder().sasToken(config.sasTokenConfig().value()).endpoint(config.endpoint()).buildClient();
        }
        if (config.connectionString() != null) {
            return new BlobServiceClientBuilder().connectionString(config.connectionString()).buildClient();
        }
        if (config.accountName() != null || config.accountKey() != null) {
            StorageSharedKeyCredential credential = new StorageSharedKeyCredential(Objects.requireNonNull(config.accountName(), "Account key is specified but no account name was provided."), Objects.requireNonNull(config.accountKey(), "Account name is specified but no account key was provided."));
            return new BlobServiceClientBuilder().endpoint(config.endpoint()).credential(credential).buildClient();
        }
        LOG.info("No connection string, sas token or account credentials are configured, using DefaultAzureCredentialBuilder for authentication.");
        return new BlobServiceClientBuilder().endpoint(config.endpoint()).credential((TokenCredential)new DefaultAzureCredentialBuilder().build()).buildClient();
    }

    BlobContainerClient getContainerClient(BlobServiceClient client, AzureBackupConfig config) {
        BlobContainerClient blobContainerClient = client.getBlobContainerClient(config.containerName());
        if (!config.createContainer()) {
            LOG.debug("Setting up Azure Store with existing container: {}", (Object)blobContainerClient.getBlobContainerName());
            if ((config.sasTokenConfig() == null || config.sasTokenConfig().type().isAccount()) && !blobContainerClient.exists()) {
                throw new AzureBackupStoreException.ContainerDoesNotExist("The container %s does not exist. Please create it before using the backup store. Otherwise set createContainer to true, to enable the creation of the container during Azure backup store initialization.".formatted(blobContainerClient.getBlobContainerName()));
            }
        }
        return blobContainerClient;
    }

    public CompletableFuture<Void> save(Backup backup) {
        return CompletableFuture.runAsync(() -> {
            ManifestManager.PersistedManifest persistedManifest = this.manifestManager.createInitialManifest(backup);
            try {
                this.fileSetManager.save(backup.id(), SNAPSHOT_FILESET_NAME, backup.snapshot());
                this.fileSetManager.save(backup.id(), SEGMENTS_FILESET_NAME, backup.segments());
                this.manifestManager.completeManifest(persistedManifest);
            }
            catch (Exception e) {
                this.manifestManager.markAsFailed((BackupIdentifier)persistedManifest.manifest().id(), e.getMessage());
                throw e;
            }
        }, this.executor);
    }

    public CompletableFuture<BackupStatus> getStatus(BackupIdentifier id) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                return BackupStatusImpl.doesNotExist((BackupIdentifier)id);
            }
            return Manifest.toStatus((Manifest)manifest);
        }, this.executor);
    }

    public CompletableFuture<Collection<BackupStatus>> list(BackupIdentifierWildcard wildcard) {
        return CompletableFuture.supplyAsync(() -> this.manifestManager.listManifests(wildcard).stream().map(Manifest::toStatus).toList(), this.executor);
    }

    public CompletableFuture<Void> delete(BackupIdentifier id) {
        return CompletableFuture.runAsync(() -> {
            this.manifestManager.deleteManifest(id);
            this.fileSetManager.delete(id, SNAPSHOT_FILESET_NAME);
            this.fileSetManager.delete(id, SEGMENTS_FILESET_NAME);
        }, this.executor);
    }

    public CompletableFuture<Backup> restore(BackupIdentifier id, Path targetFolder) {
        return CompletableFuture.supplyAsync(() -> {
            Manifest manifest = this.manifestManager.getManifest(id);
            if (manifest == null) {
                throw new BackupStoreException.UnexpectedManifestState(ERROR_MSG_BACKUP_NOT_FOUND.formatted(id));
            }
            switch (manifest.statusCode()) {
                default: {
                    throw new MatchException(null, null);
                }
                case FAILED: 
                case IN_PROGRESS: {
                    throw new BackupStoreException.UnexpectedManifestState(ERROR_MSG_BACKUP_WRONG_STATE_TO_RESTORE.formatted(id, manifest.statusCode()));
                }
                case COMPLETED: 
            }
            Manifest.CompletedManifest completed = manifest.asCompleted();
            NamedFileSet snapshot = this.fileSetManager.restore(id, SNAPSHOT_FILESET_NAME, completed.snapshot(), targetFolder);
            NamedFileSet segments = this.fileSetManager.restore(id, SEGMENTS_FILESET_NAME, completed.segments(), targetFolder);
            return new BackupImpl(id, (BackupDescriptor)manifest.descriptor(), snapshot, segments);
        }, this.executor);
    }

    public CompletableFuture<BackupStatusCode> markFailed(BackupIdentifier id, String failureReason) {
        return CompletableFuture.supplyAsync(() -> {
            this.manifestManager.markAsFailed(id, failureReason);
            return BackupStatusCode.FAILED;
        }, this.executor);
    }

    public CompletableFuture<Void> closeAsync() {
        return CompletableFuture.runAsync(() -> {
            try {
                this.executor.shutdown();
                boolean closed = this.executor.awaitTermination(1L, TimeUnit.MINUTES);
                if (!closed) {
                    LOG.warn("Failed to orderly shutdown Azure Store Executor within one minute.");
                    this.executor.shutdownNow();
                }
            }
            catch (Exception e) {
                LOG.error("Failed to shutdown of Azure Store Executor.");
                throw new RuntimeException(e);
            }
        });
    }

    public static void validateConfig(AzureBackupConfig config) {
        if (config.sasTokenConfig() != null && !config.sasTokenConfig().type().isAccount()) {
            LOG.info("User delegation or service SAS tokens are enabled, which do not have permissions to access/create containers. The creation and checks of the container existence will be skipped.");
        }
        if (AzureBackupStore.moreThanOneNonNull(config.accountKey(), config.connectionString(), config.sasTokenConfig())) {
            LOG.warn("More than one authentication method is configured, if present account SAS token will be used, followed by connection string, and then account name with account key.");
        }
        if (config.connectionString() == null && config.endpoint() == null) {
            throw new IllegalArgumentException("Connection string or endpoint is required");
        }
        if (config.accountKey() != null && config.accountName() == null) {
            throw new IllegalArgumentException("Account key is specified but account name is missing");
        }
        if (config.accountName() != null && config.accountKey() == null) {
            throw new IllegalArgumentException("Account name is specified but account key is missing");
        }
        if (config.containerName() == null) {
            throw new IllegalArgumentException("Container name cannot be null.");
        }
    }

    private boolean isCreateContainer(AzureBackupConfig config) {
        if (config.sasTokenConfig() != null && !config.sasTokenConfig().type().isAccount()) {
            return false;
        }
        return config.createContainer();
    }

    private static boolean moreThanOneNonNull(Object ... values) {
        int count = 0;
        for (Object value : values) {
            if (value != null) {
                ++count;
            }
            if (count <= 1) continue;
            return true;
        }
        return false;
    }
}

