/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.backup.azure;

import com.azure.core.http.rest.Response;
import com.azure.core.util.BinaryData;
import com.azure.core.util.Context;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.BlobRequestConditions;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.models.BlockBlobItem;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.blob.options.BlobParallelUploadOptions;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.camunda.zeebe.backup.api.Backup;
import io.camunda.zeebe.backup.api.BackupIdentifier;
import io.camunda.zeebe.backup.api.BackupIdentifierWildcard;
import io.camunda.zeebe.backup.common.BackupStoreException;
import io.camunda.zeebe.backup.common.Manifest;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Collection;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class ManifestManager {
    public static final int PRECONDITION_FAILED = 412;
    private static final String MANIFEST_PATH_FORMAT = "manifests/%s/%s/%s/manifest.json";
    private static final ObjectMapper MAPPER = new ObjectMapper().registerModule((Module)new Jdk8Module()).registerModule((Module)new JavaTimeModule()).disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS).setSerializationInclusion(JsonInclude.Include.NON_ABSENT);
    private boolean containerCreated = false;
    private final BlobContainerClient blobContainerClient;

    ManifestManager(BlobContainerClient blobContainerClient) {
        this.blobContainerClient = blobContainerClient;
    }

    PersistedManifest createInitialManifest(Backup backup) {
        byte[] serializedManifest;
        Manifest.InProgressManifest manifest = Manifest.createInProgress((Backup)backup);
        this.assureContainerCreated();
        try {
            serializedManifest = MAPPER.writeValueAsBytes((Object)manifest);
        }
        catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
        BlobClient blobClient = this.blobContainerClient.getBlobClient(ManifestManager.manifestPath((Manifest)manifest));
        try {
            BlobRequestConditions blobRequestConditions = new BlobRequestConditions();
            ManifestManager.disableOverwrite(blobRequestConditions);
            Response blockBlobItemResponse = blobClient.uploadWithResponse(new BlobParallelUploadOptions(BinaryData.fromBytes((byte[])serializedManifest)).setRequestConditions(blobRequestConditions), null, Context.NONE);
            return new PersistedManifest(((BlockBlobItem)blockBlobItemResponse.getValue()).getETag(), manifest);
        }
        catch (BlobStorageException e) {
            if (e.getErrorCode() == BlobErrorCode.BLOB_ALREADY_EXISTS) {
                throw new BackupStoreException.UnexpectedManifestState("Manifest already exists.", (Exception)((Object)e));
            }
            throw e;
        }
    }

    void completeManifest(PersistedManifest inProgressManifest) {
        byte[] serializedManifest;
        Manifest.CompletedManifest completed = inProgressManifest.manifest().complete();
        this.assureContainerCreated();
        try {
            serializedManifest = MAPPER.writeValueAsBytes((Object)completed);
        }
        catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
        BlobClient blobClient = this.blobContainerClient.getBlobClient(ManifestManager.manifestPath((Manifest)completed));
        try {
            Manifest manifest = this.getManifest((BackupIdentifier)inProgressManifest.manifest().id());
            if (manifest == null) {
                throw new BackupStoreException.UnexpectedManifestState("Manifest does not exist.");
            }
            if (manifest.statusCode() != Manifest.StatusCode.IN_PROGRESS) {
                throw new BackupStoreException.UnexpectedManifestState("Expected manifest to be in progress but was in %s".formatted(manifest.statusCode().name()));
            }
            BlobRequestConditions blobRequestConditions = new BlobRequestConditions().setIfMatch(inProgressManifest.eTag());
            blobClient.uploadWithResponse(new BlobParallelUploadOptions(BinaryData.fromBytes((byte[])serializedManifest)).setRequestConditions(blobRequestConditions), null, Context.NONE);
        }
        catch (BlobStorageException e) {
            if (e.getStatusCode() == 412) {
                throw new BackupStoreException.UnexpectedManifestState(e.getMessage());
            }
            throw new RuntimeException(e);
        }
    }

    void markAsFailed(BackupIdentifier manifestId, String failureReason) {
        Manifest.FailedManifest updatedManifest;
        this.assureContainerCreated();
        BlobClient blobClient = this.blobContainerClient.getBlobClient(ManifestManager.manifestIdPath(manifestId));
        Manifest manifest = this.getManifest(manifestId);
        if (manifest == null) {
            manifest = Manifest.createFailed((BackupIdentifier)manifestId);
        }
        switch (manifest.statusCode()) {
            default: {
                throw new MatchException(null, null);
            }
            case FAILED: {
                Manifest.FailedManifest failedManifest = manifest.asFailed();
                break;
            }
            case COMPLETED: {
                Manifest.FailedManifest failedManifest = manifest.asCompleted().fail(failureReason);
                break;
            }
            case IN_PROGRESS: {
                Manifest.FailedManifest failedManifest = updatedManifest = manifest.asInProgress().fail(failureReason);
            }
        }
        if (manifest != updatedManifest) {
            try {
                blobClient.upload(BinaryData.fromBytes((byte[])MAPPER.writeValueAsBytes((Object)updatedManifest)), true);
            }
            catch (JsonProcessingException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void deleteManifest(BackupIdentifier id) {
        BlobClient blobClient = this.blobContainerClient.getBlobClient(ManifestManager.manifestIdPath(id));
        Manifest manifest = this.getManifest(id);
        if (manifest == null) {
            return;
        }
        if (manifest.statusCode() == Manifest.StatusCode.IN_PROGRESS) {
            throw new BackupStoreException.UnexpectedManifestState("Cannot delete Backup with id '%s' while saving is in progress.".formatted(id.toString()));
        }
        blobClient.delete();
    }

    Manifest getManifest(BackupIdentifier id) {
        return this.getManifestWithPath(MANIFEST_PATH_FORMAT.formatted(id.partitionId(), id.checkpointId(), id.nodeId()));
    }

    private Manifest getManifestWithPath(String path) {
        BinaryData binaryData;
        BlobClient blobClient = this.blobContainerClient.getBlobClient(path);
        try {
            binaryData = blobClient.downloadContent();
        }
        catch (BlobStorageException e) {
            if (e.getErrorCode() == BlobErrorCode.CONTAINER_NOT_FOUND || e.getErrorCode() == BlobErrorCode.BLOB_NOT_FOUND) {
                return null;
            }
            throw new RuntimeException(e);
        }
        try {
            return (Manifest)MAPPER.readValue(binaryData.toStream(), Manifest.class);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public Collection<Manifest> listManifests(BackupIdentifierWildcard wildcard) {
        this.assureContainerCreated();
        return this.blobContainerClient.listBlobs(new ListBlobsOptions().setPrefix(this.wildcardPrefix(wildcard)), null).stream().map(BlobItem::getName).filter(path -> this.filterBlobsByWildcard(wildcard, (String)path)).map(this::getManifestWithPath).toList();
    }

    public static String manifestPath(Manifest manifest) {
        return ManifestManager.manifestIdPath((BackupIdentifier)manifest.id());
    }

    private static String manifestIdPath(BackupIdentifier backupIdentifier) {
        return MANIFEST_PATH_FORMAT.formatted(backupIdentifier.partitionId(), backupIdentifier.checkpointId(), backupIdentifier.nodeId());
    }

    private boolean filterBlobsByWildcard(BackupIdentifierWildcard wildcard, String path) {
        Predicate<String> pattern = Pattern.compile(MANIFEST_PATH_FORMAT.formatted(wildcard.partitionId().map(Object::toString).orElse("\\d+"), wildcard.checkpointId().map(Object::toString).orElse("\\d+"), wildcard.nodeId().map(Object::toString).orElse("\\d+"))).asMatchPredicate();
        return pattern.test(path);
    }

    private String wildcardPrefix(BackupIdentifierWildcard wildcard) {
        return Stream.of(wildcard.partitionId(), wildcard.checkpointId(), wildcard.nodeId()).takeWhile(Optional::isPresent).map(Optional::get).map(rec$ -> rec$.toString()).collect(Collectors.joining("/", "manifests/", ""));
    }

    void assureContainerCreated() {
        if (!this.containerCreated) {
            this.blobContainerClient.createIfNotExists();
            this.containerCreated = true;
        }
    }

    public static void disableOverwrite(BlobRequestConditions blobRequestConditions) {
        blobRequestConditions.setIfNoneMatch("*");
    }

    record PersistedManifest(String eTag, Manifest.InProgressManifest manifest) {
    }
}

