/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.raft.impl;

import io.atomix.cluster.MemberId;
import io.atomix.cluster.messaging.MessagingException;
import io.atomix.primitive.PrimitiveException;
import io.atomix.raft.RaftError;
import io.atomix.raft.RaftException;
import io.atomix.raft.RaftServer;
import io.atomix.raft.cluster.RaftMember;
import io.atomix.raft.cluster.impl.DefaultRaftMember;
import io.atomix.raft.impl.RaftContext;
import io.atomix.raft.protocol.ForceConfigureRequest;
import io.atomix.raft.protocol.JoinRequest;
import io.atomix.raft.protocol.LeaveRequest;
import io.atomix.raft.protocol.RaftResponse;
import io.atomix.raft.protocol.TransferRequest;
import io.atomix.raft.storage.system.Configuration;
import io.atomix.raft.utils.ForceConfigureQuorum;
import io.atomix.utils.concurrent.ThreadContext;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import java.net.ConnectException;
import java.time.Instant;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;

public final class ReconfigurationHelper {
    private final ThreadContext threadContext;
    private final RaftContext raftContext;
    private final Logger logger;

    public ReconfigurationHelper(RaftContext raftContext) {
        this.threadContext = raftContext.getThreadContext();
        this.raftContext = raftContext;
        this.logger = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(RaftServer.class).addValue((Object)raftContext.getName()).build());
    }

    public CompletableFuture<Void> join(Collection<MemberId> clusterMembers) {
        CompletableFuture<Void> result = new CompletableFuture<Void>();
        this.threadContext.execute(() -> {
            DefaultRaftMember joining = new DefaultRaftMember(this.raftContext.getCluster().getLocalMember().memberId(), RaftMember.Type.ACTIVE, Instant.now());
            LinkedBlockingQueue assistingMembers = clusterMembers.stream().filter(memberId -> !memberId.equals(joining.memberId())).collect(Collectors.toCollection(LinkedBlockingQueue::new));
            if (assistingMembers.isEmpty()) {
                result.completeExceptionally(new IllegalStateException("Cannot join cluster, because there are no other members in the cluster."));
                return;
            }
            this.threadContext.execute(() -> this.joinWithRetry(joining, assistingMembers, result));
        });
        return result;
    }

    private void joinWithRetry(RaftMember joining, Queue<MemberId> assistingMembers, CompletableFuture<Void> result) {
        MemberId receiver = assistingMembers.poll();
        if (receiver == null) {
            result.completeExceptionally(new IllegalStateException("Sent join request to all known members, but all failed. No more members left."));
            return;
        }
        this.raftContext.getProtocol().join(receiver, JoinRequest.builder().withJoiningMember(joining).build()).whenCompleteAsync((response, error) -> {
            if (error != null) {
                Throwable cause = error.getCause();
                if (cause instanceof MessagingException.NoSuchMemberException || cause instanceof MessagingException.NoRemoteHandler || cause instanceof TimeoutException || cause instanceof ConnectException) {
                    this.logger.debug("Join request was not acknowledged, retrying", cause);
                    this.threadContext.execute(() -> this.joinWithRetry(joining, assistingMembers, result));
                } else {
                    this.logger.error("Join request failed with an unexpected error, not retrying", error);
                    result.completeExceptionally((Throwable)error);
                }
            } else if (response.status() == RaftResponse.Status.OK) {
                this.logger.debug("Join request accepted");
                result.complete(null);
            } else if (response.error().type() == RaftError.Type.NO_LEADER || response.error().type() == RaftError.Type.UNAVAILABLE) {
                this.logger.debug("Join request failed, retrying", (Throwable)((Object)response.error().createException()));
                this.threadContext.execute(() -> this.joinWithRetry(joining, assistingMembers, result));
            } else {
                PrimitiveException errorAsException = response.error().createException();
                this.logger.error("Join request rejected, not retrying", (Throwable)((Object)errorAsException));
                result.completeExceptionally((Throwable)((Object)errorAsException));
            }
        }, (Executor)this.threadContext);
    }

    public CompletableFuture<Void> leave() {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.threadContext.execute(() -> this.leaveInternal(future));
        return future;
    }

    private void leaveInternal(CompletableFuture<Void> future) {
        RaftMember leaving = this.raftContext.getCluster().getLocalMember();
        MemberId receiver = Optional.ofNullable(this.raftContext.getLeader()).map(DefaultRaftMember::memberId).or(() -> this.raftContext.getCluster().getVotingMembers().stream().map(RaftMember::memberId).findAny()).orElseThrow();
        this.raftContext.getProtocol().leave(receiver, LeaveRequest.builder().withLeavingMember(leaving).build()).whenCompleteAsync((response, error) -> {
            if (error != null) {
                future.completeExceptionally((Throwable)error);
            } else if (response.status() == RaftResponse.Status.OK) {
                future.complete(null);
                this.raftContext.updateState(RaftContext.State.LEFT);
            } else {
                future.completeExceptionally((Throwable)((Object)response.error().createException()));
            }
        }, (Executor)this.threadContext);
    }

    public CompletableFuture<Void> forceConfigure(Map<MemberId, RaftMember.Type> newMembersIds) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.threadContext.execute(() -> this.triggerForceConfigure(newMembersIds, future));
        return future;
    }

    private void triggerForceConfigure(Map<MemberId, RaftMember.Type> newMembersIds, CompletableFuture<Void> future) {
        Configuration currentConfiguration = this.raftContext.getCluster().getConfiguration();
        Set<RaftMember> newMembers = newMembersIds.entrySet().stream().map(memberEntry -> new DefaultRaftMember((MemberId)memberEntry.getKey(), (RaftMember.Type)((Object)((Object)memberEntry.getValue())), Instant.now())).collect(Collectors.toSet());
        if (currentConfiguration == null || !currentConfiguration.force()) {
            if (this.raftContext.getRaftRole().role() == RaftServer.Role.LEADER) {
                this.raftContext.transition(RaftServer.Role.FOLLOWER);
            }
            this.logger.info("Current configuration is '{}'. Forcing configuration with members '{}'", (Object)currentConfiguration, newMembers);
            Configuration newConfiguration = new Configuration(this.raftContext.getCurrentConfigurationIndex() + 1L, this.raftContext.getTerm(), Instant.now().toEpochMilli(), newMembers, Set.of(), true);
            this.raftContext.getCluster().configure(newConfiguration);
        } else if (!currentConfiguration.allMembers().equals(newMembers)) {
            future.completeExceptionally(new IllegalStateException(String.format("Expected to force configure with members '%s', but the member is already in force configuration with a different set of members '%s'", newMembers, currentConfiguration.allMembers())));
            return;
        }
        this.sendForceConfigureRequestToAllMembers(future);
    }

    private void sendForceConfigureRequestToAllMembers(CompletableFuture<Void> future) {
        Configuration configuration = this.raftContext.getCluster().getConfiguration();
        Set<MemberId> otherMembers = configuration.newMembers().stream().map(RaftMember::memberId).filter(m -> !m.equals(this.raftContext.getCluster().getLocalMember().memberId())).collect(Collectors.toSet());
        if (otherMembers.isEmpty()) {
            future.complete(null);
            return;
        }
        ForceConfigureQuorum quorum = new ForceConfigureQuorum(success -> {
            if (Boolean.TRUE.equals(success)) {
                future.complete(null);
            } else {
                future.completeExceptionally(new RaftException.ProtocolException("Failed to force configure because not all members acknowledged the request.", new Object[0]));
            }
        }, otherMembers);
        ForceConfigureRequest request = ForceConfigureRequest.builder().withTerm(configuration.term()).withIndex(configuration.index()).withTime(configuration.time()).withNewMembers(new HashSet<RaftMember>(configuration.newMembers())).from(this.raftContext.getCluster().getLocalMember().memberId()).build();
        otherMembers.forEach(memberId -> this.sendForceConfigurationRequest((MemberId)memberId, request, quorum));
    }

    private void sendForceConfigurationRequest(MemberId memberId, ForceConfigureRequest request, ForceConfigureQuorum quorum) {
        this.logger.trace("Sending '{}' request to member '{}'", (Object)request, (Object)memberId);
        this.raftContext.getProtocol().forceConfigure(memberId, request).whenCompleteAsync((response, error) -> {
            if (error != null) {
                this.logger.warn("Failed to send force configure request to member '{}'", (Object)memberId, error);
                quorum.fail(memberId);
            } else if (response.status() == RaftResponse.Status.OK) {
                this.logger.debug("Successfully sent force configure request to member '{}'", (Object)memberId);
                quorum.succeed(memberId);
            } else {
                this.logger.warn("Failed to send force configure request to member '{}': {}", (Object)memberId, (Object)response.error());
                quorum.fail(memberId);
            }
        }, (Executor)this.threadContext);
    }

    public CompletableFuture<Void> anoint() {
        if (this.raftContext.getRaftRole().role() == RaftServer.Role.LEADER) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.threadContext.execute(() -> this.anointInternal(future));
        return future;
    }

    private void anointInternal(final CompletableFuture<Void> future) {
        Consumer<RaftMember> electionListener = new Consumer<RaftMember>(){

            @Override
            public void accept(RaftMember member) {
                if (member.memberId().equals(ReconfigurationHelper.this.raftContext.getCluster().getLocalMember().memberId())) {
                    future.complete(null);
                } else {
                    future.completeExceptionally(new RaftException.ProtocolException("Failed to transfer leadership", new Object[0]));
                }
                ReconfigurationHelper.this.raftContext.removeLeaderElectionListener(this);
            }
        };
        this.raftContext.addLeaderElectionListener(electionListener);
        RaftMember member = this.raftContext.getCluster().getLocalMember();
        DefaultRaftMember leader = this.raftContext.getLeader();
        if (leader != null) {
            this.raftContext.getProtocol().transfer(leader.memberId(), TransferRequest.builder().withMember(member.memberId()).build()).whenCompleteAsync((response, error) -> {
                if (error != null) {
                    future.completeExceptionally((Throwable)error);
                } else if (response.status() == RaftResponse.Status.ERROR) {
                    future.completeExceptionally((Throwable)((Object)response.error().createException()));
                } else {
                    this.raftContext.transition(RaftServer.Role.CANDIDATE);
                }
            }, (Executor)this.threadContext);
        } else {
            this.raftContext.transition(RaftServer.Role.CANDIDATE);
        }
    }
}

