/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.webapps.backup;

import io.camunda.webapps.backup.BackupException;
import io.camunda.webapps.backup.BackupRepository;
import io.camunda.webapps.backup.BackupService;
import io.camunda.webapps.backup.GetBackupStateResponseDto;
import io.camunda.webapps.backup.Metadata;
import io.camunda.webapps.backup.TakeBackupRequestDto;
import io.camunda.webapps.backup.TakeBackupResponseDto;
import io.camunda.webapps.backup.repository.BackupRepositoryProps;
import io.camunda.webapps.schema.descriptors.backup.BackupPriorities;
import io.camunda.webapps.schema.descriptors.backup.SnapshotIndexCollection;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupServiceImpl
implements BackupService {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupServiceImpl.class);
    private final Executor threadPoolTaskExecutor;
    private final Queue<BackupService.SnapshotRequest> requestsQueue = new ConcurrentLinkedQueue<BackupService.SnapshotRequest>();
    private final BackupPriorities backupPriorities;
    private final BackupRepositoryProps backupProps;
    private final BackupRepository repository;

    public BackupServiceImpl(Executor threadPoolTaskExecutor, BackupPriorities backupPriorities, BackupRepositoryProps backupProps, BackupRepository repository) {
        this.threadPoolTaskExecutor = threadPoolTaskExecutor;
        this.backupPriorities = backupPriorities;
        this.repository = repository;
        this.backupProps = backupProps;
    }

    @Override
    public void deleteBackup(Long backupId) {
        this.repository.validateRepositoryExists(this.backupProps.repositoryName());
        String repositoryName = this.backupProps.repositoryName();
        Optional<Metadata> backupMetadataOpt = this.repository.getMetadata(repositoryName, backupId);
        if (backupMetadataOpt.isEmpty()) {
            throw new BackupException.ResourceNotFoundException("Expected to find backup with ID '%d', but no metadata for it found".formatted(backupId));
        }
        Metadata backupMetadata = backupMetadataOpt.get();
        int partCount = backupMetadata.partCount();
        for (int partIdx = 1; partIdx <= partCount; ++partIdx) {
            String snapshotName = this.repository.snapshotNameProvider().getSnapshotName(backupMetadata.withPart(partIdx).withVersion("*"));
            this.repository.deleteSnapshot(repositoryName, snapshotName);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public TakeBackupResponseDto takeBackup(TakeBackupRequestDto request) {
        this.repository.validateRepositoryExists(this.backupProps.repositoryName());
        this.repository.validateNoDuplicateBackupId(this.backupProps.repositoryName(), request.getBackupId());
        if (!this.requestsQueue.isEmpty()) {
            throw new BackupException.InvalidRequestException("Another backup is running at the moment");
        }
        Queue<BackupService.SnapshotRequest> queue = this.requestsQueue;
        synchronized (queue) {
            if (!this.requestsQueue.isEmpty()) {
                throw new BackupException.InvalidRequestException("Another backup is running at the moment");
            }
            return this.scheduleSnapshots(request);
        }
    }

    @Override
    public GetBackupStateResponseDto getBackupState(Long backupId) {
        return this.repository.getBackupState(this.backupProps.repositoryName(), backupId);
    }

    @Override
    public List<GetBackupStateResponseDto> getBackups(boolean verbose, String pattern) {
        return this.repository.getBackups(this.backupProps.repositoryName(), verbose, pattern);
    }

    TakeBackupResponseDto scheduleSnapshots(TakeBackupRequestDto request) {
        String repositoryName = this.backupProps.repositoryName();
        List<SnapshotIndexCollection> indexPatternsOrdered = this.getValidIndexPatterns();
        int count = indexPatternsOrdered.size();
        ArrayList<String> snapshotNames = new ArrayList<String>();
        String version = this.getCurrentVersion();
        for (int index = 0; index < indexPatternsOrdered.size(); ++index) {
            SnapshotIndexCollection indexCollection = indexPatternsOrdered.get(index);
            int partNum = index + 1;
            Metadata metadata = new Metadata(request.getBackupId(), version, partNum, count);
            String snapshotName = this.repository.snapshotNameProvider().getSnapshotName(metadata);
            BackupService.SnapshotRequest snapshotRequest = new BackupService.SnapshotRequest(repositoryName, snapshotName, indexCollection, metadata);
            LOGGER.debug("Snapshot part {} contains indices {}", (Object)metadata.partNo(), (Object)indexCollection.allIndices());
            this.requestsQueue.offer(snapshotRequest);
            LOGGER.debug("Snapshot scheduled: {}", (Object)snapshotName);
            snapshotNames.add(snapshotName);
        }
        this.scheduleNextSnapshot();
        return new TakeBackupResponseDto().setScheduledSnapshots(snapshotNames);
    }

    void scheduleNextSnapshot() {
        BackupService.SnapshotRequest nextRequest = this.requestsQueue.poll();
        if (nextRequest != null) {
            this.threadPoolTaskExecutor.execute(() -> this.repository.executeSnapshotting(nextRequest, this::scheduleNextSnapshot, this.requestsQueue::clear));
            LOGGER.debug("Snapshot picked for execution: {}", (Object)nextRequest);
        }
    }

    List<SnapshotIndexCollection> getValidIndexPatterns() {
        ArrayList<SnapshotIndexCollection> list = new ArrayList<SnapshotIndexCollection>();
        ArrayList<String> missingIndicesList = new ArrayList<String>();
        for (SnapshotIndexCollection indices : this.backupPriorities.indicesSplitBySnapshot().toList()) {
            Set<String> foundIndices = this.repository.checkAllIndicesExist(indices.allIndices());
            List<String> missingRequiredIndices = indices.requiredIndices().stream().filter(idx -> !foundIndices.contains(idx)).toList();
            if (!missingRequiredIndices.isEmpty()) {
                missingIndicesList.addAll(missingRequiredIndices);
                LOGGER.warn("Missing required indices:{}. All indices found are {}", missingRequiredIndices, foundIndices);
            }
            if (foundIndices.isEmpty()) continue;
            list.add(indices);
        }
        if (!missingIndicesList.isEmpty()) {
            throw new BackupException.IndexNotFoundException(missingIndicesList);
        }
        return list;
    }

    private String getCurrentVersion() {
        return this.backupProps.version().toLowerCase();
    }
}

