/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.webapps.backup.repository.opensearch;

import io.camunda.webapps.backup.BackupException;
import io.camunda.webapps.backup.BackupRepository;
import io.camunda.webapps.backup.BackupService;
import io.camunda.webapps.backup.BackupStateDto;
import io.camunda.webapps.backup.GetBackupStateResponseDetailDto;
import io.camunda.webapps.backup.GetBackupStateResponseDto;
import io.camunda.webapps.backup.Metadata;
import io.camunda.webapps.backup.repository.BackupRepositoryProps;
import io.camunda.webapps.backup.repository.SnapshotNameProvider;
import io.camunda.webapps.backup.repository.opensearch.MetadataMarshaller;
import io.camunda.webapps.backup.repository.opensearch.OpenSearchGetSnapshotResponse;
import io.camunda.webapps.backup.repository.opensearch.OpenSearchSnapshotInfo;
import io.camunda.webapps.backup.repository.opensearch.OpensearchRequestDSL;
import io.camunda.webapps.backup.repository.opensearch.SnapshotState;
import io.camunda.webapps.util.ExceptionSupplier;
import java.net.SocketTimeoutException;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.opensearch.OpenSearchAsyncClient;
import org.opensearch.client.opensearch.OpenSearchClient;
import org.opensearch.client.opensearch._types.OpenSearchException;
import org.opensearch.client.opensearch.snapshot.CreateSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.DeleteSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.GetRepositoryResponse;
import org.opensearch.client.opensearch.snapshot.GetSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.GetSnapshotResponse;
import org.opensearch.client.opensearch.snapshot.SnapshotInfo;
import org.opensearch.client.transport.OpenSearchTransport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpensearchBackupRepository
implements BackupRepository {
    public static final String SNAPSHOT_MISSING_EXCEPTION_TYPE = "snapshot_missing_exception";
    public static final String REPOSITORY_MISSING_EXCEPTION_TYPE = "repository_missing_exception";
    private static final String INDEX_MISSING_EXCEPTION_TYPE = "index_not_found_exception";
    private static final Logger LOGGER = LoggerFactory.getLogger(OpensearchBackupRepository.class);
    private final OpenSearchClient openSearchClient;
    private final OpenSearchAsyncClient openSearchAsyncClient;
    private final BackupRepositoryProps backupProps;
    private final SnapshotNameProvider snapshotNameProvider;

    public OpensearchBackupRepository(OpenSearchClient openSearchClient, OpenSearchAsyncClient openSearchAsyncClient, BackupRepositoryProps operateProperties, SnapshotNameProvider snapshotNameProvider) {
        this.openSearchClient = openSearchClient;
        this.openSearchAsyncClient = openSearchAsyncClient;
        this.backupProps = operateProperties;
        this.snapshotNameProvider = snapshotNameProvider;
    }

    @Override
    public SnapshotNameProvider snapshotNameProvider() {
        return this.snapshotNameProvider;
    }

    @Override
    public void deleteSnapshot(String repositoryName, String snapshotName) {
        DeleteSnapshotRequest request = OpensearchRequestDSL.deleteSnapshotRequestBuilder(repositoryName, snapshotName).build();
        ((CompletableFuture)((CompletableFuture)this.safe(() -> this.openSearchAsyncClient.snapshot().delete(request), e -> "Failed to send snapshot delete request")).thenAccept(response -> LOGGER.debug("Delete snapshot was acknowledged by Opensearch node: {}", (Object)response.acknowledged()))).exceptionally(t -> {
            if (this.isSnapshotMissingException((Throwable)t)) {
                LOGGER.warn("No snapshot found for snapshot deletion: {} ", (Object)t.getMessage());
            } else {
                LOGGER.error("Exception occurred while deleting the snapshot: {} ", (Object)t.getMessage(), t);
            }
            return null;
        });
    }

    @Override
    public void validateRepositoryExists(String repositoryName) {
        try {
            GetRepositoryResponse repositoryResponse = this.openSearchClient.snapshot().getRepository(OpensearchRequestDSL.repositoryRequestBuilder(repositoryName).build());
            LOGGER.debug("Repository {} exists", (Object)repositoryResponse);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = OpensearchBackupRepository.noRepositoryErrorMessage(repositoryName);
                throw new BackupException(reason);
            }
            String reason = String.format("Exception occurred when validating existence of repository with name [%s].", repositoryName);
            throw new BackupException(reason, e);
        }
    }

    @Override
    public void validateNoDuplicateBackupId(String repositoryName, Long backupId) {
        GetSnapshotResponse response;
        String snapshot = this.snapshotNameProvider.getSnapshotNamePrefix(backupId) + "*";
        try {
            response = this.openSearchClient.snapshot().get(OpensearchRequestDSL.getSnapshotRequestBuilder(repositoryName, snapshot).build());
        }
        catch (Exception e) {
            if (this.isSnapshotMissingException(e)) {
                return;
            }
            String reason = String.format("Exception occurred when validating whether backup with ID [%s] already exists.", backupId);
            throw new BackupException(reason, e);
        }
        if (!response.snapshots().isEmpty()) {
            String reason = String.format("A backup with ID [%s] already exists. Found snapshots: [%s]", backupId, response.snapshots().stream().map(SnapshotInfo::uuid).collect(Collectors.joining(", ")));
            throw new BackupException.InvalidRequestException(reason);
        }
    }

    @Override
    public GetBackupStateResponseDto getBackupState(String repositoryName, Long backupId) {
        List<OpenSearchSnapshotInfo> snapshots = this.findSnapshots(repositoryName, backupId);
        return this.toGetBackupStateResponseDto(backupId, snapshots);
    }

    @Override
    public List<GetBackupStateResponseDto> getBackups(String repositoryName) {
        GetSnapshotRequest requestBuilder = OpensearchRequestDSL.getSnapshotRequestBuilder(repositoryName, this.snapshotNameProvider.snapshotNamePrefix() + "*").build();
        try {
            OpenSearchGetSnapshotResponse response = OpenSearchGetSnapshotResponse.fromResponse(this.openSearchClient.snapshot().get(requestBuilder));
            List<OpenSearchSnapshotInfo> snapshots = response.snapshots().stream().sorted(Comparator.comparing(OpenSearchSnapshotInfo::getStartTimeInMillis).reversed()).toList();
            LinkedHashMap groupedSnapshotInfos = snapshots.stream().collect(Collectors.groupingBy(si -> {
                Metadata metadata = MetadataMarshaller.fromMetadata(si.getMetadata(), ((OpenSearchTransport)this.openSearchClient._transport()).jsonpMapper());
                Long backupId = metadata.backupId();
                if (backupId == null) {
                    backupId = this.snapshotNameProvider.extractBackupId(si.getSnapshot());
                }
                return backupId;
            }, LinkedHashMap::new, Collectors.toList()));
            return groupedSnapshotInfos.entrySet().stream().map(entry -> this.toGetBackupStateResponseDto((Long)entry.getKey(), (List)entry.getValue())).toList();
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = OpensearchBackupRepository.noRepositoryErrorMessage(repositoryName);
                throw new BackupException(reason);
            }
            if (this.isSnapshotMissingException(e)) {
                return new ArrayList<GetBackupStateResponseDto>();
            }
            String reason = String.format("Exception occurred when searching for backups: %s", e.getMessage());
            throw new BackupException(reason, e);
        }
    }

    @Override
    public void executeSnapshotting(BackupService.SnapshotRequest snapshotRequest, boolean onlyRequired, Runnable onSuccess, Runnable onFailure) {
        Long backupId = this.backupId(snapshotRequest);
        Map<String, JsonData> metadataJson = MetadataMarshaller.asJson(snapshotRequest.metadata(), ((OpenSearchTransport)this.openSearchClient._transport()).jsonpMapper());
        CreateSnapshotRequest requestBuilder = OpensearchRequestDSL.createSnapshotRequestBuilder(snapshotRequest.repositoryName(), snapshotRequest.snapshotName(), snapshotRequest.indices(onlyRequired)).ignoreUnavailable(Boolean.valueOf(false)).includeGlobalState(Boolean.valueOf(this.backupProps.includeGlobalState())).metadata(metadataJson).featureStates("none", new String[0]).waitForCompletion(Boolean.valueOf(true)).build();
        ((CompletableFuture)((CompletableFuture)this.safe(() -> this.openSearchAsyncClient.snapshot().create(requestBuilder), e -> "Failed to send snapshot create request!")).thenAccept(response -> this.handleSnapshotReceived(response.snapshot(), onSuccess, onFailure))).exceptionally(t -> {
            block10: {
                Throwable e;
                if (t instanceof CompletionException) {
                    CompletionException ce = (CompletionException)t;
                    e = ce.getCause();
                } else {
                    e = t;
                }
                if (e instanceof SocketTimeoutException) {
                    Optional<OpenSearchSnapshotInfo> maybeCurrentSnapshot;
                    LOGGER.warn("Timeout while creating snapshot [{}] for backup id [{}]. Need to keep waiting with polling...", (Object)snapshotRequest.snapshotName(), (Object)backupId);
                    while (true) {
                        List<OpenSearchSnapshotInfo> snapshotInfos;
                        if ((maybeCurrentSnapshot = (snapshotInfos = this.findSnapshots(snapshotRequest.repositoryName(), backupId)).stream().filter(x -> Objects.equals(x.getSnapshot(), snapshotRequest.snapshotName())).findFirst()).isEmpty()) {
                            LOGGER.error("Expected (but not found) snapshot [{}] for backupId [{}].", (Object)snapshotRequest.snapshotName(), (Object)backupId);
                            onFailure.run();
                            break block10;
                        }
                        if (!SnapshotState.STARTED.equals((Object)maybeCurrentSnapshot.get().getState())) break;
                        try {
                            Thread.sleep(100L);
                        }
                        catch (InterruptedException ex) {
                            throw new RuntimeException(ex);
                        }
                    }
                    this.handleSnapshotReceived(maybeCurrentSnapshot.get(), onSuccess, onFailure);
                } else if (this.isErrorType(e, INDEX_MISSING_EXCEPTION_TYPE) && !onlyRequired) {
                    LOGGER.debug("Failed to execute snapshot because some index is missing, retry only with required indices", e);
                    this.executeSnapshotting(snapshotRequest, true, onSuccess, onFailure);
                } else {
                    LOGGER.error("Exception while creating snapshot [{}] for backup id [{}].", new Object[]{snapshotRequest.snapshotName(), backupId, e});
                    onFailure.run();
                }
            }
            return null;
        });
    }

    private boolean isErrorType(Throwable t, String errorType) {
        if (t instanceof OpenSearchException) {
            OpenSearchException oe = (OpenSearchException)t;
            return Objects.equals(oe.error().type(), errorType);
        }
        return false;
    }

    private boolean isSnapshotMissingException(Throwable t) {
        return this.isErrorType(t, SNAPSHOT_MISSING_EXCEPTION_TYPE);
    }

    private boolean isRepositoryMissingException(Exception e) {
        return this.isErrorType(e, REPOSITORY_MISSING_EXCEPTION_TYPE);
    }

    private static String noRepositoryErrorMessage(String repositoryName) {
        return String.format("No repository with name [%s] could be found.", repositoryName);
    }

    private Long backupId(BackupService.SnapshotRequest snapshotRequest) {
        return this.snapshotNameProvider.extractBackupId(snapshotRequest.snapshotName());
    }

    private void handleSnapshotReceived(OpenSearchSnapshotInfo snapshotInfo, Runnable onSuccess, Runnable onFailure) {
        if (SnapshotState.SUCCESS.equals((Object)snapshotInfo.getState())) {
            LOGGER.info("Snapshot done: {}", (Object)snapshotInfo.getUuid());
            onSuccess.run();
        } else if (SnapshotState.FAILED.equals((Object)snapshotInfo.getState())) {
            LOGGER.error("Snapshot taking failed for {}", (Object)snapshotInfo.getUuid());
            onFailure.run();
        } else {
            LOGGER.warn("Snapshot state is {} for snapshot {}", (Object)snapshotInfo.getState(), (Object)snapshotInfo.getUuid());
            onSuccess.run();
        }
    }

    private void handleSnapshotReceived(SnapshotInfo snapshotInfo, Runnable onSuccess, Runnable onFailure) {
        if (SnapshotState.SUCCESS.equals((Object)SnapshotState.valueOf(snapshotInfo.state()))) {
            LOGGER.info("Snapshot done: {}", (Object)snapshotInfo.uuid());
            onSuccess.run();
        } else if (SnapshotState.FAILED.equals((Object)SnapshotState.valueOf(snapshotInfo.state()))) {
            LOGGER.error("Snapshot taking failed for {}, reason {}", (Object)snapshotInfo.uuid(), (Object)snapshotInfo.reason());
            onFailure.run();
        } else {
            LOGGER.warn("Snapshot state is {} for snapshot {}", (Object)snapshotInfo.state(), (Object)snapshotInfo.uuid());
            onSuccess.run();
        }
    }

    private List<OpenSearchSnapshotInfo> findSnapshots(String repositoryName, Long backupId) {
        GetSnapshotRequest requestBuilder = OpensearchRequestDSL.getSnapshotRequestBuilder(repositoryName, this.snapshotNameProvider.getSnapshotNamePrefix(backupId) + "*").build();
        try {
            OpenSearchGetSnapshotResponse response = OpenSearchGetSnapshotResponse.fromResponse(this.openSearchClient.snapshot().get(requestBuilder));
            return response.snapshots();
        }
        catch (Exception e) {
            if (this.isSnapshotMissingException(e)) {
                throw new BackupException.ResourceNotFoundException(String.format("No backup with id [%s] found.", backupId));
            }
            if (this.isRepositoryMissingException(e)) {
                String reason = OpensearchBackupRepository.noRepositoryErrorMessage(repositoryName);
                throw new BackupException(reason);
            }
            String reason = String.format("Exception occurred when searching for backup with ID [%s].", backupId);
            throw new BackupException(reason, e);
        }
    }

    private BackupStateDto getState(List<OpenSearchSnapshotInfo> snapshots, Integer expectedSnapshotsCount) {
        if (snapshots.size() == expectedSnapshotsCount.intValue()) {
            if (snapshots.stream().map(OpenSearchSnapshotInfo::getState).allMatch(SnapshotState.SUCCESS::equals)) {
                return BackupStateDto.COMPLETED;
            }
        }
        if (snapshots.stream().map(OpenSearchSnapshotInfo::getState).anyMatch(s -> SnapshotState.FAILED.equals(s) || SnapshotState.PARTIAL.equals(s))) {
            return BackupStateDto.FAILED;
        }
        if (snapshots.stream().map(OpenSearchSnapshotInfo::getState).anyMatch(SnapshotState.STARTED::equals)) {
            return BackupStateDto.IN_PROGRESS;
        }
        if (snapshots.size() < expectedSnapshotsCount) {
            if (this.isIncompleteCheckTimedOut(this.backupProps.incompleteCheckTimeoutInSeconds(), snapshots.getLast().getEndTimeInMillis())) {
                return BackupStateDto.INCOMPLETE;
            }
            return BackupStateDto.IN_PROGRESS;
        }
        return BackupStateDto.FAILED;
    }

    private GetBackupStateResponseDto toGetBackupStateResponseDto(Long backupId, List<OpenSearchSnapshotInfo> snapshots) {
        GetBackupStateResponseDto response = new GetBackupStateResponseDto(backupId);
        Metadata metadata = MetadataMarshaller.fromMetadata(snapshots.getFirst().getMetadata(), ((OpenSearchTransport)this.openSearchClient._transport()).jsonpMapper());
        Integer expectedSnapshotsCount = metadata.partCount();
        response.setState(this.getState(snapshots, expectedSnapshotsCount));
        response.setDetails(this.getBackupStateDetails(snapshots));
        String failureReason = this.getFailureReason(snapshots, response.getState(), expectedSnapshotsCount);
        if (failureReason != null) {
            response.setFailureReason(failureReason);
        }
        return response;
    }

    private List<GetBackupStateResponseDetailDto> getBackupStateDetails(List<OpenSearchSnapshotInfo> snapshots) {
        ArrayList<GetBackupStateResponseDetailDto> details = new ArrayList<GetBackupStateResponseDetailDto>();
        for (OpenSearchSnapshotInfo snapshot : snapshots) {
            GetBackupStateResponseDetailDto detail = new GetBackupStateResponseDetailDto();
            detail.setSnapshotName(snapshot.getSnapshot());
            detail.setStartTime(OffsetDateTime.ofInstant(Instant.ofEpochMilli(snapshot.getStartTimeInMillis()), ZoneId.systemDefault()));
            if (!snapshot.getFailures().isEmpty()) {
                detail.setFailures((String[])snapshot.getFailures().stream().map(Object::toString).toArray(String[]::new));
            }
            detail.setState(snapshot.getState().toString());
            details.add(detail);
        }
        return details;
    }

    private String getFailureReason(List<OpenSearchSnapshotInfo> snapshots, BackupStateDto state, Integer expectedSnapshotsCount) {
        if (state == BackupStateDto.FAILED) {
            String failedSnapshots = snapshots.stream().filter(s -> SnapshotState.FAILED.equals((Object)s.getState())).map(OpenSearchSnapshotInfo::getSnapshot).collect(Collectors.joining(", "));
            if (!failedSnapshots.isEmpty()) {
                return String.format("There were failures with the following snapshots: %s", failedSnapshots);
            }
            String partialSnapshot = snapshots.stream().filter(s -> SnapshotState.PARTIAL.equals((Object)s.getState())).map(OpenSearchSnapshotInfo::getSnapshot).collect(Collectors.joining(", "));
            if (!partialSnapshot.isEmpty()) {
                return String.format("Some of the snapshots are partial: %s", partialSnapshot);
            }
            if (snapshots.size() > expectedSnapshotsCount) {
                return "More snapshots found than expected.";
            }
        }
        return null;
    }

    private <R> R safe(ExceptionSupplier<R> supplier, Function<Exception, String> errorMessage) {
        try {
            return (R)supplier.get();
        }
        catch (OpenSearchException e) {
            throw e;
        }
        catch (Exception e) {
            String message = errorMessage.apply(e);
            LOGGER.error(message, (Throwable)e);
            throw new BackupException(message, e);
        }
    }
}

