/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.tasklist.webapp.service;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.security.auth.CamundaAuthentication;
import io.camunda.security.auth.CamundaAuthenticationProvider;
import io.camunda.tasklist.Metrics;
import io.camunda.tasklist.exceptions.TasklistRuntimeException;
import io.camunda.tasklist.property.Auth0Properties;
import io.camunda.tasklist.property.TasklistProperties;
import io.camunda.tasklist.store.FormStore;
import io.camunda.tasklist.store.TaskMetricsStore;
import io.camunda.tasklist.store.TaskStore;
import io.camunda.tasklist.store.VariableStore;
import io.camunda.tasklist.util.CollectionUtil;
import io.camunda.tasklist.views.TaskSearchView;
import io.camunda.tasklist.webapp.dto.TaskDTO;
import io.camunda.tasklist.webapp.dto.TaskQueryDTO;
import io.camunda.tasklist.webapp.dto.VariableDTO;
import io.camunda.tasklist.webapp.dto.VariableInputDTO;
import io.camunda.tasklist.webapp.es.TaskValidator;
import io.camunda.tasklist.webapp.rest.exception.ForbiddenActionException;
import io.camunda.tasklist.webapp.rest.exception.InvalidRequestException;
import io.camunda.tasklist.webapp.security.TasklistAuthenticationUtil;
import io.camunda.tasklist.webapp.service.OrganizationService;
import io.camunda.tasklist.webapp.service.VariableService;
import io.camunda.tasklist.zeebe.TasklistServicesAdapter;
import io.camunda.webapps.schema.entities.usertask.TaskEntity;
import io.camunda.webapps.schema.entities.usertask.TaskState;
import java.io.IOException;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpServerErrorException;

@Component
public class TaskService {
    private static final Logger LOGGER = LoggerFactory.getLogger(TaskService.class);
    private static final String DEFAULT_USER = "No name";
    @Autowired
    private CamundaAuthenticationProvider authenticationProvider;
    @Autowired
    private TaskStore taskStore;
    @Autowired
    private VariableService variableService;
    @Autowired
    private FormStore formStore;
    @Autowired
    @Qualifier(value="tasklistObjectMapper")
    private ObjectMapper objectMapper;
    @Autowired
    private Metrics metrics;
    @Autowired
    private TaskMetricsStore taskMetricsStore;
    @Autowired
    private TaskValidator taskValidator;
    @Autowired
    private TasklistServicesAdapter tasklistServicesAdapter;
    @Autowired
    private TasklistProperties tasklistProperties;
    @Autowired
    private OrganizationService organizationService;

    public List<TaskDTO> getTasks(TaskQueryDTO query) {
        return this.getTasks(query, Collections.emptySet(), false);
    }

    public List<TaskDTO> getTasks(TaskQueryDTO query, Set<String> includeVariableNames, boolean fetchFullValuesFromDB) {
        if (CollectionUtil.countNonNullObjects((Object[])new Object[]{query.getSearchAfter(), query.getSearchAfterOrEqual(), query.getSearchBefore(), query.getSearchBeforeOrEqual()}) > 1L) {
            throw new InvalidRequestException("Only one of [searchAfter, searchAfterOrEqual, searchBefore, searchBeforeOrEqual] must be present in request.");
        }
        if (query.getPageSize() <= 0) {
            throw new InvalidRequestException("Page size should be a positive number");
        }
        if (query.getImplementation() != null && !query.getImplementation().equals((Object)TaskEntity.TaskImplementation.ZEEBE_USER_TASK) && !query.getImplementation().equals((Object)TaskEntity.TaskImplementation.JOB_WORKER)) {
            throw new InvalidRequestException(String.format("Invalid implementation, the valid values are %s and %s", TaskEntity.TaskImplementation.ZEEBE_USER_TASK, TaskEntity.TaskImplementation.JOB_WORKER));
        }
        List tasks = this.taskStore.getTasks(query.toTaskQuery());
        Set fieldNames = fetchFullValuesFromDB ? Collections.emptySet() : Set.of("id", "name", "previewValue", "isValueTruncated");
        Map variablesPerTaskId = CollectionUtils.isEmpty(includeVariableNames) ? Collections.emptyMap() : this.variableService.getVariablesPerTaskId(tasks.stream().map(taskView -> VariableStore.GetVariablesRequest.createFrom((TaskSearchView)taskView, new ArrayList(includeVariableNames), (Set)fieldNames)).toList());
        return tasks.stream().map(it -> TaskDTO.createFrom(it, Optional.ofNullable((List)variablesPerTaskId.get(it.getId())).map(list -> list.toArray(new VariableDTO[list.size()])).orElse(null), this.objectMapper)).toList();
    }

    public TaskDTO getTask(String taskId) {
        TaskEntity task = this.taskStore.getTask(taskId);
        if (this.taskFormLinkIsNotComplete(task)) {
            LOGGER.debug("Task with id {} found having incorrect form linking to form with key {}", (Object)taskId, (Object)task.getFormKey());
            Optional linkedForm = this.formStore.getFormByKey(task.getFormKey());
            linkedForm.ifPresent(form -> {
                this.updateTaskLinkedForm(task, (FormStore.FormIdView)form);
                task.setFormId(form.bpmnId());
                task.setFormVersion(form.version());
            });
        }
        return TaskDTO.createFrom(task, this.objectMapper);
    }

    public TaskDTO assignTask(String taskId, String assignee, Boolean allowOverrideAssignment) {
        TaskEntity claimedTask;
        if (allowOverrideAssignment == null) {
            allowOverrideAssignment = true;
        }
        boolean isApiUser = TasklistAuthenticationUtil.isApiUser(this.authenticationProvider.getCamundaAuthentication());
        if (StringUtils.isEmpty((CharSequence)assignee) && isApiUser) {
            throw new InvalidRequestException("Assignee must be specified");
        }
        boolean allowNonSelfAssignment = Optional.ofNullable(this.tasklistProperties.getFeatureFlag().getAllowNonSelfAssignment()).orElse(false);
        CamundaAuthentication currentAuthentication = this.authenticationProvider.getCamundaAuthentication();
        String currentUsername = currentAuthentication.authenticatedUsername();
        if (!allowNonSelfAssignment && StringUtils.isNotEmpty((CharSequence)assignee) && !isApiUser && !assignee.equals(currentUsername)) {
            throw new ForbiddenActionException("User doesn't have the permission to assign another user to this task");
        }
        TaskEntity taskBefore = this.taskStore.getTask(taskId);
        if (taskBefore.getImplementation() == TaskEntity.TaskImplementation.ZEEBE_USER_TASK) {
            taskAssignee = this.determineTaskAssignee(assignee, currentUsername);
            this.tasklistServicesAdapter.assignUserTask(taskBefore, taskAssignee);
            claimedTask = this.taskStore.makeCopyOf(taskBefore).setAssignee(taskAssignee);
        } else {
            this.taskValidator.validateCanAssign(taskBefore, allowOverrideAssignment);
            taskAssignee = this.determineTaskAssignee(assignee, currentUsername);
            this.tasklistServicesAdapter.assignUserTask(taskBefore, taskAssignee);
            claimedTask = this.taskStore.persistTaskClaim(taskBefore, taskAssignee);
        }
        String[] assignedTaskMetrics = this.getTaskMetricLabels(claimedTask, currentUsername);
        this.updateClaimedMetric(assignedTaskMetrics);
        this.updateTaskAssignedMetric(claimedTask);
        return TaskDTO.createFrom(claimedTask, this.objectMapper);
    }

    public void updateTaskAssignedMetric(TaskEntity task) {
        if (TaskEntity.TaskImplementation.JOB_WORKER.equals((Object)task.getImplementation())) {
            this.taskMetricsStore.registerTaskAssigned(task);
        }
    }

    private String determineTaskAssignee(String assignee, String authenticatedUsername) {
        return StringUtils.isEmpty((CharSequence)assignee) && !TasklistAuthenticationUtil.isApiUser(this.authenticationProvider.getCamundaAuthentication()) ? authenticatedUsername : assignee;
    }

    public TaskDTO completeTask(String taskId, List<VariableInputDTO> variables, boolean withDraftVariableValues) {
        HashMap variablesMap = new HashMap();
        Objects.requireNonNullElse(variables, Collections.emptyList()).forEach(variable -> variablesMap.put(variable.getName(), this.extractTypedValue((VariableInputDTO)variable)));
        try {
            TaskEntity completedTaskEntity;
            LOGGER.info("Starting completion of task with ID: {}", (Object)taskId);
            TaskEntity task = this.taskStore.getTask(taskId);
            if (task.getImplementation() == TaskEntity.TaskImplementation.ZEEBE_USER_TASK) {
                this.tasklistServicesAdapter.completeUserTask(task, variablesMap);
                completedTaskEntity = this.taskStore.makeCopyOf(task).setState(TaskState.COMPLETED).setCompletionTime(OffsetDateTime.now());
            } else {
                this.taskValidator.validateCanComplete(task);
                this.tasklistServicesAdapter.completeUserTask(task, variablesMap);
                completedTaskEntity = this.taskStore.persistTaskCompletion(task);
            }
            try {
                LOGGER.info("Start variable persistence: {}", (Object)taskId);
                this.variableService.persistTaskVariables(taskId, variables, withDraftVariableValues);
                this.deleteDraftTaskVariablesSafely(taskId);
                this.updateCompletedMetric(completedTaskEntity);
                LOGGER.info("Task with ID {} completed successfully.", (Object)taskId);
            }
            catch (Exception e) {
                LOGGER.error("Task with key {} was COMPLETED but error happened after completion: {}.", (Object)taskId, (Object)e.getMessage());
            }
            return TaskDTO.createFrom(completedTaskEntity, this.objectMapper);
        }
        catch (HttpServerErrorException e) {
            LOGGER.error("Error completing task with ID: {}. Details: {}", new Object[]{taskId, e.getMessage(), e});
            throw new TasklistRuntimeException("Error completing task with ID: " + taskId, (Throwable)e);
        }
    }

    void deleteDraftTaskVariablesSafely(String taskId) {
        try {
            LOGGER.info("Start deletion of draft task variables associated with task with id='{}'", (Object)taskId);
            this.variableService.deleteDraftTaskVariables(taskId);
        }
        catch (Exception ex) {
            String errorMessage = String.format("Error during deletion of draft task variables associated with task with id='%s'", taskId);
            LOGGER.error(errorMessage, (Throwable)ex);
        }
    }

    private Object extractTypedValue(VariableInputDTO variable) {
        if (variable.getValue().equals("null")) {
            return this.objectMapper.nullNode();
        }
        try {
            return this.objectMapper.readValue(variable.getValue(), Object.class);
        }
        catch (IOException e) {
            throw new TasklistRuntimeException(e.getMessage(), (Throwable)e);
        }
    }

    public TaskDTO unassignTask(String taskId) {
        TaskEntity taskBefore = this.taskStore.getTask(taskId);
        if (taskBefore.getImplementation() == TaskEntity.TaskImplementation.ZEEBE_USER_TASK) {
            this.tasklistServicesAdapter.unassignUserTask(taskBefore);
            TaskEntity taskEntity = this.taskStore.makeCopyOf(taskBefore).setAssignee(null);
            return TaskDTO.createFrom(taskEntity, this.objectMapper);
        }
        this.taskValidator.validateCanUnassign(taskBefore);
        TaskEntity taskEntity = this.taskStore.persistTaskUnclaim(taskBefore);
        try {
            this.tasklistServicesAdapter.unassignUserTask(taskEntity);
        }
        catch (Exception e) {
            this.taskStore.persistTaskClaim(taskBefore, taskBefore.getAssignee());
            throw e;
        }
        return TaskDTO.createFrom(taskEntity, this.objectMapper);
    }

    private boolean taskFormLinkIsNotComplete(TaskEntity task) {
        return task.getFormKey() != null && task.getFormId() == null && (task.getIsFormEmbedded() == null || task.getIsFormEmbedded() == false) && task.getExternalFormReference() == null;
    }

    private void updateTaskLinkedForm(TaskEntity task, FormStore.FormIdView form) {
        CompletableFuture.runAsync(() -> {
            this.taskStore.updateTaskLinkedForm(task, form.bpmnId(), form.version().longValue());
            LOGGER.debug("Updated Task with id {} form link of key {} to formId {} and version {}", new Object[]{task.getKey(), task.getFormKey(), form.bpmnId(), form.version()});
        });
    }

    private void updateClaimedMetric(String[] metricsLabels) {
        this.metrics.recordCounts("claimed.tasks", 1L, metricsLabels);
    }

    private void updateCompletedMetric(TaskEntity task) {
        LOGGER.info("Updating completed task metric for task with ID: {}", (Object)task.getKey());
        try {
            CamundaAuthentication currentAuthentication = this.authenticationProvider.getCamundaAuthentication();
            String authenticatedUsername = currentAuthentication.authenticatedUsername();
            String[] completedTaskLabels = this.getTaskMetricLabels(task, authenticatedUsername);
            this.metrics.recordCounts("completed.tasks", 1L, completedTaskLabels);
        }
        catch (Exception e) {
            LOGGER.error("Error updating completed task metric for task with ID: {}", (Object)task.getKey(), (Object)e);
            throw new TasklistRuntimeException("Error updating completed task metric for task with ID: " + task.getKey(), (Throwable)e);
        }
    }

    private String[] getTaskMetricLabels(TaskEntity task, String username) {
        String keyUserId = TasklistAuthenticationUtil.isApiUser(this.authenticationProvider.getCamundaAuthentication()) ? (task.getAssignee() != null ? task.getAssignee() : DEFAULT_USER) : username;
        return new String[]{"bpmnProcessId", task.getBpmnProcessId(), "flowNodeId", task.getFlowNodeBpmnId(), "userId", keyUserId, "organizationId", this.organizationService.getOrganizationIfPresent()};
    }

    private String getOrganizationIfPresent() {
        return Optional.ofNullable(this.tasklistProperties.getAuth0()).map(Auth0Properties::getOrganization).orElse("null");
    }
}

