/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.tasklist.webapp.es.backup.os;

import io.camunda.tasklist.data.conditionals.OpenSearchCondition;
import io.camunda.tasklist.exceptions.TasklistElasticsearchConnectionException;
import io.camunda.tasklist.exceptions.TasklistRuntimeException;
import io.camunda.tasklist.property.TasklistProperties;
import io.camunda.tasklist.webapp.es.backup.BackupManager;
import io.camunda.tasklist.webapp.es.backup.Metadata;
import io.camunda.tasklist.webapp.es.backup.os.GetCustomSnapshotResponse;
import io.camunda.tasklist.webapp.rest.exception.InvalidRequestException;
import io.camunda.tasklist.webapp.rest.exception.NotFoundApiException;
import io.camunda.webapps.backup.BackupStateDto;
import io.camunda.webapps.backup.GetBackupStateResponseDetailDto;
import io.camunda.webapps.backup.GetBackupStateResponseDto;
import io.camunda.webapps.backup.TakeBackupRequestDto;
import io.camunda.webapps.backup.TakeBackupResponseDto;
import java.io.IOException;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.stream.Collectors;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.opensearch.OpenSearchAsyncClient;
import org.opensearch.client.opensearch.OpenSearchClient;
import org.opensearch.client.opensearch._types.OpenSearchException;
import org.opensearch.client.opensearch.snapshot.CreateSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.DeleteSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.DeleteSnapshotResponse;
import org.opensearch.client.opensearch.snapshot.GetRepositoryRequest;
import org.opensearch.client.opensearch.snapshot.GetRepositoryResponse;
import org.opensearch.client.opensearch.snapshot.GetSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.SnapshotInfo;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.JsonEndpoint;
import org.opensearch.client.transport.OpenSearchTransport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

@Component
@Configuration
@Conditional(value={OpenSearchCondition.class})
public class BackupManagerOpenSearch
extends BackupManager {
    public static final String SNAPSHOT_MISSING_EXCEPTION_TYPE = "snapshot_missing_exception";
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupManagerOpenSearch.class);
    private static final String REPOSITORY_MISSING_EXCEPTION_TYPE = "repository_missing_exception";
    private final Queue<CreateSnapshotRequest> requestsQueue = new ConcurrentLinkedQueue<CreateSnapshotRequest>();
    @Autowired
    private TasklistProperties tasklistProperties;
    @Autowired
    @Qualifier(value="tasklistOsAsyncClient")
    private OpenSearchAsyncClient openSearchAsyncClient;
    @Autowired
    @Qualifier(value="tasklistOsClient")
    private OpenSearchClient openSearchClient;

    public void deleteBackup(Long backupId) {
        this.validateRepositoryExists();
        String repositoryName = this.getRepositoryName();
        int count = this.getIndexPatternsOrdered().length;
        String version = this.getCurrentTasklistVersion();
        for (int index = 0; index < count; ++index) {
            String snapshotName = new Metadata().setVersion(version).setPartCount(count).setPartNo(index + 1).setBackupId(backupId).buildSnapshotName();
            DeleteSnapshotRequest request = DeleteSnapshotRequest.of(dsr -> dsr.repository(repositoryName).snapshot(snapshotName));
            try {
                this.openSearchAsyncClient.snapshot().delete(request).whenComplete(BackupManagerOpenSearch::handleSnapshotDeletion);
                continue;
            }
            catch (IOException | OpenSearchException e) {
                LOGGER.error("Exception occurred while deleting the snapshot: " + e.getMessage(), e);
                throw new TasklistRuntimeException("Exception occurred while deleting the snapshot", e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TakeBackupResponseDto takeBackup(TakeBackupRequestDto request) {
        this.validateRepositoryExists();
        this.validateNoDuplicateBackupId(request.getBackupId());
        if (!this.requestsQueue.isEmpty()) {
            throw new InvalidRequestException("Another backup is running at the moment");
        }
        Queue<CreateSnapshotRequest> queue = this.requestsQueue;
        synchronized (queue) {
            if (!this.requestsQueue.isEmpty()) {
                throw new InvalidRequestException("Another backup is running at the moment");
            }
            return this.scheduleSnapshots(request);
        }
    }

    public GetBackupStateResponseDto getBackupState(Long backupId) {
        List<SnapshotInfo> snapshots = this.findSnapshots(backupId);
        return this.getBackupResponse(backupId, snapshots);
    }

    public List<GetBackupStateResponseDto> getBackups() {
        GetSnapshotRequest snapshotStatusRequest = GetSnapshotRequest.of(gsr -> gsr.repository(this.getRepositoryName()).snapshot("camunda_tasklist_*", new String[0]));
        try {
            GetCustomSnapshotResponse response = this.getCustomSnapshotResponse(snapshotStatusRequest);
            List<SnapshotInfo> snapshots = response.snapshots().stream().sorted(Comparator.comparing(SnapshotInfo::startTimeInMillis).reversed()).toList();
            LinkedHashMap groupedSnapshotInfos = snapshots.stream().collect(Collectors.groupingBy(si -> {
                Map jsonDataMap = si.metadata();
                Metadata metadata = BackupManagerOpenSearch.getMetadata(jsonDataMap);
                Long backupId = metadata.getBackupId();
                if (backupId == null) {
                    backupId = Metadata.extractBackupIdFromSnapshotName(si.snapshot());
                }
                return backupId;
            }, LinkedHashMap::new, Collectors.toList()));
            return groupedSnapshotInfos.entrySet().stream().map(entry -> this.getBackupResponse((Long)entry.getKey(), (List)entry.getValue())).collect(Collectors.toList());
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for snapshots. Repository name: [%s].", this.getRepositoryName());
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", this.tasklistProperties.getBackup().getRepositoryName());
                throw new TasklistRuntimeException(reason);
            }
            if (BackupManagerOpenSearch.isSnapshotMissingException(e)) {
                return new ArrayList<GetBackupStateResponseDto>();
            }
            String reason = String.format("Exception occurred when searching for backups: %s", e.getMessage());
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
    }

    private static void handleSnapshotDeletion(DeleteSnapshotResponse result, Throwable ex) {
        if (ex != null) {
            if (BackupManagerOpenSearch.isSnapshotMissingException(ex)) {
                LOGGER.warn("No snapshot found for snapshot deletion: " + ex.getMessage());
            } else {
                LOGGER.error("Exception occurred while deleting the snapshot: " + ex.getMessage(), ex);
            }
        } else {
            LOGGER.debug("Delete snapshot was acknowledged by OpenSearch node: " + result.acknowledged());
        }
    }

    private void validateRepositoryExists() {
        String repositoryName = this.getRepositoryName();
        GetRepositoryRequest getRepositoryRequest = GetRepositoryRequest.of(grr -> grr.name(repositoryName, new String[0]));
        try {
            this.getRepository(getRepositoryRequest);
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while retrieving repository with name [%s].", repositoryName);
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", repositoryName);
                throw new TasklistRuntimeException(reason);
            }
            String reason = String.format("Exception occurred when validating existence of repository with name [%s].", repositoryName);
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
    }

    private GetRepositoryResponse getRepository(GetRepositoryRequest getRepositoryRequest) throws IOException {
        return (GetRepositoryResponse)this.openSearchAsyncClient.snapshot().getRepository(getRepositoryRequest).join();
    }

    private static boolean isSnapshotMissingException(Throwable e) {
        return e.getMessage().contains(SNAPSHOT_MISSING_EXCEPTION_TYPE);
    }

    private boolean isRepositoryMissingException(Exception e) {
        return e.getMessage().contains(REPOSITORY_MISSING_EXCEPTION_TYPE);
    }

    private void validateNoDuplicateBackupId(Long backupId) {
        GetCustomSnapshotResponse response;
        GetSnapshotRequest snapshotsStatusRequest = GetSnapshotRequest.of(gsr -> gsr.repository(this.getRepositoryName()).snapshot(Metadata.buildSnapshotNamePrefix(backupId) + "*", new String[0]));
        try {
            response = this.getCustomSnapshotResponse(snapshotsStatusRequest);
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for duplicate backup. Repository name: [%s].", this.getRepositoryName());
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (BackupManagerOpenSearch.isSnapshotMissingException(e)) {
                return;
            }
            String reason = String.format("Exception occurred when validating whether backup with ID [%s] already exists.", backupId);
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
        if (!response.snapshots().isEmpty()) {
            String reason = String.format("A backup with ID [%s] already exists. Found snapshots: [%s]", backupId, response.snapshots().stream().map(this::getSnapshotId).collect(Collectors.joining(", ")));
            throw new InvalidRequestException(reason);
        }
    }

    private TakeBackupResponseDto scheduleSnapshots(TakeBackupRequestDto request) {
        String repositoryName = this.getRepositoryName();
        int count = this.getIndexPatternsOrdered().length;
        ArrayList<String> snapshotNames = new ArrayList<String>();
        String version = this.getCurrentTasklistVersion();
        for (int index = 0; index < count; ++index) {
            String[] indexPattern = this.getIndexPatternsOrdered()[index];
            Metadata metadata = new Metadata().setVersion(version).setPartCount(count).setPartNo(index + 1).setBackupId(request.getBackupId());
            String snapshotName = metadata.buildSnapshotName();
            this.requestsQueue.offer(CreateSnapshotRequest.of(csr -> csr.repository(repositoryName).snapshot(snapshotName).indices(Arrays.stream(indexPattern).toList()).ignoreUnavailable(Boolean.valueOf(false)).includeGlobalState(Boolean.valueOf(true)).metadata(Map.of("backupId", JsonData.of((Object)metadata.getBackupId()), "version", JsonData.of((Object)metadata.getVersion()), "partNo", JsonData.of((Object)metadata.getPartNo()), "partCount", JsonData.of((Object)metadata.getPartCount()))).featureStates("none", new String[0]).waitForCompletion(Boolean.valueOf(true))));
            LOGGER.debug("Snapshot scheduled: " + snapshotName);
            snapshotNames.add(snapshotName);
        }
        this.scheduleNextSnapshot();
        return new TakeBackupResponseDto().setScheduledSnapshots(snapshotNames);
    }

    private void scheduleNextSnapshot() {
        CreateSnapshotRequest nextRequest = this.requestsQueue.poll();
        if (nextRequest != null) {
            this.getTaskExecutor().submit(() -> this.executeSnapshotting(nextRequest));
            LOGGER.debug("Snapshot picked for execution: snapshot [{}:{}]", (Object)nextRequest.repository(), (Object)nextRequest.snapshot());
        }
    }

    private void executeSnapshotting(CreateSnapshotRequest snapshotRequest) {
        try {
            this.openSearchAsyncClient.snapshot().create(snapshotRequest).whenComplete((response, ex) -> {
                if (ex != null) {
                    LOGGER.error("Snapshot taking failed", ex);
                    this.requestsQueue.clear();
                } else {
                    switch (Objects.requireNonNullElse(response.snapshot().state(), "null")) {
                        case "SUCCESS": {
                            LOGGER.info("Snapshot done: " + this.getSnapshotId(response.snapshot()));
                            this.scheduleNextSnapshot();
                            break;
                        }
                        case "FAILED": {
                            LOGGER.error("Snapshot taking failed for {}, reason {}", (Object)this.getSnapshotId(response.snapshot()), (Object)response.snapshot().reason());
                            this.requestsQueue.clear();
                            break;
                        }
                        default: {
                            LOGGER.warn("Snapshot status {} for the {}", (Object)response.snapshot().state(), (Object)this.getSnapshotId(response.snapshot()));
                            this.scheduleNextSnapshot();
                        }
                    }
                }
            });
        }
        catch (IOException e) {
            throw new TasklistRuntimeException((Throwable)e);
        }
    }

    private String getSnapshotId(SnapshotInfo snapshotInfo) {
        return String.format("%s/%s", snapshotInfo.snapshot(), snapshotInfo.uuid());
    }

    private List<SnapshotInfo> findSnapshots(Long backupId) {
        GetSnapshotRequest snapshotStatusRequest = GetSnapshotRequest.of(gsr -> gsr.repository(this.getRepositoryName()).snapshot(Metadata.buildSnapshotNamePrefix(backupId) + "*", new String[0]));
        try {
            return this.getCustomSnapshotResponse(snapshotStatusRequest).snapshots();
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for snapshots. Repository name: [%s].", this.getRepositoryName());
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (BackupManagerOpenSearch.isSnapshotMissingException(e)) {
                throw new NotFoundApiException(String.format("No backup with id [%s] found.", backupId), e);
            }
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", this.tasklistProperties.getBackup().getRepositoryName());
                throw new TasklistRuntimeException(reason);
            }
            String reason = String.format("Exception occurred when searching for backup with ID [%s].", backupId);
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
    }

    public GetCustomSnapshotResponse getCustomSnapshotResponse(GetSnapshotRequest request) throws IOException, OpenSearchException {
        JsonEndpoint endpoint = (JsonEndpoint)GetCustomSnapshotResponse.ENDPOINT;
        return (GetCustomSnapshotResponse)((OpenSearchTransport)this.openSearchClient._transport()).performRequest((Object)request, (Endpoint)endpoint, this.openSearchClient._transportOptions());
    }

    private GetBackupStateResponseDto getBackupResponse(Long backupId, List<SnapshotInfo> snapshots) {
        GetBackupStateResponseDto response = new GetBackupStateResponseDto(backupId);
        Map jsonDataMap = snapshots.get(0).metadata();
        Metadata metadata = BackupManagerOpenSearch.getMetadata(jsonDataMap);
        Integer expectedSnapshotsCount = metadata.getPartCount();
        if (snapshots.size() == expectedSnapshotsCount.intValue() && snapshots.stream().map(SnapshotInfo::state).allMatch("SUCCESS"::equals)) {
            response.setState(BackupStateDto.COMPLETED);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch(s -> "FAILED".equals(s) || "PARTIAL".equals(s))) {
            response.setState(BackupStateDto.FAILED);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch("INCOMPATIBLE"::equals)) {
            response.setState(BackupStateDto.INCOMPATIBLE);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch("IN_PROGRESS"::equals)) {
            response.setState(BackupStateDto.IN_PROGRESS);
        } else if (snapshots.size() < expectedSnapshotsCount) {
            response.setState(BackupStateDto.INCOMPLETE);
        } else {
            response.setState(BackupStateDto.FAILED);
        }
        ArrayList<GetBackupStateResponseDetailDto> details = new ArrayList<GetBackupStateResponseDetailDto>();
        for (SnapshotInfo snapshot : snapshots) {
            GetBackupStateResponseDetailDto detail = new GetBackupStateResponseDetailDto();
            detail.setSnapshotName(snapshot.snapshot());
            detail.setStartTime(OffsetDateTime.ofInstant(Instant.ofEpochMilli(Long.parseLong(Objects.requireNonNull(snapshot.startTimeInMillis()))), ZoneId.systemDefault()));
            if (snapshot.failures() != null) {
                detail.setFailures((String[])snapshot.failures().stream().map(Object::toString).toArray(String[]::new));
            }
            detail.setState(snapshot.state());
            details.add(detail);
        }
        response.setDetails(details);
        if (response.getState().equals((Object)BackupStateDto.FAILED)) {
            String failureReason = null;
            String failedSnapshots = snapshots.stream().filter(s -> "FAILED".equals(s.state())).map(SnapshotInfo::snapshot).collect(Collectors.joining(", "));
            if (!failedSnapshots.isEmpty()) {
                failureReason = String.format("There were failures with the following snapshots: %s", failedSnapshots);
            } else {
                String partialSnapshot = snapshots.stream().filter(s -> "PARTIAL".equals(s.state())).map(SnapshotInfo::snapshot).collect(Collectors.joining(", "));
                if (!partialSnapshot.isEmpty()) {
                    failureReason = String.format("Some of the snapshots are partial: %s", partialSnapshot);
                } else if (snapshots.size() > expectedSnapshotsCount) {
                    failureReason = "More snapshots found than expected.";
                }
            }
            if (failureReason != null) {
                response.setFailureReason(failureReason);
            }
        }
        return response;
    }

    private static Metadata getMetadata(Map<String, JsonData> jsonDataMap) {
        return new Metadata().setBackupId((Long)jsonDataMap.get("backupId").to(Long.class)).setPartCount((Integer)jsonDataMap.get("partCount").to(Integer.class)).setPartNo((Integer)jsonDataMap.get("partNo").to(Integer.class)).setVersion((String)jsonDataMap.get("version").to(String.class));
    }

    @Bean(value={"tasklistBackupThreadPoolExecutor"})
    public ThreadPoolTaskExecutor getTaskExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(1);
        executor.setMaxPoolSize(1);
        executor.setThreadNamePrefix("backup_os_");
        executor.setQueueCapacity(6);
        executor.initialize();
        return executor;
    }
}

