/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.tasklist.webapp.security;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.tasklist.data.conditionals.OpenSearchCondition;
import io.camunda.tasklist.os.RetryOpenSearchClient;
import io.camunda.tasklist.schema.indices.TasklistWebSessionIndex;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import jakarta.servlet.http.HttpServletRequest;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import org.opensearch.client.opensearch.core.SearchRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.context.annotation.Conditional;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.support.GenericConversionService;
import org.springframework.core.serializer.support.DeserializingConverter;
import org.springframework.core.serializer.support.SerializingConverter;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.security.authentication.InsufficientAuthenticationException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.session.MapSession;
import org.springframework.session.Session;
import org.springframework.session.SessionRepository;
import org.springframework.session.config.annotation.web.http.EnableSpringHttpSession;
import org.springframework.stereotype.Component;

@ConditionalOnExpression(value="${camunda.tasklist.persistent.sessions.enabled:false} or ${camunda.tasklist.persistentSessionsEnabled:false}")
@Component
@EnableSpringHttpSession
@Conditional(value={OpenSearchCondition.class})
public class OpenSearchSessionRepository
implements SessionRepository<OpenSearchSession> {
    public static final int DELETE_EXPIRED_SESSIONS_DELAY = 1800000;
    public static final String POLLING_HEADER = "x-is-polling";
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenSearchSessionRepository.class);
    @Autowired
    @Qualifier(value="sessionThreadPoolScheduler")
    public ThreadPoolTaskScheduler taskScheduler;
    @Autowired
    private RetryOpenSearchClient retryOpenSearchClient;
    @Autowired
    private GenericConversionService conversionService;
    @Autowired
    private TasklistWebSessionIndex tasklistWebSessionIndex;
    @Autowired
    private HttpServletRequest request;
    @Autowired
    @Qualifier(value="tasklistObjectMapper")
    private ObjectMapper objectMapper;

    @PostConstruct
    private void setUp() {
        LOGGER.debug("Persistent sessions in OpenSearch enabled");
        this.setupConverter();
        this.startExpiredSessionCheck();
    }

    @PreDestroy
    private void tearDown() {
        LOGGER.debug("Shutdown OpenSearchSessionRepository");
    }

    private void setupConverter() {
        this.conversionService.addConverter(Object.class, byte[].class, (Converter)new SerializingConverter());
        this.conversionService.addConverter(byte[].class, Object.class, (Converter)new DeserializingConverter());
    }

    private void startExpiredSessionCheck() {
        this.taskScheduler.scheduleAtFixedRate(this::removedExpiredSessions, 1800000L);
    }

    private void removedExpiredSessions() {
        LOGGER.debug("Check for expired sessions");
        SearchRequest.Builder searchRequest = new SearchRequest.Builder();
        searchRequest.index(this.tasklistWebSessionIndex.getFullQualifiedName(), new String[0]);
        this.retryOpenSearchClient.doWithEachSearchResult(searchRequest, sh -> {
            Map document = (Map)this.objectMapper.convertValue(sh.source(), (TypeReference)new TypeReference<Map<String, Object>>(this){});
            Optional<OpenSearchSession> maybeSession = this.documentToSession(document);
            if (maybeSession.isPresent()) {
                Session session = maybeSession.get();
                LOGGER.debug("Check if session {} is expired: {}", (Object)session, (Object)session.isExpired());
                if (session.isExpired()) {
                    this.deleteById(session.getId());
                }
            } else {
                this.deleteById(this.getSessionIdFrom(document));
            }
        });
    }

    private boolean shouldDeleteSession(OpenSearchSession session) {
        return session.isExpired() || session.containsAuthentication() && !session.isAuthenticated();
    }

    public OpenSearchSession createSession() {
        String sessionId = UUID.randomUUID().toString().replace("-", "");
        OpenSearchSession session = new OpenSearchSession(sessionId);
        LOGGER.debug("Create session {} with maxInactiveInterval {} s", (Object)session, (Object)session.getMaxInactiveInterval());
        return session;
    }

    public void save(OpenSearchSession session) {
        LOGGER.debug("Save session {}", (Object)session);
        if (this.shouldDeleteSession(session)) {
            this.deleteById(session.getId());
            return;
        }
        if (session.isChanged()) {
            LOGGER.debug("Session {} changed, save in OpenSearch.", (Object)session);
            this.retryOpenSearchClient.createOrUpdateDocument(this.tasklistWebSessionIndex.getFullQualifiedName(), session.getId(), this.sessionToDocument(session));
            session.clearChangeFlag();
        }
    }

    public OpenSearchSession findById(String id) {
        Map document;
        LOGGER.debug("Retrieve session {} from OpenSearch", (Object)id);
        try {
            document = this.retryOpenSearchClient.getDocument(this.tasklistWebSessionIndex.getFullQualifiedName(), id);
        }
        catch (Exception e) {
            document = null;
        }
        if (document == null) {
            return null;
        }
        Optional<OpenSearchSession> maybeSession = this.documentToSession(document);
        if (maybeSession.isEmpty()) {
            this.deleteById(this.getSessionIdFrom(document));
            return null;
        }
        OpenSearchSession session = maybeSession.get();
        if (this.shouldDeleteSession(session)) {
            this.deleteById(session.getId());
            return null;
        }
        return session;
    }

    public void deleteById(String id) {
        LOGGER.debug("Delete session {}", (Object)id);
        this.executeAsyncOpenSearchRequest(() -> this.retryOpenSearchClient.deleteDocument(this.tasklistWebSessionIndex.getFullQualifiedName(), id));
    }

    private byte[] serialize(Object object) {
        return (byte[])this.conversionService.convert(object, TypeDescriptor.valueOf(Object.class), TypeDescriptor.valueOf(byte[].class));
    }

    private Object deserialize(byte[] bytes) {
        return this.conversionService.convert((Object)bytes, TypeDescriptor.valueOf(byte[].class), TypeDescriptor.valueOf(Object.class));
    }

    private Map<String, Object> sessionToDocument(OpenSearchSession session) {
        HashMap attributes = new HashMap();
        session.getAttributeNames().forEach(name -> attributes.put(name, this.serialize(session.getAttribute((String)name))));
        return Map.of("id", session.getId(), "creationTime", session.getCreationTime().toEpochMilli(), "lastAccessedTime", session.getLastAccessedTime().toEpochMilli(), "maxInactiveIntervalInSeconds", session.getMaxInactiveInterval().getSeconds(), "attributes", attributes);
    }

    private String getSessionIdFrom(Map<String, Object> document) {
        return (String)document.get("id");
    }

    private Instant getInstantFor(Object object) {
        if (object == null) {
            return null;
        }
        if (object instanceof Long) {
            return Instant.ofEpochMilli((Long)object);
        }
        return null;
    }

    private Duration getDurationFor(Object object) {
        if (object == null) {
            return null;
        }
        if (object instanceof Integer) {
            return Duration.ofSeconds(((Integer)object).intValue());
        }
        return null;
    }

    private Optional<OpenSearchSession> documentToSession(Map<String, Object> document) {
        try {
            String sessionId = this.getSessionIdFrom(document);
            OpenSearchSession session = new OpenSearchSession(sessionId);
            session.setLastAccessedTime(this.getInstantFor(document.get("lastAccessedTime")));
            try {
                if (this.request != null && this.request.getHeader(POLLING_HEADER) != null && !this.request.getHeader(POLLING_HEADER).equals(true)) {
                    session.setPolling(true);
                }
            }
            catch (Exception e) {
                LOGGER.debug("Expected Exception: is not possible to access request as currently this is not on a request context");
            }
            session.setCreationTime(this.getInstantFor(document.get("creationTime")));
            session.setMaxInactiveInterval(this.getDurationFor(document.get("maxInactiveIntervalInSeconds")));
            Object attributesObject = document.get("attributes");
            if (attributesObject != null && attributesObject.getClass().isInstance(new LinkedHashMap())) {
                Map attributes = (Map)document.get("attributes");
                attributes.keySet().forEach(name -> session.setAttribute((String)name, this.deserialize(Base64.getDecoder().decode((String)attributes.get(name)))));
            }
            return Optional.of(session);
        }
        catch (Exception e) {
            LOGGER.error("Could not restore session.", (Throwable)e);
            return Optional.empty();
        }
    }

    private void executeAsyncOpenSearchRequest(Runnable requestRunnable) {
        this.taskScheduler.execute(requestRunnable);
    }

    static class OpenSearchSession
    implements Session {
        private final MapSession delegate;
        private boolean changed;
        private boolean polling;

        public OpenSearchSession(String id) {
            this.delegate = new MapSession(id);
            this.polling = false;
        }

        boolean isChanged() {
            return this.changed;
        }

        void clearChangeFlag() {
            this.changed = false;
        }

        public String getId() {
            return this.delegate.getId();
        }

        public String changeSessionId() {
            String newId = this.delegate.changeSessionId();
            this.changed = true;
            return newId;
        }

        public OpenSearchSession setId(String id) {
            this.delegate.setId(id);
            return this;
        }

        public <T> T getAttribute(String attributeName) {
            return (T)this.delegate.getAttribute(attributeName);
        }

        public Set<String> getAttributeNames() {
            return this.delegate.getAttributeNames();
        }

        public void setAttribute(String attributeName, Object attributeValue) {
            this.delegate.setAttribute(attributeName, attributeValue);
            this.changed = true;
        }

        public void removeAttribute(String attributeName) {
            this.delegate.removeAttribute(attributeName);
            this.changed = true;
        }

        public Instant getCreationTime() {
            return this.delegate.getCreationTime();
        }

        public void setCreationTime(Instant creationTime) {
            this.delegate.setCreationTime(creationTime);
            this.changed = true;
        }

        public int hashCode() {
            return Objects.hash(this.getId());
        }

        public void setLastAccessedTime(Instant lastAccessedTime) {
            if (!this.polling) {
                this.delegate.setLastAccessedTime(lastAccessedTime);
                this.changed = true;
            }
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            OpenSearchSession session = (OpenSearchSession)o;
            return Objects.equals(this.getId(), session.getId());
        }

        public Instant getLastAccessedTime() {
            return this.delegate.getLastAccessedTime();
        }

        public void setMaxInactiveInterval(Duration interval) {
            this.delegate.setMaxInactiveInterval(interval);
            this.changed = true;
        }

        public Duration getMaxInactiveInterval() {
            return this.delegate.getMaxInactiveInterval();
        }

        public String toString() {
            return String.format("OpenSearchSession: %s ", this.getId());
        }

        public boolean isExpired() {
            return this.delegate.isExpired();
        }

        public boolean containsAuthentication() {
            return this.getAuthentication() != null;
        }

        public boolean isAuthenticated() {
            Authentication authentication = this.getAuthentication();
            try {
                return authentication != null && authentication.isAuthenticated();
            }
            catch (InsufficientAuthenticationException ex) {
                return false;
            }
        }

        private Authentication getAuthentication() {
            SecurityContext securityContext = (SecurityContext)this.delegate.getAttribute("SPRING_SECURITY_CONTEXT");
            Authentication authentication = securityContext != null ? securityContext.getAuthentication() : null;
            return authentication;
        }

        public boolean isPolling() {
            return this.polling;
        }

        public OpenSearchSession setPolling(boolean polling) {
            this.polling = polling;
            return this;
        }
    }
}

