/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.tasklist.webapp.es.backup.os;

import io.camunda.tasklist.data.conditionals.OpenSearchCondition;
import io.camunda.tasklist.exceptions.TasklistElasticsearchConnectionException;
import io.camunda.tasklist.exceptions.TasklistRuntimeException;
import io.camunda.tasklist.property.TasklistProperties;
import io.camunda.tasklist.util.Either;
import io.camunda.tasklist.util.ThreadUtil;
import io.camunda.tasklist.webapp.es.backup.BackupManager;
import io.camunda.tasklist.webapp.es.backup.Metadata;
import io.camunda.tasklist.webapp.es.backup.os.GetCustomSnapshotResponse;
import io.camunda.tasklist.webapp.es.backup.os.response.SnapshotState;
import io.camunda.tasklist.webapp.management.dto.BackupStateDto;
import io.camunda.tasklist.webapp.management.dto.GetBackupStateResponseDetailDto;
import io.camunda.tasklist.webapp.management.dto.GetBackupStateResponseDto;
import io.camunda.tasklist.webapp.management.dto.TakeBackupRequestDto;
import io.camunda.tasklist.webapp.management.dto.TakeBackupResponseDto;
import io.camunda.tasklist.webapp.rest.exception.InvalidRequestException;
import io.camunda.tasklist.webapp.rest.exception.NotFoundApiException;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.opensearch.OpenSearchAsyncClient;
import org.opensearch.client.opensearch.OpenSearchClient;
import org.opensearch.client.opensearch._types.OpenSearchException;
import org.opensearch.client.opensearch.snapshot.CreateSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.DeleteSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.DeleteSnapshotResponse;
import org.opensearch.client.opensearch.snapshot.GetRepositoryRequest;
import org.opensearch.client.opensearch.snapshot.GetSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.SnapshotInfo;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.JsonEndpoint;
import org.opensearch.client.transport.OpenSearchTransport;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

@Component
@Configuration
@Conditional(value={OpenSearchCondition.class})
public class BackupManagerOpenSearch
extends BackupManager {
    public static final String SNAPSHOT_MISSING_EXCEPTION_TYPE = "snapshot_missing_exception";
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupManagerOpenSearch.class);
    private static final String REPOSITORY_MISSING_EXCEPTION_TYPE = "repository_missing_exception";
    private final Queue<CreateSnapshotRequest> requestsQueue = new ConcurrentLinkedQueue<CreateSnapshotRequest>();
    private final AtomicReference<CreateSnapshotRequest> currentRequest = new AtomicReference<Object>(null);
    @Autowired
    private TasklistProperties tasklistProperties;
    @Autowired
    @Qualifier(value="tasklistOsAsyncClient")
    private OpenSearchAsyncClient openSearchAsyncClient;
    @Autowired
    @Qualifier(value="tasklistOsClient")
    private OpenSearchClient openSearchClient;

    @Override
    public void deleteBackup(Long backupId) {
        this.validateRepositoryExists();
        String repositoryName = this.getRepositoryName();
        int count = this.getIndexPatternsOrdered().length;
        for (int index = 0; index < count; ++index) {
            String snapshotName = new Metadata().setVersion("*").setPartCount(count).setPartNo(index + 1).setBackupId(backupId).buildSnapshotName();
            DeleteSnapshotRequest request = DeleteSnapshotRequest.of(dsr -> dsr.repository(repositoryName).snapshot(snapshotName));
            try {
                this.openSearchAsyncClient.snapshot().delete(request).whenComplete(BackupManagerOpenSearch::handleSnapshotDeletion);
                continue;
            }
            catch (IOException | OpenSearchException e) {
                LOGGER.error("Exception occurred while deleting the snapshot: " + e.getMessage(), e);
                throw new TasklistRuntimeException("Exception occurred while deleting the snapshot", e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public TakeBackupResponseDto takeBackup(TakeBackupRequestDto request) {
        this.validateRepositoryExists();
        this.validateNoDuplicateBackupId(request.getBackupId());
        if (this.requestsQueue.size() > 0) {
            throw new InvalidRequestException("Another backup is running at the moment");
        }
        Queue<CreateSnapshotRequest> queue = this.requestsQueue;
        synchronized (queue) {
            if (this.requestsQueue.size() > 0) {
                throw new InvalidRequestException("Another backup is running at the moment");
            }
            return this.scheduleSnapshots(request);
        }
    }

    @Override
    public GetBackupStateResponseDto getBackupState(Long backupId) {
        boolean isBackupInProgress = this.isBackupInProgress(backupId);
        List<SnapshotInfo> snapshots = this.findSnapshots(backupId);
        return this.getBackupResponse(backupId, snapshots, isBackupInProgress);
    }

    @Override
    public List<GetBackupStateResponseDto> getBackups(boolean verbose, String pattern) {
        Either<Throwable, String> validatedPattern = BackupManagerOpenSearch.validPattern(pattern);
        validatedPattern.ifLeft(ex -> {
            throw new InvalidRequestException(ex.getMessage(), (Throwable)ex);
        });
        GetSnapshotRequest snapshotStatusRequest = GetSnapshotRequest.of(gsr -> gsr.repository(this.getRepositoryName()).snapshot("camunda_tasklist_" + (String)validatedPattern.get(), new String[0]).verbose(Boolean.valueOf(verbose)));
        try {
            GetCustomSnapshotResponse response = this.getCustomSnapshotResponse(snapshotStatusRequest);
            List<SnapshotInfo> snapshots = response.snapshots();
            if (verbose) {
                snapshots = snapshots.stream().sorted(Comparator.comparing(SnapshotInfo::startTimeInMillis).reversed()).toList();
            }
            LinkedHashMap groupedSnapshotInfos = snapshots.stream().collect(Collectors.groupingBy(si -> {
                Metadata metadata = BackupManagerOpenSearch.getMetadata(si.metadata(), si.snapshot());
                Long backupId = metadata.getBackupId();
                if (backupId == null) {
                    backupId = Metadata.extractBackupIdFromSnapshotName(si.snapshot());
                }
                return backupId;
            }, LinkedHashMap::new, Collectors.toList()));
            return groupedSnapshotInfos.entrySet().stream().map(entry -> this.getBackupResponse((Long)entry.getKey(), (List)entry.getValue(), this.isBackupInProgress((Long)entry.getKey()))).collect(Collectors.toList());
        }
        catch (IOException ex2) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for snapshots. Repository name: [%s].", this.getRepositoryName());
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex2);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", this.tasklistProperties.getBackup().getRepositoryName());
                throw new TasklistRuntimeException(reason);
            }
            if (BackupManagerOpenSearch.isSnapshotMissingException(e)) {
                return new ArrayList<GetBackupStateResponseDto>();
            }
            String reason = String.format("Exception occurred when searching for backups: %s", e.getMessage());
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
    }

    private static void handleSnapshotDeletion(DeleteSnapshotResponse result, Throwable ex) {
        if (ex != null) {
            if (BackupManagerOpenSearch.isSnapshotMissingException(ex)) {
                LOGGER.warn("No snapshot found for snapshot deletion: " + ex.getMessage());
            } else {
                LOGGER.error("Exception occurred while deleting the snapshot: " + ex.getMessage(), ex);
            }
        } else {
            LOGGER.debug("Delete snapshot was acknowledged by OpenSearch node: " + result.acknowledged());
        }
    }

    void validateRepositoryExists() {
        String repositoryName = this.getRepositoryName();
        GetRepositoryRequest getRepositoryRequest = GetRepositoryRequest.of(grr -> grr.name(repositoryName, new String[0]));
        try {
            this.getRepository(getRepositoryRequest);
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while retrieving repository with name [%s].", repositoryName);
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", repositoryName);
                throw new TasklistRuntimeException(reason);
            }
            String reason = String.format("Exception occurred when validating existence of repository with name [%s].", repositoryName);
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
    }

    private void getRepository(GetRepositoryRequest getRepositoryRequest) throws IOException {
        SimpleEndpoint endpoint = ((SimpleEndpoint)GetRepositoryRequest._ENDPOINT).withResponseDeserializer(null);
        ((OpenSearchTransport)this.openSearchAsyncClient._transport()).performRequestAsync((Object)getRepositoryRequest, (Endpoint)endpoint, this.openSearchAsyncClient._transportOptions()).join();
    }

    private static boolean isSnapshotMissingException(Throwable e) {
        return e.getMessage() != null && e.getMessage().contains(SNAPSHOT_MISSING_EXCEPTION_TYPE);
    }

    private boolean isRepositoryMissingException(Exception e) {
        return e.getMessage() != null && e.getMessage().contains(REPOSITORY_MISSING_EXCEPTION_TYPE);
    }

    private void validateNoDuplicateBackupId(Long backupId) {
        GetCustomSnapshotResponse response;
        GetSnapshotRequest snapshotsStatusRequest = GetSnapshotRequest.of(gsr -> gsr.repository(this.getRepositoryName()).snapshot(Metadata.buildSnapshotNamePrefix(backupId) + "*", new String[0]));
        try {
            response = this.getCustomSnapshotResponse(snapshotsStatusRequest);
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for duplicate backup. Repository name: [%s].", this.getRepositoryName());
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (BackupManagerOpenSearch.isSnapshotMissingException(e)) {
                return;
            }
            String reason = String.format("Exception occurred when validating whether backup with ID [%s] already exists.", backupId);
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
        if (!response.snapshots().isEmpty()) {
            String reason = String.format("A backup with ID [%s] already exists. Found snapshots: [%s]", backupId, response.snapshots().stream().map(this::getSnapshotId).collect(Collectors.joining(", ")));
            throw new InvalidRequestException(reason);
        }
    }

    private TakeBackupResponseDto scheduleSnapshots(TakeBackupRequestDto request) {
        String repositoryName = this.getRepositoryName();
        int count = this.getIndexPatternsOrdered().length;
        ArrayList<String> snapshotNames = new ArrayList<String>();
        String version = this.getCurrentTasklistVersion();
        for (int index = 0; index < count; ++index) {
            String[] indexPattern = this.getIndexPatternsOrdered()[index];
            Metadata metadata = new Metadata().setVersion(version).setPartCount(count).setPartNo(index + 1).setBackupId(request.getBackupId());
            String snapshotName = metadata.buildSnapshotName();
            this.requestsQueue.offer(CreateSnapshotRequest.of(csr -> csr.repository(repositoryName).snapshot(snapshotName).indices(Arrays.stream(indexPattern).toList()).ignoreUnavailable(Boolean.valueOf(false)).includeGlobalState(Boolean.valueOf(true)).metadata(Map.of("backupId", JsonData.of((Object)metadata.getBackupId()), "version", JsonData.of((Object)metadata.getVersion()), "partNo", JsonData.of((Object)metadata.getPartNo()), "partCount", JsonData.of((Object)metadata.getPartCount()))).featureStates("none", new String[0]).waitForCompletion(Boolean.valueOf(true))));
            LOGGER.debug("Snapshot scheduled: " + snapshotName);
            snapshotNames.add(snapshotName);
        }
        this.scheduleNextSnapshot();
        return new TakeBackupResponseDto().setScheduledSnapshots(snapshotNames);
    }

    private void scheduleNextSnapshot() {
        CreateSnapshotRequest nextRequest = this.requestsQueue.poll();
        this.currentRequest.set(nextRequest);
        if (nextRequest != null) {
            this.getTaskExecutor().submit(() -> this.executeSnapshotting(nextRequest));
            LOGGER.debug("Snapshot picked for execution: snapshot [{}:{}]", (Object)nextRequest.repository(), (Object)nextRequest.snapshot());
        }
    }

    protected void executeSnapshotting(CreateSnapshotRequest snapshotRequest) {
        try {
            ((CompletableFuture)this.openSearchAsyncClient.snapshot().create(snapshotRequest).whenComplete((response, ex) -> this.handleSnapshotResponse(response.snapshot()))).exceptionally(e -> {
                Long backupId = Metadata.extractBackupIdFromSnapshotName(snapshotRequest.snapshot());
                if (e.getCause() instanceof SocketTimeoutException) {
                    LOGGER.warn(String.format("Timeout while creating snapshot [%s] for backup id [%d]. Need to keep waiting with polling...", snapshotRequest.snapshot(), Metadata.extractBackupIdFromSnapshotName(snapshotRequest.snapshot())));
                    List<SnapshotInfo> snapshotInfos = this.findSnapshots(backupId);
                    Optional<SnapshotInfo> maybeCurrentSnapshot = snapshotInfos.stream().filter(x -> Objects.equals(x.snapshot(), snapshotRequest.snapshot())).findFirst();
                    if (maybeCurrentSnapshot.isEmpty()) {
                        LOGGER.error(String.format("Expected (but not found) snapshot [%s] for backupId [%d].", snapshotRequest.snapshot(), backupId));
                        this.onSnapshotFailure();
                    } else if (this.isSnapshotFinishedWithinTimeout(maybeCurrentSnapshot.get().snapshot())) {
                        this.scheduleNextSnapshot();
                    } else {
                        this.onSnapshotFailure();
                    }
                } else {
                    LOGGER.error(String.format("Exception while creating snapshot [%s] for backup id [%d].", snapshotRequest.snapshot(), backupId), e);
                    this.onSnapshotFailure();
                }
                return null;
            });
        }
        catch (IOException e2) {
            throw new TasklistRuntimeException((Throwable)e2);
        }
    }

    private void onSnapshotFailure() {
        this.currentRequest.set(null);
        this.requestsQueue.clear();
    }

    private void handleSnapshotResponse(SnapshotInfo snapshotInfo) {
        String snapshotState;
        String snapshotId = this.getSnapshotId(snapshotInfo);
        switch (snapshotState = Objects.requireNonNullElse(snapshotInfo.state(), "null")) {
            case "SUCCESS": {
                LOGGER.info("Snapshot done: {}", (Object)snapshotId);
                this.scheduleNextSnapshot();
                break;
            }
            case "FAILED": {
                LOGGER.error("Snapshot failed for {}, reason {}", (Object)snapshotId, (Object)snapshotInfo.reason());
                this.onSnapshotFailure();
                break;
            }
            default: {
                LOGGER.error("Unexpected snapshot state '{}' for {}", (Object)snapshotState, (Object)snapshotId);
                this.onSnapshotFailure();
            }
        }
    }

    private boolean isSnapshotFinished(SnapshotInfo snapshotInfo) {
        return Objects.requireNonNullElse(snapshotInfo.snapshot(), "null").equals(SnapshotState.SUCCESS.name());
    }

    private String getSnapshotId(SnapshotInfo snapshotInfo) {
        return String.format("%s/%s", snapshotInfo.snapshot(), snapshotInfo.uuid());
    }

    List<SnapshotInfo> findSnapshots(Long backupId) {
        GetSnapshotRequest snapshotStatusRequest = GetSnapshotRequest.of(gsr -> gsr.repository(this.getRepositoryName()).snapshot(Metadata.buildSnapshotNamePrefix(backupId) + "*", new String[0]));
        try {
            return this.getCustomSnapshotResponse(snapshotStatusRequest).snapshots();
        }
        catch (IOException ex) {
            String reason = String.format("Encountered an error connecting to Elasticsearch while searching for snapshots. Repository name: [%s].", this.getRepositoryName());
            throw new TasklistElasticsearchConnectionException(reason, (Throwable)ex);
        }
        catch (Exception e) {
            if (BackupManagerOpenSearch.isSnapshotMissingException(e)) {
                throw new NotFoundApiException(String.format("No backup with id [%s] found.", backupId), e);
            }
            if (this.isRepositoryMissingException(e)) {
                String reason = String.format("No repository with name [%s] could be found.", this.tasklistProperties.getBackup().getRepositoryName());
                throw new TasklistRuntimeException(reason);
            }
            String reason = String.format("Exception occurred when searching for backup with ID [%s].", backupId);
            throw new TasklistRuntimeException(reason, (Throwable)e);
        }
    }

    public GetCustomSnapshotResponse getCustomSnapshotResponse(GetSnapshotRequest request) throws IOException, OpenSearchException {
        JsonEndpoint endpoint = (JsonEndpoint)GetCustomSnapshotResponse.ENDPOINT;
        return (GetCustomSnapshotResponse)((OpenSearchTransport)this.openSearchClient._transport()).performRequest((Object)request, (Endpoint)endpoint, this.openSearchClient._transportOptions());
    }

    private GetBackupStateResponseDto getBackupResponse(Long backupId, List<SnapshotInfo> snapshots, boolean isBackupInProgress) {
        GetBackupStateResponseDto response = new GetBackupStateResponseDto(backupId);
        Map jsonDataMap = snapshots.get(0).metadata();
        Metadata metadata = BackupManagerOpenSearch.getMetadata(jsonDataMap, snapshots.get(0).snapshot());
        Integer expectedSnapshotsCount = metadata.getPartCount();
        if (snapshots.size() == expectedSnapshotsCount.intValue() && snapshots.stream().map(SnapshotInfo::state).allMatch("SUCCESS"::equals)) {
            response.setState(BackupStateDto.COMPLETED);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch(s -> "FAILED".equals(s) || "PARTIAL".equals(s))) {
            response.setState(BackupStateDto.FAILED);
        } else if (snapshots.stream().map(SnapshotInfo::state).anyMatch("INCOMPATIBLE"::equals)) {
            response.setState(BackupStateDto.INCOMPATIBLE);
        } else if (isBackupInProgress || snapshots.stream().map(SnapshotInfo::state).anyMatch("IN_PROGRESS"::equals)) {
            response.setState(BackupStateDto.IN_PROGRESS);
        } else if (snapshots.size() < expectedSnapshotsCount) {
            response.setState(this.isWithinGracePeriodForIncomplete(snapshots.getLast().endTimeInMillis()) ? BackupStateDto.IN_PROGRESS : BackupStateDto.INCOMPLETE);
        } else {
            response.setState(BackupStateDto.FAILED);
        }
        ArrayList<GetBackupStateResponseDetailDto> details = new ArrayList<GetBackupStateResponseDetailDto>();
        for (SnapshotInfo snapshot : snapshots) {
            GetBackupStateResponseDetailDto detail = new GetBackupStateResponseDetailDto();
            detail.setSnapshotName(snapshot.snapshot());
            if (detail.getStartTime() != null) {
                detail.setStartTime(OffsetDateTime.ofInstant(Instant.ofEpochMilli(Long.parseLong(Objects.requireNonNull(snapshot.startTimeInMillis()))), ZoneId.systemDefault()));
            }
            if (snapshot.failures() != null) {
                detail.setFailures((String[])snapshot.failures().stream().map(Object::toString).toArray(String[]::new));
            }
            detail.setState(snapshot.state());
            details.add(detail);
        }
        response.setDetails(details);
        if (response.getState().equals((Object)BackupStateDto.FAILED)) {
            String failureReason = null;
            String failedSnapshots = snapshots.stream().filter(s -> "FAILED".equals(s.state())).map(SnapshotInfo::snapshot).collect(Collectors.joining(", "));
            if (!failedSnapshots.isEmpty()) {
                failureReason = String.format("There were failures with the following snapshots: %s", failedSnapshots);
            } else {
                String partialSnapshot = snapshots.stream().filter(s -> "PARTIAL".equals(s.state())).map(SnapshotInfo::snapshot).collect(Collectors.joining(", "));
                if (!partialSnapshot.isEmpty()) {
                    failureReason = String.format("Some of the snapshots are partial: %s", partialSnapshot);
                } else if (snapshots.size() > expectedSnapshotsCount) {
                    failureReason = "More snapshots found than expected.";
                }
            }
            if (failureReason != null) {
                response.setFailureReason(failureReason);
            }
        }
        return response;
    }

    private boolean isWithinGracePeriodForIncomplete(String lastSnapshotFinishedTime) {
        if (lastSnapshotFinishedTime == null) {
            return false;
        }
        long incompleteCheckTimeoutInMilliseconds = this.tasklistProperties.getBackup().getIncompleteCheckTimeoutInSeconds() * 1000L;
        try {
            return Instant.now().toEpochMilli() - Long.valueOf(lastSnapshotFinishedTime) < incompleteCheckTimeoutInMilliseconds;
        }
        catch (Exception e) {
            LOGGER.warn("Couldn't check incomplete timeout for backup. Return incomplete check is timed out", (Throwable)e);
            return true;
        }
    }

    private static Metadata getMetadata(Map<String, JsonData> jsonDataMap, String snapshotName) {
        return Metadata.extractFromMetadataOrName(Metadata.fromOSJsonData(jsonDataMap), snapshotName);
    }

    @Bean(value={"tasklistBackupThreadPoolExecutor"})
    public ThreadPoolTaskExecutor getTaskExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(1);
        executor.setMaxPoolSize(1);
        executor.setThreadNamePrefix("backup_os_");
        executor.setQueueCapacity(6);
        executor.initialize();
        return executor;
    }

    protected boolean isSnapshotFinishedWithinTimeout(String snapshotName) {
        int count = 0;
        long startTime = System.currentTimeMillis();
        int snapshotTimeout = this.tasklistProperties.getBackup().getSnapshotTimeout();
        long backupId = Metadata.extractBackupIdFromSnapshotName(snapshotName);
        while (snapshotTimeout == 0 || System.currentTimeMillis() - startTime <= (long)(snapshotTimeout * 1000)) {
            List<SnapshotInfo> snapshotInfos = this.findSnapshots(backupId);
            SnapshotInfo currentSnapshot = snapshotInfos.stream().filter(x -> Objects.equals(x.snapshot(), snapshotName)).findFirst().orElse(null);
            if (currentSnapshot == null) {
                LOGGER.error(String.format("Expected (but not found) snapshot [%s] for backupId [%d].", snapshotName, backupId));
                return false;
            }
            if (currentSnapshot.state().equals(SnapshotState.IN_PROGRESS.name())) {
                ThreadUtil.sleepFor((long)100L);
                if (++count % 600 != 0) continue;
                LOGGER.info(String.format("Waiting for snapshot [%s] to finish.", snapshotName));
                continue;
            }
            return this.isSnapshotFinished(currentSnapshot);
        }
        LOGGER.error(String.format("Snapshot [%s] did not finish after configured timeout. Snapshot process won't continue.", snapshotName));
        return false;
    }

    public long sleepFor(long milliseconds) {
        try {
            Thread.sleep(milliseconds);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return milliseconds;
    }

    private boolean isBackupInProgress(Long backupId) {
        return Stream.concat(Stream.ofNullable(this.currentRequest.get()), this.requestsQueue.stream()).anyMatch(request -> backupId.equals(Metadata.fromOSJsonData(request.metadata()).getBackupId()));
    }
}

