/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.optimize.service.db.os.reader;

import io.camunda.optimize.dto.optimize.rest.SnapshotInfoDto;
import io.camunda.optimize.dto.optimize.rest.SnapshotState;
import io.camunda.optimize.service.db.os.OptimizeOpenSearchClient;
import io.camunda.optimize.service.db.reader.AbstractBackupReader;
import io.camunda.optimize.service.exceptions.OptimizeConfigurationException;
import io.camunda.optimize.service.exceptions.OptimizeOpenSearchConnectionException;
import io.camunda.optimize.service.exceptions.OptimizeRuntimeException;
import io.camunda.optimize.service.exceptions.OptimizeSnapshotRepositoryNotFoundException;
import io.camunda.optimize.service.util.SnapshotUtil;
import io.camunda.optimize.service.util.configuration.ConfigurationService;
import io.camunda.optimize.service.util.configuration.condition.OpenSearchCondition;
import java.io.IOException;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.opensearch.client.opensearch._types.OpenSearchException;
import org.opensearch.client.opensearch.snapshot.GetRepositoryRequest;
import org.opensearch.client.opensearch.snapshot.GetSnapshotRequest;
import org.opensearch.client.opensearch.snapshot.GetSnapshotResponse;
import org.opensearch.client.opensearch.snapshot.SnapshotInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Conditional;
import org.springframework.stereotype.Component;

@Component
@Conditional(value={OpenSearchCondition.class})
public class BackupReaderOS
extends AbstractBackupReader {
    private static final Logger LOG = LoggerFactory.getLogger(BackupReaderOS.class);
    private final ConfigurationService configurationService;
    private final OptimizeOpenSearchClient osClient;

    public BackupReaderOS(ConfigurationService configurationService, OptimizeOpenSearchClient osClient) {
        this.configurationService = configurationService;
        this.osClient = osClient;
    }

    @Override
    protected String getSnapshotRepositoryName() {
        return this.configurationService.getOpenSearchConfiguration().getSnapshotRepositoryName();
    }

    @Override
    protected void validateRepositoryExistsOrFail() {
        String repositoryName = this.getRepositoryName();
        if (StringUtils.isEmpty((CharSequence)repositoryName)) {
            String reason = "Cannot trigger backup because no Opensearch snapshot repository name found in Optimize configuration.";
            LOG.error("Cannot trigger backup because no Opensearch snapshot repository name found in Optimize configuration.");
            throw new OptimizeConfigurationException("Cannot trigger backup because no Opensearch snapshot repository name found in Optimize configuration.");
        }
        GetRepositoryRequest getRepositoriesRequest = GetRepositoryRequest.of(b -> b.name(repositoryName, new String[0]));
        try {
            this.osClient.verifyRepositoryExists(getRepositoriesRequest);
        }
        catch (OpenSearchException e) {
            if (StringUtils.contains((CharSequence)e.getMessage(), (CharSequence)"repository_missing_exception")) {
                String reason = String.format("No repository with name [%s] could be found.", repositoryName);
                LOG.error(reason, (Throwable)e);
                throw new OptimizeSnapshotRepositoryNotFoundException(reason, e);
            }
            String reason = String.format("Error while retrieving repository with name [%s] due to an OpenSearchException.", repositoryName);
            LOG.error(reason, (Throwable)e);
            throw new OptimizeRuntimeException(reason, (Throwable)e);
        }
        catch (IOException e) {
            String reason = String.format("Encountered an error connecting to OpenSearch while retrieving repository with name [%s].", repositoryName);
            LOG.error(reason, (Throwable)e);
            throw new OptimizeOpenSearchConnectionException(reason, e);
        }
    }

    @Override
    public Map<Long, List<SnapshotInfoDto>> getAllOptimizeSnapshotsByBackupId() {
        return this.getAllOptimizeSnapshots().stream().map(BackupReaderOS::toSnapshotInfoDto).collect(Collectors.groupingBy(snapshotInfo -> SnapshotUtil.getBackupIdFromSnapshotName(snapshotInfo.getSnapshotName())));
    }

    @Override
    public List<SnapshotInfoDto> getOptimizeSnapshotsForBackupId(Long backupId) {
        return this.getOptimizeSnapshots(SnapshotUtil.getAllWildcardedSnapshotNamesForBackupId(backupId)).stream().map(BackupReaderOS::toSnapshotInfoDto).toList();
    }

    private static SnapshotInfoDto toSnapshotInfoDto(SnapshotInfo snapshotInfo) {
        long startTimeMillis = Long.parseLong(snapshotInfo.startTimeInMillis());
        List<String> shardFailures = snapshotInfo.shards().failures().stream().map(Object::toString).toList();
        return new SnapshotInfoDto(snapshotInfo.snapshot(), SnapshotState.valueOf((String)snapshotInfo.state()), OffsetDateTime.ofInstant(Instant.ofEpochMilli(startTimeMillis), ZoneId.systemDefault()), shardFailures);
    }

    private List<SnapshotInfo> getAllOptimizeSnapshots() {
        return this.getOptimizeSnapshots(SnapshotUtil.getAllWildcardedSnapshotNamesForWildcardedBackupId());
    }

    private List<SnapshotInfo> getOptimizeSnapshots(String[] snapshots) {
        GetSnapshotResponse response;
        GetSnapshotRequest snapshotsStatusRequest = GetSnapshotRequest.of(b -> b.repository(this.getRepositoryName()).snapshot(Arrays.stream(snapshots).toList()));
        try {
            response = this.osClient.getSnapshots(snapshotsStatusRequest);
        }
        catch (OpenSearchException e) {
            if (StringUtils.contains((CharSequence)e.getMessage(), (CharSequence)"snapshot_missing_exception")) {
                return Collections.emptyList();
            }
            String reason = String.format("Could not retrieve snapshots with names [%s] due to an OpenSearchException.", String.join((CharSequence)", ", snapshots));
            LOG.error(reason);
            throw new OptimizeRuntimeException(reason, (Throwable)e);
        }
        catch (IOException e) {
            String reason = String.format("Encountered an error connecting to OpenSearch while retrieving snapshots with names [%s].", String.join((CharSequence)", ", snapshots));
            LOG.error(reason, (Throwable)e);
            throw new OptimizeOpenSearchConnectionException(reason, e);
        }
        return response.snapshots();
    }

    private String getRepositoryName() {
        return this.configurationService.getOpenSearchConfiguration().getSnapshotRepositoryName();
    }
}

