/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.optimize.service.importing;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.camunda.optimize.dto.optimize.ImportRequestDto;
import io.camunda.optimize.dto.optimize.ProcessDefinitionOptimizeDto;
import io.camunda.optimize.dto.optimize.ProcessInstanceDto;
import io.camunda.optimize.service.db.repository.ProcessInstanceRepository;
import io.camunda.optimize.service.db.writer.ProcessDefinitionWriter;
import io.camunda.optimize.service.db.writer.ProcessInstanceWriter;
import io.camunda.optimize.service.security.util.LocalDateUtil;
import io.camunda.optimize.service.util.configuration.ConfigurationService;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

@Component
public class CustomerOnboardingDataImportService {
    private static final String CUSTOMER_ONBOARDING_DEFINITION = "customer_onboarding_definition.json";
    private static final String PROCESSED_INSTANCES = "customer_onboarding_process_instances.json";
    private static final int BATCH_SIZE = 2000;
    private static final Logger LOG = LoggerFactory.getLogger(CustomerOnboardingDataImportService.class);
    private final ProcessDefinitionWriter processDefinitionWriter;
    private final ObjectMapper objectMapper;
    private final ConfigurationService configurationService;
    private final ProcessInstanceWriter processInstanceWriter;
    private final ProcessInstanceRepository processInstanceRepository;

    public CustomerOnboardingDataImportService(ProcessDefinitionWriter processDefinitionWriter, ObjectMapper objectMapper, ConfigurationService configurationService, ProcessInstanceWriter processInstanceWriter, ProcessInstanceRepository processInstanceRepository) {
        this.processDefinitionWriter = processDefinitionWriter;
        this.objectMapper = objectMapper;
        this.configurationService = configurationService;
        this.processInstanceWriter = processInstanceWriter;
        this.processInstanceRepository = processInstanceRepository;
    }

    @EventListener(value={ApplicationReadyEvent.class})
    public void importData() {
        this.importData(PROCESSED_INSTANCES, CUSTOMER_ONBOARDING_DEFINITION, 2000);
    }

    public void importData(String processInstances, String processDefinition, int batchSize) {
        if (this.configurationService.getCustomerOnboardingImport()) {
            LOG.info("C8 Customer onboarding data enabled, importing data");
            this.importCustomerOnboardingData(processDefinition, processInstances, batchSize);
        } else {
            LOG.info("C8 Customer onboarding data disabled, will not perform data import");
        }
    }

    private void importCustomerOnboardingData(String processDefinition, String pathToProcessInstances, int batchSize) {
        try (InputStream customerOnboardingDefinition = this.getClass().getClassLoader().getResourceAsStream(processDefinition);){
            if (customerOnboardingDefinition != null) {
                String result = new String(customerOnboardingDefinition.readAllBytes(), StandardCharsets.UTF_8);
                ProcessDefinitionOptimizeDto processDefinitionDto = (ProcessDefinitionOptimizeDto)this.objectMapper.readValue(result, ProcessDefinitionOptimizeDto.class);
                if (processDefinitionDto != null) {
                    Optional.ofNullable(processDefinitionDto.getKey()).ifPresentOrElse(key -> {
                        this.processDefinitionWriter.importProcessDefinitions(List.of(processDefinitionDto));
                        this.readProcessInstanceJson(pathToProcessInstances, batchSize);
                    }, () -> LOG.error("Process definition data is invalid. Please check your json file."));
                } else {
                    LOG.error("Could not extract process definition from file in path: customer_onboarding_definition.json");
                }
            } else {
                LOG.error("Process definition could not be loaded. Please validate your json file.");
            }
        }
        catch (IOException e) {
            LOG.error("Unable to add a process definition to the database", (Throwable)e);
        }
        LOG.info("Customer onboarding data import complete");
    }

    private void readProcessInstanceJson(String pathToProcessInstances, int batchSize) {
        ArrayList<ProcessInstanceDto> processInstanceDtos = new ArrayList<ProcessInstanceDto>();
        try (InputStream customerOnboardingProcessInstances = this.getClass().getClassLoader().getResourceAsStream(pathToProcessInstances);){
            if (customerOnboardingProcessInstances != null) {
                String result = new String(customerOnboardingProcessInstances.readAllBytes(), StandardCharsets.UTF_8);
                List rawProcessInstanceDtos = (List)this.objectMapper.readValue(result, (TypeReference)new TypeReference<List<ProcessInstanceDto>>(this){});
                for (ProcessInstanceDto processInstance : rawProcessInstanceDtos) {
                    if (processInstance != null) {
                        Optional<Long> processInstanceDuration = Optional.ofNullable(processInstance.getDuration());
                        if (processInstance.getProcessDefinitionKey() == null || !processInstanceDuration.isEmpty() && processInstanceDuration.get() < 0L) continue;
                        processInstanceDtos.add(processInstance);
                        continue;
                    }
                    LOG.error("Process instance not loaded correctly. Please check your json file.");
                }
                this.loadProcessInstancesToDatabase(processInstanceDtos, batchSize);
            } else {
                LOG.error("Could not load Camunda Customer Onboarding Demo process instances to input stream. Please validate the process instance json file.");
            }
        }
        catch (IOException e) {
            LOG.error("Could not parse Camunda Customer Onboarding Demo process instances file.", (Throwable)e);
        }
    }

    private void loadProcessInstancesToDatabase(List<ProcessInstanceDto> rawProcessInstanceDtos, int batchSize) {
        ArrayList<ProcessInstanceDto> processInstanceDtos = new ArrayList<ProcessInstanceDto>();
        Optional<OffsetDateTime> maxOfEndAndStartDate = rawProcessInstanceDtos.stream().flatMap(instance -> Stream.of(instance.getStartDate(), instance.getEndDate())).filter(Objects::nonNull).max(OffsetDateTime::compareTo);
        for (ProcessInstanceDto rawProcessInstance : rawProcessInstanceDtos) {
            if (!maxOfEndAndStartDate.isPresent()) continue;
            ProcessInstanceDto processInstanceDto = this.modifyProcessInstanceDates(rawProcessInstance, maxOfEndAndStartDate.get());
            processInstanceDtos.add(processInstanceDto);
            if (processInstanceDtos.size() % batchSize != 0) continue;
            this.insertProcessInstancesToDatabase(processInstanceDtos);
            processInstanceDtos.clear();
        }
        if (!processInstanceDtos.isEmpty()) {
            this.insertProcessInstancesToDatabase(processInstanceDtos);
        }
    }

    private void insertProcessInstancesToDatabase(List<ProcessInstanceDto> processInstanceDtos) {
        List<ProcessInstanceDto> completedProcessInstances = processInstanceDtos.stream().filter(processInstanceDto -> processInstanceDto.getEndDate() != null).collect(Collectors.toList());
        List<ProcessInstanceDto> runningProcessInstances = processInstanceDtos.stream().filter(processInstanceDto -> processInstanceDto.getEndDate() == null).collect(Collectors.toList());
        List<ImportRequestDto> completedProcessInstanceImports = this.processInstanceWriter.generateCompletedProcessInstanceImports(completedProcessInstances);
        this.processInstanceRepository.bulkImport("Completed process instances", completedProcessInstanceImports);
        List<ImportRequestDto> runningProcessInstanceImports = this.processInstanceWriter.generateRunningProcessInstanceImports(runningProcessInstances);
        if (!runningProcessInstanceImports.isEmpty()) {
            this.processInstanceRepository.bulkImport("Running process instances", runningProcessInstanceImports);
        }
    }

    private ProcessInstanceDto modifyProcessInstanceDates(ProcessInstanceDto processInstanceDto, OffsetDateTime maxOfEndAndStartDate) {
        OffsetDateTime now = LocalDateUtil.getCurrentDateTime();
        long offset = ChronoUnit.SECONDS.between(maxOfEndAndStartDate, now);
        Optional.ofNullable(processInstanceDto.getStartDate()).ifPresent(startDate -> processInstanceDto.setStartDate(startDate.plusSeconds(offset)));
        Optional.ofNullable(processInstanceDto.getEndDate()).ifPresent(endDate -> processInstanceDto.setEndDate(endDate.plusSeconds(offset)));
        processInstanceDto.getFlowNodeInstances().forEach(flowNodeInstanceDto -> {
            Optional.ofNullable(flowNodeInstanceDto.getStartDate()).ifPresent(startDate -> flowNodeInstanceDto.setStartDate(startDate.plusSeconds(offset)));
            Optional.ofNullable(flowNodeInstanceDto.getEndDate()).ifPresent(endDate -> flowNodeInstanceDto.setEndDate(endDate.plusSeconds(offset)));
        });
        return processInstanceDto;
    }
}

