/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.optimize.service.security;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import io.camunda.optimize.rest.exceptions.NotAuthorizedException;
import io.camunda.optimize.service.exceptions.OptimizeRuntimeException;
import io.camunda.optimize.service.security.AuthCookieService;
import io.camunda.optimize.service.security.TerminatedSessionService;
import io.camunda.optimize.service.security.util.LocalDateUtil;
import io.camunda.optimize.service.util.configuration.ConfigurationReloadable;
import io.camunda.optimize.service.util.configuration.ConfigurationService;
import io.camunda.optimize.service.util.configuration.security.AuthConfiguration;
import jakarta.servlet.http.HttpServletRequest;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.chrono.ChronoLocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.Optional;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

@Component
public class SessionService
implements ConfigurationReloadable {
    private static final String ISSUER = "Optimize-8.8.0-alpha5-rc1";
    private static final byte[] DEFAULT_SECRET_BYTES = new byte[64];
    private static final Logger LOG = LoggerFactory.getLogger(SessionService.class);
    private final TerminatedSessionService terminatedSessionService;
    private final ConfigurationService configurationService;
    private Algorithm hashingAlgorithm;
    private JWTVerifier jwtVerifier;

    public SessionService(TerminatedSessionService terminatedSessionService, ConfigurationService configurationService) {
        this.terminatedSessionService = terminatedSessionService;
        this.configurationService = configurationService;
        this.initJwtVerifier();
    }

    public String createAuthToken(String userId) {
        return this.generateAuthToken(UUID.randomUUID().toString(), userId);
    }

    public boolean isValidToken(String token) {
        return Optional.ofNullable(token).map(this::isValidAuthToken).orElse(false);
    }

    private boolean isValidAuthToken(String token) {
        boolean isValid = false;
        try {
            DecodedJWT decodedJWT = this.jwtVerifier.verify(token);
            isValid = this.isStillValid(decodedJWT);
        }
        catch (JWTVerificationException exception) {
            LOG.error("Error while validating authentication token. Invalid signature or claims!", (Throwable)exception);
        }
        return isValid;
    }

    public String getRequestUserOrFailNotAuthorized(HttpServletRequest request) {
        return (String)AuthCookieService.getAuthCookieToken(request).flatMap(AuthCookieService::getTokenSubject).orElseThrow(() -> new NotAuthorizedException("Could not extract request user!"));
    }

    public void reloadConfiguration(ApplicationContext context) {
        this.initJwtVerifier();
    }

    public ConfigurationService getConfigurationService() {
        return this.configurationService;
    }

    private boolean isStillValid(DecodedJWT decodedJWT) {
        boolean isValid = true;
        Optional<LocalDateTime> dynamicExpiresAtDate = this.getDynamicExpiresAtLocalDateTime(decodedJWT);
        if (dynamicExpiresAtDate.map(date -> LocalDateUtil.getCurrentLocalDateTime().isAfter((ChronoLocalDateTime<?>)date)).orElse(false).booleanValue()) {
            LOG.debug("Authentication token [{}] has expired at {}!", (Object)decodedJWT.getToken(), dynamicExpiresAtDate);
            isValid = false;
        }
        try {
            if (this.terminatedSessionService.isSessionTerminated(decodedJWT.getId())) {
                LOG.warn("Authentication token [{}] of already terminated session {} was used!", (Object)decodedJWT.getToken(), (Object)decodedJWT.getId());
                isValid = false;
            }
        }
        catch (OptimizeRuntimeException e) {
            LOG.warn("Failed checking if session {} is a terminated session, defaulting to handle it as not terminated", (Object)decodedJWT.getId(), (Object)e);
        }
        return isValid;
    }

    private Optional<LocalDateTime> getDynamicExpiresAtLocalDateTime(DecodedJWT decodedJWT) {
        return Optional.ofNullable(decodedJWT.getIssuedAt()).map(Date::toInstant).map(instant -> instant.atZone(ZoneId.systemDefault()).toLocalDateTime()).map(localDateTime -> localDateTime.plus(this.getAuthConfiguration().getTokenLifeTimeMinutes(), ChronoUnit.MINUTES));
    }

    private String generateAuthToken(String sessionId, String userId) {
        Instant issuedAt = LocalDateUtil.getCurrentLocalDateTime().atZone(ZoneId.systemDefault()).toInstant();
        return JWT.create().withJWTId(sessionId).withIssuer(ISSUER).withSubject(userId).withIssuedAt(Date.from(issuedAt)).sign(this.hashingAlgorithm);
    }

    private void initJwtVerifier() {
        byte[] secretBytes = this.getAuthConfiguration().getTokenSecret().map(secretString -> secretString.getBytes(StandardCharsets.UTF_8)).orElse(DEFAULT_SECRET_BYTES);
        this.hashingAlgorithm = Algorithm.HMAC256((byte[])secretBytes);
        this.jwtVerifier = JWT.require((Algorithm)this.hashingAlgorithm).withIssuer(ISSUER).ignoreIssuedAt().build();
    }

    private AuthConfiguration getAuthConfiguration() {
        return this.configurationService.getAuthConfiguration();
    }

    static {
        new SecureRandom().nextBytes(DEFAULT_SECRET_BYTES);
    }
}

