/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.optimize.service.security;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.auth0.jwt.interfaces.Payload;
import io.camunda.optimize.service.util.configuration.ConfigurationService;
import io.camunda.optimize.service.util.configuration.security.AuthConfiguration;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.oauth2.core.OAuth2AccessToken;
import org.springframework.stereotype.Component;

@Component
public class AuthCookieService {
    private static final Logger LOG = LoggerFactory.getLogger(AuthCookieService.class);
    private final ConfigurationService configurationService;

    public AuthCookieService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public Cookie createDeleteOptimizeAuthCookie() {
        LOG.trace("Deleting Optimize authentication cookie.");
        return this.createDeleteCookie("X-Optimize-Authorization", "", "https");
    }

    public Cookie createDeleteOptimizeAuthNewCookie(boolean secure) {
        LOG.trace("Deleting Optimize authentication cookie.");
        Cookie cookie = new Cookie("X-Optimize-Authorization", "");
        cookie.setPath(this.getCookiePath());
        cookie.setDomain(null);
        cookie.setMaxAge(0);
        cookie.setSecure(secure);
        cookie.setHttpOnly(true);
        return cookie;
    }

    public Cookie createDeleteOptimizeRefreshCookie() {
        LOG.trace("Deleting Optimize refresh cookie.");
        return this.createDeleteCookie("X-Optimize-Refresh-Token", "", "https");
    }

    public Cookie createDeleteOptimizeRefreshNewCookie(boolean secure) {
        LOG.trace("Deleting Optimize refresh cookie.");
        Cookie cookie = new Cookie("X-Optimize-Refresh-Token", "");
        cookie.setPath(this.getCookiePath());
        cookie.setDomain(null);
        cookie.setMaxAge(0);
        cookie.setSecure(secure);
        cookie.setHttpOnly(true);
        return cookie;
    }

    public Optional<Instant> getOptimizeAuthCookieTokenExpiryDate(String optimizeAuthCookieToken) {
        return AuthCookieService.getTokenIssuedAt(optimizeAuthCookieToken).map(Date::toInstant).map(issuedAt -> issuedAt.plus((long)this.getAuthConfiguration().getTokenLifeTimeMinutes(), ChronoUnit.MINUTES));
    }

    public Cookie createDeleteCookie(String cookieName, String cookieValue, String requestScheme) {
        return this.createCookie(cookieName, cookieValue, Instant.now(), requestScheme, true);
    }

    public Cookie createOptimizeAuthCookie(String cookieValue, Instant expiresAt, String requestScheme) {
        return this.createCookie("X-Optimize-Authorization", cookieValue, expiresAt, requestScheme, false);
    }

    public Cookie createCookie(String cookieName, String cookieValue, Instant expiresAt, String requestScheme) {
        return this.createCookie(cookieName, cookieValue, expiresAt, requestScheme, false);
    }

    public List<Cookie> createOptimizeServiceTokenCookies(OAuth2AccessToken accessToken, Instant expiresAt, String requestScheme) {
        LOG.trace("Creating Optimize service token cookie(s).");
        String tokenValue = accessToken.getTokenValue();
        int maxCookieLength = this.configurationService.getAuthConfiguration().getCookieConfiguration().getMaxSize();
        int numberOfCookies = (int)Math.ceil((double)tokenValue.length() / (double)maxCookieLength);
        ArrayList<Cookie> cookies = new ArrayList<Cookie>();
        for (int i = 0; i < numberOfCookies; ++i) {
            cookies.add(this.createCookie(AuthCookieService.getServiceCookieNameWithSuffix(i), i == numberOfCookies - 1 ? tokenValue.substring(i * maxCookieLength) : tokenValue.substring(i * maxCookieLength, i * maxCookieLength + maxCookieLength), expiresAt, requestScheme));
        }
        return cookies;
    }

    public Cookie createCookie(String cookieName, String cookieValue, Date expiresAt, String requestScheme) {
        String cookiePath = this.getCookiePath();
        int maxAge = (int)(expiresAt.toInstant().toEpochMilli() - System.currentTimeMillis());
        Cookie cookie = new Cookie(cookieName, cookieValue);
        cookie.setPath(cookiePath);
        cookie.setDomain(null);
        cookie.setMaxAge(maxAge);
        cookie.setSecure(this.isSecureScheme(requestScheme));
        cookie.setHttpOnly(true);
        if (this.getAuthConfiguration().getCookieConfiguration().isSameSiteFlagEnabled()) {
            cookie.setAttribute("SameSite", "Strict");
        }
        return cookie;
    }

    public static Optional<String> getAuthCookieToken(HttpServletRequest servletRequest) {
        return Optional.ofNullable((String)servletRequest.getAttribute("X-Optimize-Authorization")).or(() -> AuthCookieService.extractAuthorizationValueFromCookies(servletRequest)).or(() -> AuthCookieService.extractAuthorizationValueFromCookieHeader(servletRequest)).flatMap(AuthCookieService::extractTokenFromAuthorizationValue);
    }

    public static Optional<String> getServiceAccessToken(HttpServletRequest servletRequest) {
        boolean serviceTokenExtracted = false;
        int serviceTokenSuffixToExtract = 0;
        StringBuilder serviceAccessToken = new StringBuilder();
        while (!serviceTokenExtracted) {
            String serviceCookieName = AuthCookieService.getServiceCookieNameWithSuffix(serviceTokenSuffixToExtract);
            String authorizationValue = null;
            if (servletRequest.getCookies() != null) {
                for (Cookie cookie : servletRequest.getCookies()) {
                    if (!serviceCookieName.equals(cookie.getName())) continue;
                    authorizationValue = cookie.getValue();
                }
            }
            if (authorizationValue != null) {
                serviceAccessToken.append(authorizationValue);
                ++serviceTokenSuffixToExtract;
                continue;
            }
            serviceTokenExtracted = true;
        }
        if (serviceAccessToken.length() != 0) {
            return Optional.of(serviceAccessToken.toString().trim());
        }
        return Optional.empty();
    }

    public static Optional<String> getTokenSubject(String token) {
        return AuthCookieService.getTokenAttribute(token, Payload::getSubject);
    }

    public static String createOptimizeAuthCookieValue(String tokenValue) {
        return "Bearer " + tokenValue;
    }

    private String getCookiePath() {
        return "/" + this.configurationService.getAuthConfiguration().getCloudAuthConfiguration().getClusterId();
    }

    private boolean isSecureScheme(String requestScheme) {
        return this.configurationService.getAuthConfiguration().getCookieConfiguration().resolveSecureFlagValue(requestScheme);
    }

    private AuthConfiguration getAuthConfiguration() {
        return this.configurationService.getAuthConfiguration();
    }

    private static Optional<Date> getTokenIssuedAt(String token) {
        return AuthCookieService.getTokenAttribute(token, Payload::getIssuedAt);
    }

    private static <T> Optional<T> getTokenAttribute(String token, Function<DecodedJWT, T> getTokenAttributeFunction) {
        try {
            DecodedJWT decoded = JWT.decode((String)token);
            return Optional.of(getTokenAttributeFunction.apply(decoded));
        }
        catch (Exception e) {
            LOG.debug("Could not decode security token to extract attribute!", (Throwable)e);
            return Optional.empty();
        }
    }

    private static String getServiceCookieNameWithSuffix(int suffix) {
        return "X-Optimize-Service-Token_" + suffix;
    }

    private static Optional<String> extractTokenFromAuthorizationValue(String authCookieValue) {
        if (authCookieValue != null && authCookieValue.startsWith("Bearer ")) {
            return Optional.of(authCookieValue.substring("Bearer ".length()).trim());
        }
        return Optional.ofNullable(authCookieValue);
    }

    private Cookie createCookie(String cookieName, String cookieValue, Instant expiresAt, String requestScheme, boolean isDelete) {
        String cookiePath = this.getCookiePath();
        Cookie cookie = new Cookie(cookieName, cookieValue);
        cookie.setPath(cookiePath);
        cookie.setHttpOnly(true);
        cookie.setSecure(this.isSecureScheme(requestScheme));
        if (this.getAuthConfiguration().getCookieConfiguration().isSameSiteFlagEnabled()) {
            cookie.setAttribute("SameSite", "Strict");
        }
        if (expiresAt == null) {
            cookie.setMaxAge(-1);
        } else {
            cookie.setMaxAge(isDelete ? 0 : (int)Duration.between(Instant.now(), expiresAt).toSeconds());
        }
        return cookie;
    }

    private static Optional<String> extractAuthorizationValueFromCookies(HttpServletRequest servletRequest) {
        Cookie[] cookies = servletRequest.getCookies();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (!"X-Optimize-Authorization".equals(cookie.getName())) continue;
                return Optional.of(cookie.getValue());
            }
        }
        return Optional.empty();
    }

    private static Optional<String> extractAuthorizationValueFromCookieHeader(HttpServletRequest servletRequest) {
        String cookieHeader = servletRequest.getHeader("Cookie");
        if (cookieHeader != null) {
            String[] cookiePairs;
            for (String cookiePair : cookiePairs = cookieHeader.split(";")) {
                Pattern pattern = Pattern.compile("\\s*X-Optimize-Authorization\\s*=\\s*(.*)");
                Matcher matcher = pattern.matcher(cookiePair);
                if (!matcher.find()) continue;
                String value = matcher.group(1);
                return Optional.of(value.replace("\"", "").trim());
            }
        }
        return Optional.empty();
    }
}

