/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.optimize.service.alert;

import io.camunda.optimize.dto.optimize.alert.AlertNotificationDto;
import io.camunda.optimize.dto.optimize.query.alert.AlertDefinitionDto;
import io.camunda.optimize.service.alert.AlertNotificationService;
import io.camunda.optimize.service.exceptions.OptimizeRuntimeException;
import io.camunda.optimize.service.util.configuration.ConfigurationReloadable;
import io.camunda.optimize.service.util.configuration.ConfigurationService;
import io.camunda.optimize.service.util.configuration.ProxyConfiguration;
import io.camunda.optimize.service.util.configuration.WebhookConfiguration;
import jakarta.annotation.PreDestroy;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import org.apache.hc.core5.http.Method;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.routing.HttpRoutePlanner;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.DefaultProxyRoutePlanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

@Component
public class WebhookNotificationService
implements AlertNotificationService,
ConfigurationReloadable {
    private static final Logger LOG = LoggerFactory.getLogger(WebhookNotificationService.class);
    private final ConfigurationService configurationService;
    private Map<String, CloseableHttpClient> webhookClientsByWebhookName;

    public WebhookNotificationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
        this.webhookClientsByWebhookName = this.buildHttpClientMap();
    }

    @PreDestroy
    public void close() {
        this.webhookClientsByWebhookName.forEach((key, value) -> {
            try {
                value.close();
            }
            catch (IOException e) {
                LOG.error("Could not close Http client for webhook with name: {}. Exception: {}", key, (Object)e);
            }
        });
    }

    public void reloadConfiguration(ApplicationContext context) {
        this.close();
        this.webhookClientsByWebhookName = this.buildHttpClientMap();
    }

    @Override
    public void notify(AlertNotificationDto notification) {
        if (notification == null) {
            throw new OptimizeRuntimeException("Notification cannot be null");
        }
        AlertDefinitionDto alert = notification.getAlert();
        String destination = alert.getWebhook();
        if (StringUtils.isEmpty((CharSequence)destination)) {
            LOG.debug("No webhook configured for alert [id: {}, name: {}], no action performed.", (Object)alert.getId(), (Object)alert.getName());
            return;
        }
        Map webhookConfigurationMap = this.configurationService.getConfiguredWebhooks();
        if (!webhookConfigurationMap.containsKey(destination)) {
            LOG.error("Could not send webhook notification as the configuration for webhook with name {} no longer exists in the configuration file.", (Object)destination);
            return;
        }
        WebhookConfiguration webhookConfiguration = (WebhookConfiguration)webhookConfigurationMap.get(destination);
        LOG.info("Sending webhook notification for alert [id: {}, name: {}] to webhook: '{}'.", new Object[]{alert.getId(), alert.getName(), destination});
        this.sendWebhookRequest(notification, webhookConfiguration, destination);
    }

    @Override
    public String getNotificationDescription() {
        return "webhook notification";
    }

    private void sendWebhookRequest(AlertNotificationDto notification, WebhookConfiguration webhook, String webhookName) {
        HttpEntityEnclosingRequestBase request = this.buildRequestFromMethod(webhook);
        request.setEntity((HttpEntity)new StringEntity(this.composePayload(notification, webhook), this.resolveContentTypeFromHeaders(webhook)));
        if (webhook.getHeaders() != null) {
            request.setHeader("Content-Type", "application/json");
            for (Map.Entry headerEntry : webhook.getHeaders().entrySet()) {
                request.setHeader((String)headerEntry.getKey(), (String)headerEntry.getValue());
            }
        }
        try (CloseableHttpResponse response = this.webhookClientsByWebhookName.get(webhookName).execute((HttpUriRequest)request);){
            HttpStatus status = HttpStatus.resolve((int)response.getStatusLine().getStatusCode());
            if (!status.is2xxSuccessful()) {
                LOG.error("Unexpected response when sending webhook notification: " + String.valueOf(status));
            }
        }
        catch (IOException e) {
            throw new OptimizeRuntimeException("There was a problem when sending webhook notification.", (Throwable)e);
        }
    }

    private HttpEntityEnclosingRequestBase buildRequestFromMethod(WebhookConfiguration webhook) {
        String httpMethod = webhook.getHttpMethod();
        return switch (Method.normalizedValueOf((String)httpMethod)) {
            case Method.POST -> new HttpPost(webhook.getUrl());
            case Method.PATCH -> new HttpPatch(webhook.getUrl());
            case Method.PUT -> new HttpPut(webhook.getUrl());
            default -> throw new OptimizeRuntimeException("Http method not possible for webhook notifications: " + httpMethod);
        };
    }

    private String composePayload(AlertNotificationDto notification, WebhookConfiguration webhook) {
        String payloadString = webhook.getDefaultPayload();
        for (WebhookConfiguration.Placeholder placeholder : WebhookConfiguration.Placeholder.values()) {
            String value = placeholder.extractValue(notification);
            payloadString = payloadString.replace(placeholder.getPlaceholderString(), value.replace("\n", "\\n"));
        }
        return payloadString;
    }

    private ContentType resolveContentTypeFromHeaders(WebhookConfiguration webhook) {
        if (webhook.getHeaders() == null || !webhook.getHeaders().containsKey("Content-Type")) {
            return ContentType.APPLICATION_JSON;
        }
        return ContentType.create((String)((String)webhook.getHeaders().get("Content-Type")));
    }

    private CloseableHttpClient createClientForWebhook(WebhookConfiguration webhookConfig) {
        ProxyConfiguration proxyConfiguration = webhookConfig.getProxy();
        if (proxyConfiguration == null || !proxyConfiguration.isEnabled()) {
            return HttpClients.createDefault();
        }
        HttpHost proxy = new HttpHost(proxyConfiguration.getHost(), proxyConfiguration.getPort().intValue(), proxyConfiguration.isSslEnabled() ? "https" : "http");
        return HttpClients.custom().setRoutePlanner((HttpRoutePlanner)new DefaultProxyRoutePlanner(proxy)).build();
    }

    private Map<String, CloseableHttpClient> buildHttpClientMap() {
        HashMap<String, CloseableHttpClient> httpClientMap = new HashMap<String, CloseableHttpClient>();
        this.configurationService.getConfiguredWebhooks().entrySet().forEach(entry -> httpClientMap.put((String)entry.getKey(), this.createClientForWebhook((WebhookConfiguration)entry.getValue())));
        return httpClientMap;
    }
}

