/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.optimize.service.alert;

import io.camunda.optimize.dto.optimize.alert.AlertNotificationDto;
import io.camunda.optimize.dto.optimize.alert.AlertNotificationType;
import io.camunda.optimize.dto.optimize.query.alert.AlertDefinitionDto;
import io.camunda.optimize.dto.optimize.query.alert.AlertThresholdOperator;
import io.camunda.optimize.dto.optimize.query.report.ReportDefinitionDto;
import io.camunda.optimize.dto.optimize.query.report.SingleReportEvaluationResult;
import io.camunda.optimize.dto.optimize.query.report.single.ViewProperty;
import io.camunda.optimize.dto.optimize.query.report.single.process.ProcessReportDataDto;
import io.camunda.optimize.service.alert.AlertJobResult;
import io.camunda.optimize.service.alert.AlertNotificationService;
import io.camunda.optimize.service.db.reader.AlertReader;
import io.camunda.optimize.service.db.reader.ReportReader;
import io.camunda.optimize.service.db.report.PlainReportEvaluationHandler;
import io.camunda.optimize.service.db.report.ReportEvaluationInfo;
import io.camunda.optimize.service.db.writer.AlertWriter;
import io.camunda.optimize.service.exceptions.OptimizeRuntimeException;
import io.camunda.optimize.service.util.DurationFormatterUtil;
import io.camunda.optimize.service.util.configuration.ConfigurationService;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;

@Scope(value="prototype")
public class AlertJob
implements Job {
    private static final String HTTP_PREFIX = "http://";
    private static final String HTTPS_PREFIX = "https://";
    private static final Logger LOG = LoggerFactory.getLogger(AlertJob.class);
    private final ConfigurationService configurationService;
    private final List<AlertNotificationService> notificationServices;
    private final AlertReader alertReader;
    private final ReportReader reportReader;
    private final AlertWriter alertWriter;
    private final PlainReportEvaluationHandler reportEvaluator;

    public AlertJob(ConfigurationService configurationService, List<AlertNotificationService> notificationServices, AlertReader alertReader, ReportReader reportReader, AlertWriter alertWriter, PlainReportEvaluationHandler reportEvaluator) {
        this.configurationService = configurationService;
        this.notificationServices = notificationServices;
        this.alertReader = alertReader;
        this.reportReader = reportReader;
        this.alertWriter = alertWriter;
        this.reportEvaluator = reportEvaluator;
    }

    public void execute(JobExecutionContext jobExecutionContext) {
        JobDataMap dataMap = jobExecutionContext.getJobDetail().getJobDataMap();
        String alertId = dataMap.getString("alertId");
        LOG.debug("Executing status check for alert [{}]", (Object)alertId);
        AlertDefinitionDto alert = this.alertReader.getAlert(alertId).orElseThrow(() -> new OptimizeRuntimeException("Alert does not exist!"));
        try {
            ReportDefinitionDto reportDefinition = this.reportReader.getReport(alert.getReportId()).orElseThrow(() -> new OptimizeRuntimeException("Was not able to retrieve report with id " + alert.getReportId() + "] from Elasticsearch. Report does not exist."));
            ReportEvaluationInfo reportEvaluationInfo = ReportEvaluationInfo.builder(reportDefinition).build();
            SingleReportEvaluationResult evaluationResult = (SingleReportEvaluationResult)this.reportEvaluator.evaluateReport(reportEvaluationInfo).getEvaluationResult();
            Double reportResult = (Double)evaluationResult.getFirstCommandResult().getFirstMeasureData();
            if (this.thresholdExceeded(alert, reportResult)) {
                jobExecutionContext.setResult((Object)this.handleAlertTriggered(jobExecutionContext.getJobDetail().getKey(), alert, reportDefinition, reportResult));
            } else if (alert.isTriggered()) {
                jobExecutionContext.setResult((Object)this.handleAlertResolved(alert, reportDefinition, reportResult));
            }
        }
        catch (Exception e) {
            LOG.error("Error while processing alert [{}] for report [{}]", new Object[]{alertId, alert.getReportId(), e});
        }
    }

    private AlertJobResult handleAlertTriggered(JobKey key, AlertDefinitionDto alert, ReportDefinitionDto<?> reportDefinition, Double currentValue) {
        boolean isReminder = this.isReminder(key);
        boolean haveToSendReminder = isReminder && alert.isTriggered();
        boolean haveToNotify = haveToSendReminder || !alert.isTriggered();
        alert.setTriggered(true);
        AlertJobResult alertJobResult = new AlertJobResult(alert);
        if (haveToNotify) {
            this.alertWriter.writeAlertTriggeredStatus(true, alert.getId());
            this.fanoutNotification(this.createAlertNotification(alert, isReminder ? AlertNotificationType.REMINDER : AlertNotificationType.NEW, reportDefinition, currentValue));
            alertJobResult.setStatusChanged(true);
        }
        return alertJobResult;
    }

    private AlertJobResult handleAlertResolved(AlertDefinitionDto alert, ReportDefinitionDto<?> reportDefinition, Double reportResult) {
        alert.setTriggered(false);
        AlertJobResult alertJobResult = new AlertJobResult(alert);
        alertJobResult.setStatusChanged(true);
        this.alertWriter.writeAlertTriggeredStatus(false, alert.getId());
        if (alert.isFixNotification()) {
            this.fanoutNotification(this.createResolvedAlertNotification(alert, reportDefinition, reportResult));
        }
        return alertJobResult;
    }

    private AlertNotificationDto createAlertNotification(AlertDefinitionDto alert, AlertNotificationType notificationType, ReportDefinitionDto<?> reportDefinition, Double currentValue) {
        return new AlertNotificationDto(alert, currentValue, notificationType, this.composeAlertText(alert, reportDefinition, notificationType, currentValue), this.createReportViewLink(alert, notificationType));
    }

    private AlertNotificationDto createResolvedAlertNotification(AlertDefinitionDto alert, ReportDefinitionDto<?> reportDefinition, Double currentValue) {
        return new AlertNotificationDto(alert, currentValue, AlertNotificationType.RESOLVED, this.composeFixText(alert, reportDefinition, currentValue), this.createReportViewLink(alert, AlertNotificationType.RESOLVED));
    }

    private void fanoutNotification(AlertNotificationDto notification) {
        for (AlertNotificationService notificationService : this.notificationServices) {
            try {
                notificationService.notify(notification);
            }
            catch (Exception e) {
                LOG.error("Exception thrown while trying to send notification: {}", (Object)notificationService.getNotificationDescription(), (Object)e);
            }
        }
    }

    private boolean isReminder(JobKey key) {
        return key.getName().toLowerCase(Locale.ENGLISH).contains("reminder");
    }

    private String composeAlertText(AlertDefinitionDto alert, ReportDefinitionDto<?> reportDefinition, AlertNotificationType notificationType, Double result) {
        String statusText = AlertThresholdOperator.LESS.equals((Object)alert.getThresholdOperator()) ? "is not reached" : "was exceeded";
        return this.composeAlertText(alert, reportDefinition, result, notificationType, statusText);
    }

    private String composeFixText(AlertDefinitionDto alert, ReportDefinitionDto<?> reportDefinition, Double result) {
        String statusText = AlertThresholdOperator.LESS.equals((Object)alert.getThresholdOperator()) ? "has been reached" : "is not exceeded anymore";
        return this.composeAlertText(alert, reportDefinition, result, AlertNotificationType.RESOLVED, statusText);
    }

    private String composeAlertText(AlertDefinitionDto alert, ReportDefinitionDto<?> reportDefinition, Double result, AlertNotificationType notificationType, String statusText) {
        return this.configurationService.getNotificationEmailCompanyBranding() + " Optimize - Report Status\nAlert name: " + alert.getName() + "\nReport name: " + reportDefinition.getName() + "\nStatus: Given threshold [" + this.formatValueToHumanReadableString(alert.getThreshold(), reportDefinition) + "] " + statusText + ". Current value: " + this.formatValueToHumanReadableString(result, reportDefinition) + ". Please check your Optimize report for more information! \n" + this.createReportViewLink(alert, notificationType);
    }

    private boolean thresholdExceeded(AlertDefinitionDto alert, Double result) {
        boolean exceeded = false;
        if (result != null) {
            if (AlertThresholdOperator.GREATER.equals((Object)alert.getThresholdOperator())) {
                exceeded = result > alert.getThreshold();
            } else if (AlertThresholdOperator.LESS.equals((Object)alert.getThresholdOperator())) {
                exceeded = result < alert.getThreshold();
            }
        }
        return exceeded;
    }

    private String formatValueToHumanReadableString(Double value, ReportDefinitionDto<?> reportDefinition) {
        return this.isDurationReport(reportDefinition) ? this.durationInMsToReadableFormat(value) : String.valueOf(value);
    }

    private boolean isDurationReport(ReportDefinitionDto<?> reportDefinition) {
        if (reportDefinition.getData() instanceof ProcessReportDataDto) {
            ProcessReportDataDto data = (ProcessReportDataDto)reportDefinition.getData();
            return data.getView().getFirstProperty().equals((Object)ViewProperty.DURATION);
        }
        return false;
    }

    private String durationInMsToReadableFormat(Double durationInMsAsDouble) {
        if (durationInMsAsDouble == null) {
            return String.valueOf(durationInMsAsDouble);
        }
        long durationInMs = durationInMsAsDouble.longValue();
        return DurationFormatterUtil.formatMilliSecondsToReadableDurationString(durationInMs);
    }

    private String createReportViewLink(AlertDefinitionDto alert, AlertNotificationType notificationType) {
        Optional containerAccessUrl = this.configurationService.getContainerAccessUrl();
        if (containerAccessUrl.isPresent()) {
            return (String)containerAccessUrl.get() + this.createReportViewLinkPath(alert, notificationType);
        }
        Optional containerHttpPort = this.configurationService.getContainerHttpPort();
        String httpPrefix = containerHttpPort.map(p -> HTTP_PREFIX).orElse(HTTPS_PREFIX);
        Integer port = containerHttpPort.orElse(this.configurationService.getContainerHttpsPort());
        return httpPrefix + this.configurationService.getContainerHost() + ":" + port + this.configurationService.getContextPath().orElse("") + this.createReportViewLinkPath(alert, notificationType);
    }

    private String createReportViewLinkPath(AlertDefinitionDto alert, AlertNotificationType notificationType) {
        ReportDefinitionDto reportDefinition = this.reportReader.getReport(alert.getReportId()).orElseThrow(() -> new OptimizeRuntimeException("Was not able to retrieve report with id [" + alert.getReportId() + "] from Elasticsearch. Report does not exist."));
        String collectionId = reportDefinition.getCollectionId();
        if (collectionId != null) {
            return String.format("/#/collection/%s/report/%s?utm_source=%s", collectionId, alert.getReportId(), notificationType.getUtmSource());
        }
        return String.format("/#/report/%s?utm_source=%s", alert.getReportId(), notificationType.getUtmSource());
    }
}

