/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.engine.processing.streamprocessor;

import io.camunda.zeebe.db.TransactionContext;
import io.camunda.zeebe.db.ZeebeDbTransaction;
import io.camunda.zeebe.engine.metrics.StreamProcessorMetrics;
import io.camunda.zeebe.engine.processing.streamprocessor.EventFilter;
import io.camunda.zeebe.engine.processing.streamprocessor.MetadataEventFilter;
import io.camunda.zeebe.engine.processing.streamprocessor.MetadataFilter;
import io.camunda.zeebe.engine.processing.streamprocessor.ProcessingContext;
import io.camunda.zeebe.engine.processing.streamprocessor.RecordProcessorMap;
import io.camunda.zeebe.engine.processing.streamprocessor.RecordProtocolVersionFilter;
import io.camunda.zeebe.engine.processing.streamprocessor.RecordValues;
import io.camunda.zeebe.engine.processing.streamprocessor.StreamProcessorListener;
import io.camunda.zeebe.engine.processing.streamprocessor.TypedEventImpl;
import io.camunda.zeebe.engine.processing.streamprocessor.TypedRecord;
import io.camunda.zeebe.engine.processing.streamprocessor.TypedRecordProcessor;
import io.camunda.zeebe.engine.processing.streamprocessor.sideeffect.SideEffectProducer;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.TypedResponseWriter;
import io.camunda.zeebe.engine.processing.streamprocessor.writers.TypedStreamWriter;
import io.camunda.zeebe.engine.state.mutable.MutableLastProcessedPositionState;
import io.camunda.zeebe.engine.state.mutable.MutableZeebeState;
import io.camunda.zeebe.logstreams.impl.Loggers;
import io.camunda.zeebe.logstreams.log.LogStream;
import io.camunda.zeebe.logstreams.log.LogStreamReader;
import io.camunda.zeebe.logstreams.log.LoggedEvent;
import io.camunda.zeebe.protocol.impl.record.RecordMetadata;
import io.camunda.zeebe.protocol.impl.record.UnifiedRecordValue;
import io.camunda.zeebe.protocol.impl.record.value.error.ErrorRecord;
import io.camunda.zeebe.protocol.record.RecordType;
import io.camunda.zeebe.protocol.record.RecordValue;
import io.camunda.zeebe.protocol.record.RejectionType;
import io.camunda.zeebe.protocol.record.intent.ErrorIntent;
import io.camunda.zeebe.protocol.record.intent.Intent;
import io.camunda.zeebe.util.buffer.BufferReader;
import io.camunda.zeebe.util.exception.RecoverableException;
import io.camunda.zeebe.util.retry.AbortableRetryStrategy;
import io.camunda.zeebe.util.retry.RecoverableRetryStrategy;
import io.camunda.zeebe.util.retry.RetryStrategy;
import io.camunda.zeebe.util.sched.ActorControl;
import io.camunda.zeebe.util.sched.clock.ActorClock;
import io.camunda.zeebe.util.sched.future.ActorFuture;
import io.prometheus.client.Histogram;
import java.time.Duration;
import java.util.function.BooleanSupplier;
import org.slf4j.Logger;

public final class ProcessingStateMachine {
    private static final Logger LOG = Loggers.PROCESSOR_LOGGER;
    private static final String ERROR_MESSAGE_WRITE_EVENT_ABORTED = "Expected to write one or more follow up events for event '{} {}' without errors, but exception was thrown.";
    private static final String ERROR_MESSAGE_ROLLBACK_ABORTED = "Expected to roll back the current transaction for event '{} {}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED = "Expected to execute side effects for event '{} {}' successfully, but exception was thrown.";
    private static final String ERROR_MESSAGE_UPDATE_STATE_FAILED = "Expected to successfully update state for event '{} {}', but caught an exception. Retry.";
    private static final String ERROR_MESSAGE_ON_EVENT_FAILED_SKIP_EVENT = "Expected to find event processor for event '{} {}', but caught an exception. Skip this event.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_SKIP_EVENT = "Expected to successfully process event '{} {}' with processor, but caught an exception. Skip this event.";
    private static final String ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING = "Expected to process event '{} {}' successfully on stream processor, but caught recoverable exception. Retry processing.";
    private static final String PROCESSING_ERROR_MESSAGE = "Expected to process event '%s' without errors, but exception occurred with message '%s'.";
    private static final String NOTIFY_PROCESSED_LISTENER_ERROR_MESSAGE = "Expected to invoke processed listener for record {} successfully, but exception was thrown.";
    private static final String NOTIFY_SKIPPED_LISTENER_ERROR_MESSAGE = "Expected to invoke skipped listener for record '{} {}' successfully, but exception was thrown.";
    private static final Duration PROCESSING_RETRY_DELAY = Duration.ofMillis(250L);
    private static final MetadataFilter PROCESSING_FILTER = recordMetadata -> recordMetadata.getRecordType() == RecordType.COMMAND;
    private final EventFilter eventFilter = new MetadataEventFilter(new RecordProtocolVersionFilter().and(PROCESSING_FILTER));
    private final MutableZeebeState zeebeState;
    private final MutableLastProcessedPositionState lastProcessedPositionState;
    private final RecordMetadata metadata = new RecordMetadata();
    private final TypedResponseWriter responseWriter;
    private final ActorControl actor;
    private final LogStream logStream;
    private final LogStreamReader logStreamReader;
    private final TypedStreamWriter logStreamWriter;
    private final TransactionContext transactionContext;
    private final RetryStrategy writeRetryStrategy;
    private final RetryStrategy sideEffectsRetryStrategy;
    private final RetryStrategy updateStateRetryStrategy;
    private final BooleanSupplier shouldProcessNext;
    private final BooleanSupplier abortCondition;
    private final ErrorRecord errorRecord = new ErrorRecord();
    private final RecordValues recordValues;
    private final RecordProcessorMap recordProcessorMap;
    private final TypedEventImpl typedEvent;
    private final StreamProcessorMetrics metrics;
    private final StreamProcessorListener streamProcessorListener;
    private SideEffectProducer sideEffectProducer;
    private LoggedEvent currentEvent;
    private TypedRecordProcessor<?> currentProcessor;
    private ZeebeDbTransaction zeebeDbTransaction;
    private long writtenEventPosition = -1L;
    private long lastSuccessfulProcessedEventPosition = -1L;
    private long lastWrittenEventPosition = -1L;
    private volatile boolean onErrorHandlingLoop;
    private int onErrorRetries;
    private Histogram.Timer processingTimer;

    public ProcessingStateMachine(ProcessingContext context, BooleanSupplier shouldProcessNext) {
        this.actor = context.getActor();
        this.recordProcessorMap = context.getRecordProcessorMap();
        this.recordValues = context.getRecordValues();
        this.logStreamReader = context.getLogStreamReader();
        this.logStreamWriter = context.getLogStreamWriter();
        this.logStream = context.getLogStream();
        this.zeebeState = context.getZeebeState();
        this.transactionContext = context.getTransactionContext();
        this.abortCondition = context.getAbortCondition();
        this.lastProcessedPositionState = context.getLastProcessedPositionState();
        this.writeRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.sideEffectsRetryStrategy = new AbortableRetryStrategy(this.actor);
        this.updateStateRetryStrategy = new RecoverableRetryStrategy(this.actor);
        this.shouldProcessNext = shouldProcessNext;
        int partitionId = this.logStream.getPartitionId();
        this.typedEvent = new TypedEventImpl(partitionId);
        this.responseWriter = context.getWriters().response();
        this.metrics = new StreamProcessorMetrics(partitionId);
        this.streamProcessorListener = context.getStreamProcessorListener();
    }

    private void skipRecord() {
        this.notifySkippedListener(this.currentEvent);
        this.actor.submit(this::readNextEvent);
        this.metrics.eventSkipped();
    }

    void readNextEvent() {
        if (this.onErrorRetries > 0) {
            this.onErrorHandlingLoop = false;
            this.onErrorRetries = 0;
        }
        this.tryToReadNextEvent();
    }

    private void tryToReadNextEvent() {
        if (this.shouldProcessNext.getAsBoolean() && this.logStreamReader.hasNext() && this.currentProcessor == null) {
            this.currentEvent = (LoggedEvent)this.logStreamReader.next();
            if (this.eventFilter.applies(this.currentEvent)) {
                this.processEvent(this.currentEvent);
            } else {
                this.skipRecord();
            }
        }
    }

    private void processEvent(LoggedEvent event) {
        this.metadata.reset();
        event.readMetadata((BufferReader)this.metadata);
        this.currentProcessor = this.chooseNextProcessor(event);
        if (this.currentProcessor == null) {
            this.skipRecord();
            return;
        }
        long processingStartTime = ActorClock.currentTimeMillis();
        this.processingTimer = this.metrics.startProcessingDurationTimer(this.metadata.getRecordType());
        try {
            UnifiedRecordValue value = this.recordValues.readRecordValue(event, this.metadata.getValueType());
            this.typedEvent.wrap(event, this.metadata, value);
            this.metrics.processingLatency(event.getTimestamp(), processingStartTime);
            this.processInTransaction(this.typedEvent);
            this.metrics.commandsProcessed();
            this.writeEvent();
        }
        catch (RecoverableException recoverableException) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_RETRY_PROCESSING, new Object[]{event, this.metadata, recoverableException});
            this.actor.runDelayed(PROCESSING_RETRY_DELAY, () -> this.processEvent(this.currentEvent));
        }
        catch (Exception e) {
            LOG.error(ERROR_MESSAGE_PROCESSING_FAILED_SKIP_EVENT, new Object[]{event, this.metadata, e});
            this.onError(e, this::writeEvent);
        }
    }

    private TypedRecordProcessor<?> chooseNextProcessor(LoggedEvent event) {
        TypedRecordProcessor typedRecordProcessor = null;
        try {
            typedRecordProcessor = this.recordProcessorMap.get(this.metadata.getRecordType(), this.metadata.getValueType(), this.metadata.getIntent().value());
        }
        catch (Exception e) {
            LOG.error(ERROR_MESSAGE_ON_EVENT_FAILED_SKIP_EVENT, new Object[]{event, this.metadata, e});
        }
        return typedRecordProcessor;
    }

    private void processInTransaction(TypedEventImpl typedRecord) throws Exception {
        this.zeebeDbTransaction = this.transactionContext.getCurrentTransaction();
        this.zeebeDbTransaction.run(() -> {
            boolean isNotOnBlacklist;
            long position = typedRecord.getPosition();
            this.resetOutput(position);
            this.sideEffectProducer = this.responseWriter;
            boolean bl = isNotOnBlacklist = !this.zeebeState.getBlackListState().isOnBlacklist(typedRecord);
            if (isNotOnBlacklist) {
                this.currentProcessor.processRecord(position, typedRecord, this.responseWriter, this.logStreamWriter, this::setSideEffectProducer);
            }
            this.lastProcessedPositionState.markAsProcessed(position);
        });
    }

    private void resetOutput(long sourceRecordPosition) {
        this.responseWriter.reset();
        this.logStreamWriter.reset();
        this.logStreamWriter.configureSourceContext(sourceRecordPosition);
    }

    public void setSideEffectProducer(SideEffectProducer sideEffectProducer) {
        this.sideEffectProducer = sideEffectProducer;
    }

    private void onError(Throwable processingException, Runnable nextStep) {
        ++this.onErrorRetries;
        if (this.onErrorRetries > 1) {
            this.onErrorHandlingLoop = true;
        }
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.rollback();
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_ROLLBACK_ABORTED, new Object[]{this.currentEvent, this.metadata, throwable});
            }
            try {
                this.errorHandlingInTransaction(processingException);
                nextStep.run();
            }
            catch (Exception ex) {
                this.onError(ex, nextStep);
            }
        });
    }

    private void errorHandlingInTransaction(Throwable processingException) throws Exception {
        this.zeebeDbTransaction = this.transactionContext.getCurrentTransaction();
        this.zeebeDbTransaction.run(() -> {
            long position = this.typedEvent.getPosition();
            this.resetOutput(position);
            this.writeRejectionOnCommand(processingException);
            this.errorRecord.initErrorRecord(processingException, position);
            this.zeebeState.getBlackListState().tryToBlacklist(this.typedEvent, arg_0 -> ((ErrorRecord)this.errorRecord).setProcessInstanceKey(arg_0));
            this.logStreamWriter.appendFollowUpEvent(this.typedEvent.getKey(), (Intent)ErrorIntent.CREATED, (RecordValue)this.errorRecord);
        });
    }

    private void writeRejectionOnCommand(Throwable exception) {
        String errorMessage = String.format(PROCESSING_ERROR_MESSAGE, this.typedEvent, exception.getMessage());
        LOG.error(errorMessage, exception);
        this.logStreamWriter.appendRejection(this.typedEvent, RejectionType.PROCESSING_ERROR, errorMessage);
        this.responseWriter.writeRejectionOnCommand(this.typedEvent, RejectionType.PROCESSING_ERROR, errorMessage);
    }

    private void writeEvent() {
        ActorFuture retryFuture = this.writeRetryStrategy.runWithRetry(() -> {
            long position = this.logStreamWriter.flush();
            if (position > 0L) {
                this.writtenEventPosition = position;
            }
            return position >= 0L;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, t) -> {
            if (t != null) {
                LOG.error(ERROR_MESSAGE_WRITE_EVENT_ABORTED, new Object[]{this.currentEvent, this.metadata, t});
                this.onError((Throwable)t, this::writeEvent);
            } else {
                long amount = this.writtenEventPosition - this.lastWrittenEventPosition;
                this.metrics.recordsWritten(amount);
                this.updateState();
            }
        });
    }

    private void updateState() {
        ActorFuture retryFuture = this.updateStateRetryStrategy.runWithRetry(() -> {
            this.zeebeDbTransaction.commit();
            this.lastSuccessfulProcessedEventPosition = this.currentEvent.getPosition();
            this.metrics.setLastProcessedPosition(this.lastSuccessfulProcessedEventPosition);
            this.lastWrittenEventPosition = this.writtenEventPosition;
            return true;
        }, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_UPDATE_STATE_FAILED, new Object[]{this.currentEvent, this.metadata, throwable});
                this.onError((Throwable)throwable, this::updateState);
            } else {
                this.executeSideEffects();
            }
        });
    }

    private void executeSideEffects() {
        ActorFuture retryFuture = this.sideEffectsRetryStrategy.runWithRetry(this.sideEffectProducer::flush, this.abortCondition);
        this.actor.runOnCompletion(retryFuture, (bool, throwable) -> {
            if (throwable != null) {
                LOG.error(ERROR_MESSAGE_EXECUTE_SIDE_EFFECT_ABORTED, new Object[]{this.currentEvent, this.metadata, throwable});
            }
            this.notifyProcessedListener(this.typedEvent);
            this.processingTimer.close();
            this.currentProcessor = null;
            this.actor.submit(this::readNextEvent);
        });
    }

    private void notifyProcessedListener(TypedRecord processedRecord) {
        try {
            this.streamProcessorListener.onProcessed(processedRecord);
        }
        catch (Exception e) {
            LOG.error(NOTIFY_PROCESSED_LISTENER_ERROR_MESSAGE, (Object)processedRecord, (Object)e);
        }
    }

    private void notifySkippedListener(LoggedEvent skippedRecord) {
        try {
            this.streamProcessorListener.onSkipped(skippedRecord);
        }
        catch (Exception e) {
            LOG.error(NOTIFY_SKIPPED_LISTENER_ERROR_MESSAGE, new Object[]{skippedRecord, this.metadata, e});
        }
    }

    public long getLastSuccessfulProcessedEventPosition() {
        return this.lastSuccessfulProcessedEventPosition;
    }

    public long getLastWrittenEventPosition() {
        return this.lastWrittenEventPosition;
    }

    public boolean isMakingProgress() {
        return !this.onErrorHandlingLoop;
    }

    public void startProcessing(long lastProcessedPosition) {
        this.logStreamReader.seekToNextEvent(lastProcessedPosition);
        if (this.lastSuccessfulProcessedEventPosition == -1L) {
            this.lastSuccessfulProcessedEventPosition = lastProcessedPosition;
        }
        this.actor.submit(this::readNextEvent);
    }
}

