/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.snapshots.impl;

import io.camunda.zeebe.snapshots.ConstructableSnapshotStore;
import io.camunda.zeebe.snapshots.PersistableSnapshot;
import io.camunda.zeebe.snapshots.PersistedSnapshot;
import io.camunda.zeebe.snapshots.PersistedSnapshotListener;
import io.camunda.zeebe.snapshots.ReceivableSnapshotStore;
import io.camunda.zeebe.snapshots.SnapshotException;
import io.camunda.zeebe.snapshots.SnapshotId;
import io.camunda.zeebe.snapshots.TransientSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedReceivedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshot;
import io.camunda.zeebe.snapshots.impl.FileBasedSnapshotMetadata;
import io.camunda.zeebe.snapshots.impl.FileBasedTransientSnapshot;
import io.camunda.zeebe.snapshots.impl.InvalidSnapshotChecksum;
import io.camunda.zeebe.snapshots.impl.SfvChecksum;
import io.camunda.zeebe.snapshots.impl.SnapshotChecksum;
import io.camunda.zeebe.snapshots.impl.SnapshotMetrics;
import io.camunda.zeebe.util.Either;
import io.camunda.zeebe.util.FileUtil;
import io.camunda.zeebe.util.sched.Actor;
import io.camunda.zeebe.util.sched.ActorTask;
import io.camunda.zeebe.util.sched.ActorThread;
import io.camunda.zeebe.util.sched.future.ActorFuture;
import io.camunda.zeebe.util.sched.future.CompletableActorFuture;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileBasedSnapshotStore
extends Actor
implements ConstructableSnapshotStore,
ReceivableSnapshotStore {
    private static final String RECEIVING_DIR_FORMAT = "%s-%d";
    private static final Logger LOGGER = LoggerFactory.getLogger(FileBasedSnapshotStore.class);
    private static final String CHECKSUM_SUFFIX = ".checksum";
    private final Path snapshotsDirectory;
    private final Path pendingDirectory;
    private final Set<PersistedSnapshotListener> listeners;
    private final SnapshotMetrics snapshotMetrics;
    private final AtomicReference<FileBasedSnapshot> currentPersistedSnapshotRef = new AtomicReference();
    private final AtomicLong receivingSnapshotStartCount;
    private final Set<PersistableSnapshot> pendingSnapshots = new HashSet<PersistableSnapshot>();
    private final String actorName;
    private final int partitionId;

    public FileBasedSnapshotStore(int nodeId, int partitionId, SnapshotMetrics snapshotMetrics, Path snapshotsDirectory, Path pendingDirectory) {
        this.snapshotsDirectory = snapshotsDirectory;
        this.pendingDirectory = pendingDirectory;
        this.snapshotMetrics = snapshotMetrics;
        this.receivingSnapshotStartCount = new AtomicLong();
        this.listeners = new CopyOnWriteArraySet<PersistedSnapshotListener>();
        this.actorName = FileBasedSnapshotStore.buildActorName((int)nodeId, (String)"SnapshotStore", (int)partitionId);
        this.partitionId = partitionId;
    }

    protected Map<String, String> createContext() {
        Map context = super.createContext();
        context.put("partitionId", Integer.toString(this.partitionId));
        return context;
    }

    public String getName() {
        return this.actorName;
    }

    protected void onActorStarting() {
        this.currentPersistedSnapshotRef.set(this.loadLatestSnapshot(this.snapshotsDirectory));
        this.purgePendingSnapshotsDirectory();
    }

    protected void onActorClosing() {
        this.listeners.clear();
    }

    private FileBasedSnapshot loadLatestSnapshot(Path snapshotDirectory) {
        FileBasedSnapshot latestPersistedSnapshot = null;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(snapshotDirectory, p -> !p.getFileName().toString().endsWith(CHECKSUM_SUFFIX));){
            for (Path path : stream) {
                FileBasedSnapshot snapshot = this.collectSnapshot(path);
                if (snapshot == null || latestPersistedSnapshot != null && snapshot.getMetadata().compareTo(latestPersistedSnapshot.getMetadata()) < 0) continue;
                latestPersistedSnapshot = snapshot;
            }
            if (latestPersistedSnapshot != null) {
                this.cleanupSnapshotDirectory(snapshotDirectory, latestPersistedSnapshot);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return latestPersistedSnapshot;
    }

    private void cleanupSnapshotDirectory(Path snapshotDirectory, FileBasedSnapshot latestPersistedSnapshot) throws IOException {
        Path latestChecksumFile = latestPersistedSnapshot.getChecksumFile();
        Path latestDirectory = latestPersistedSnapshot.getDirectory();
        try (DirectoryStream<Path> paths = Files.newDirectoryStream(snapshotDirectory, p -> !p.equals(latestDirectory) && !p.equals(latestChecksumFile));){
            LOGGER.debug("Deleting snapshots other than {}", (Object)latestPersistedSnapshot.getId());
            paths.forEach(p -> {
                try {
                    LOGGER.debug("Deleting {}", p);
                    FileUtil.deleteFolderIfExists((Path)p);
                }
                catch (IOException e) {
                    LOGGER.warn("Unable to delete {}", p, (Object)e);
                }
            });
        }
    }

    private FileBasedSnapshot collectSnapshot(Path path) throws IOException {
        Optional<FileBasedSnapshotMetadata> optionalMeta = FileBasedSnapshotMetadata.ofPath(path);
        if (optionalMeta.isEmpty()) {
            return null;
        }
        FileBasedSnapshotMetadata metadata = optionalMeta.get();
        Path checksumPath = this.buildSnapshotsChecksumPath(metadata);
        if (!Files.exists(checksumPath, new LinkOption[0])) {
            LOGGER.debug("Snapshot {} does not have a checksum file, which most likely indicates a partial write (e.g. crash during move), and will be deleted", (Object)path);
            try {
                FileUtil.deleteFolder((Path)path);
            }
            catch (Exception e) {
                LOGGER.debug("Failed to delete partial snapshot {}", (Object)path, (Object)e);
            }
            return null;
        }
        try {
            SfvChecksum expectedChecksum = SnapshotChecksum.read(checksumPath);
            SfvChecksum actualChecksum = SnapshotChecksum.calculate(path);
            if (expectedChecksum.getCombinedValue() != actualChecksum.getCombinedValue()) {
                LOGGER.warn("Expected snapshot {} to have checksum {}, but the actual checksum is {}; the snapshot is most likely corrupted. The startup will fail if there is no other valid snapshot and the log has been compacted.", new Object[]{path, expectedChecksum.getCombinedValue(), actualChecksum.getCombinedValue()});
                return null;
            }
            return new FileBasedSnapshot(path, checksumPath, actualChecksum.getCombinedValue(), metadata);
        }
        catch (Exception e) {
            LOGGER.warn("Could not load snapshot in {}", (Object)path, (Object)e);
            return null;
        }
    }

    private void purgePendingSnapshotsDirectory() {
        try (Stream<Path> files = Files.list(this.pendingDirectory);){
            files.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(this::purgePendingSnapshot);
        }
        catch (IOException e) {
            LOGGER.error("Failed to purge pending snapshots, which may result in unnecessary disk usage and should be monitored", (Throwable)e);
        }
    }

    @Override
    public boolean hasSnapshotId(String id) {
        Optional<PersistedSnapshot> optLatestSnapshot = this.getLatestSnapshot();
        if (optLatestSnapshot.isPresent()) {
            PersistedSnapshot snapshot = optLatestSnapshot.get();
            return snapshot.getPath().getFileName().toString().equals(id);
        }
        return false;
    }

    @Override
    public Optional<PersistedSnapshot> getLatestSnapshot() {
        return Optional.ofNullable((PersistedSnapshot)this.currentPersistedSnapshotRef.get());
    }

    @Override
    public ActorFuture<Void> purgePendingSnapshots() {
        CompletableActorFuture abortFuture = new CompletableActorFuture();
        this.actor.run(() -> {
            List abortedAll = this.pendingSnapshots.stream().map(PersistableSnapshot::abort).collect(Collectors.toList());
            this.actor.runOnCompletion(abortedAll, error -> {
                if (error == null) {
                    abortFuture.complete(null);
                } else {
                    abortFuture.completeExceptionally(error);
                }
            });
        });
        return abortFuture;
    }

    @Override
    public ActorFuture<Boolean> addSnapshotListener(PersistedSnapshotListener listener) {
        return this.actor.call(() -> this.listeners.add(listener));
    }

    @Override
    public ActorFuture<Boolean> removeSnapshotListener(PersistedSnapshotListener listener) {
        return this.actor.call(() -> this.listeners.remove(listener));
    }

    @Override
    public long getCurrentSnapshotIndex() {
        return this.getLatestSnapshot().map(PersistedSnapshot::getIndex).orElse(0L);
    }

    @Override
    public ActorFuture<Void> delete() {
        return this.actor.call(() -> {
            this.currentPersistedSnapshotRef.set(null);
            try {
                LOGGER.debug("DELETE FOLDER {}", (Object)this.snapshotsDirectory);
                FileUtil.deleteFolder((Path)this.snapshotsDirectory);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            try {
                LOGGER.debug("DELETE FOLDER {}", (Object)this.pendingDirectory);
                FileUtil.deleteFolder((Path)this.pendingDirectory);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    @Override
    public Path getPath() {
        return this.snapshotsDirectory;
    }

    @Override
    public ActorFuture<Void> copySnapshot(PersistedSnapshot snapshot, Path targetDirectory) {
        CompletableActorFuture result = new CompletableActorFuture();
        this.actor.run(() -> {
            if (!Files.exists(snapshot.getPath(), new LinkOption[0])) {
                result.completeExceptionally(String.format("Expected to copy snapshot %s to directory %s, but snapshot directory %s does not exists. Snapshot may have been deleted.", snapshot.getId(), targetDirectory, snapshot.getPath()), (Throwable)new FileNotFoundException());
            } else {
                try {
                    FileUtil.copySnapshot((Path)snapshot.getPath(), (Path)targetDirectory);
                    result.complete(null);
                }
                catch (Exception e) {
                    result.completeExceptionally(String.format("Failed to copy snapshot %s to directory %s.", snapshot.getId(), targetDirectory), (Throwable)e);
                }
            }
        });
        return result;
    }

    @Override
    public FileBasedReceivedSnapshot newReceivedSnapshot(String snapshotId) {
        Optional<FileBasedSnapshotMetadata> optMetadata = FileBasedSnapshotMetadata.ofFileName(snapshotId);
        FileBasedSnapshotMetadata metadata = optMetadata.orElseThrow(() -> new IllegalArgumentException("Expected snapshot id in a format like 'index-term-processedPosition-exportedPosition', got '" + snapshotId + "'."));
        long nextStartCount = this.receivingSnapshotStartCount.incrementAndGet();
        String pendingDirectoryName = String.format(RECEIVING_DIR_FORMAT, metadata.getSnapshotIdAsString(), nextStartCount);
        Path pendingSnapshotDir = this.pendingDirectory.resolve(pendingDirectoryName);
        FileBasedReceivedSnapshot newPendingSnapshot = new FileBasedReceivedSnapshot(metadata, pendingSnapshotDir, this, this.actor);
        this.addPendingSnapshot(newPendingSnapshot);
        return newPendingSnapshot;
    }

    @Override
    public Either<SnapshotException, TransientSnapshot> newTransientSnapshot(long index, long term, long processedPosition, long exportedPosition) {
        FileBasedSnapshotMetadata newSnapshotId = new FileBasedSnapshotMetadata(index, term, processedPosition, exportedPosition);
        FileBasedSnapshot currentSnapshot = this.currentPersistedSnapshotRef.get();
        if (currentSnapshot != null && currentSnapshot.getMetadata().compareTo(newSnapshotId) == 0) {
            String error = String.format("Previous snapshot was taken for the same processed position %d and exported position %d.", processedPosition, exportedPosition);
            return Either.left((Object)new SnapshotException.SnapshotAlreadyExistsException(error));
        }
        Path directory = this.buildPendingSnapshotDirectory(newSnapshotId);
        FileBasedTransientSnapshot newPendingSnapshot = new FileBasedTransientSnapshot(newSnapshotId, directory, this, this.actor);
        this.addPendingSnapshot(newPendingSnapshot);
        return Either.right((Object)newPendingSnapshot);
    }

    private void addPendingSnapshot(PersistableSnapshot pendingSnapshot) {
        Runnable action = () -> this.pendingSnapshots.add(pendingSnapshot);
        if (!this.isCurrentActor()) {
            this.actor.submit(action);
        } else {
            action.run();
        }
    }

    void removePendingSnapshot(PersistableSnapshot pendingSnapshot) {
        this.pendingSnapshots.remove(pendingSnapshot);
    }

    private boolean isCurrentActor() {
        ActorTask task;
        ActorThread currentActorThread = ActorThread.current();
        if (currentActorThread != null && (task = currentActorThread.getCurrentTask()) != null) {
            return task.getActor() == this;
        }
        return false;
    }

    private void observeSnapshotSize(FileBasedSnapshot persistedSnapshot) {
        try (DirectoryStream<Path> contents = Files.newDirectoryStream(persistedSnapshot.getPath());){
            long totalSize = 0L;
            long totalCount = 0L;
            for (Path path : contents) {
                if (!Files.isRegularFile(path, new LinkOption[0])) continue;
                long size = Files.size(path);
                this.snapshotMetrics.observeSnapshotFileSize(size);
                totalSize += size;
                ++totalCount;
            }
            this.snapshotMetrics.observeSnapshotSize(totalSize);
            this.snapshotMetrics.observeSnapshotChunkCount(totalCount);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to observe size for snapshot {}", (Object)persistedSnapshot, (Object)e);
        }
    }

    private void purgePendingSnapshots(SnapshotId cutoffSnapshot) {
        LOGGER.trace("Search for orphaned snapshots below oldest valid snapshot with index {} in {}", (Object)cutoffSnapshot.getSnapshotIdAsString(), (Object)this.pendingDirectory);
        this.pendingSnapshots.stream().filter(pendingSnapshot -> pendingSnapshot.snapshotId().compareTo(cutoffSnapshot) < 0).forEach(PersistableSnapshot::abort);
        try (DirectoryStream<Path> pendingSnapshotsDirectories = Files.newDirectoryStream(this.pendingDirectory);){
            for (Path pendingSnapshot2 : pendingSnapshotsDirectories) {
                this.purgePendingSnapshot(cutoffSnapshot, pendingSnapshot2);
            }
        }
        catch (IOException e) {
            LOGGER.warn("Failed to delete orphaned snapshots, could not list pending directory {}", (Object)this.pendingDirectory, (Object)e);
        }
    }

    private void purgePendingSnapshot(SnapshotId cutoffIndex, Path pendingSnapshot) {
        Optional<FileBasedSnapshotMetadata> optionalMetadata = FileBasedSnapshotMetadata.ofPath(pendingSnapshot);
        if (optionalMetadata.isPresent() && optionalMetadata.get().compareTo(cutoffIndex) < 0) {
            try {
                FileUtil.deleteFolder((Path)pendingSnapshot);
                LOGGER.debug("Deleted orphaned snapshot {}", (Object)pendingSnapshot);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to delete orphaned snapshot {}, risk using unnecessary disk space", (Object)pendingSnapshot, (Object)e);
            }
        }
    }

    private boolean isCurrentSnapshotNewer(FileBasedSnapshotMetadata metadata) {
        FileBasedSnapshot persistedSnapshot = this.currentPersistedSnapshotRef.get();
        return persistedSnapshot != null && persistedSnapshot.getMetadata().compareTo(metadata) >= 0;
    }

    FileBasedSnapshot newSnapshot(FileBasedSnapshotMetadata metadata, Path directory, long expectedChecksum) {
        boolean failed;
        SfvChecksum actualChecksum;
        FileBasedSnapshot currentPersistedSnapshot = this.currentPersistedSnapshotRef.get();
        if (this.isCurrentSnapshotNewer(metadata)) {
            FileBasedSnapshotMetadata currentPersistedSnapshotMetadata = currentPersistedSnapshot.getMetadata();
            LOGGER.debug("Snapshot is older than the latest snapshot {}. Snapshot {} won't be committed.", (Object)currentPersistedSnapshotMetadata, (Object)metadata);
            this.purgePendingSnapshots(currentPersistedSnapshotMetadata);
            return currentPersistedSnapshot;
        }
        Path destination = this.buildSnapshotDirectory(metadata);
        this.moveToSnapshotDirectory(directory, destination);
        Path checksumPath = this.buildSnapshotsChecksumPath(metadata);
        try {
            actualChecksum = SnapshotChecksum.calculate(destination);
            if (actualChecksum.getCombinedValue() != expectedChecksum) {
                this.rollbackPartialSnapshot(destination);
                throw new InvalidSnapshotChecksum(directory, expectedChecksum, actualChecksum.getCombinedValue());
            }
            SnapshotChecksum.persist(checksumPath, actualChecksum);
        }
        catch (IOException e) {
            this.rollbackPartialSnapshot(destination);
            throw new UncheckedIOException(e);
        }
        FileBasedSnapshot newPersistedSnapshot = new FileBasedSnapshot(destination, checksumPath, actualChecksum.getCombinedValue(), metadata);
        boolean bl = failed = !this.currentPersistedSnapshotRef.compareAndSet(currentPersistedSnapshot, newPersistedSnapshot);
        if (failed) {
            String errorMessage = "Expected that last snapshot is '%s', which should be replace with '%s', but last snapshot was '%s'.";
            throw new ConcurrentModificationException(String.format("Expected that last snapshot is '%s', which should be replace with '%s', but last snapshot was '%s'.", currentPersistedSnapshot, newPersistedSnapshot.getMetadata(), this.currentPersistedSnapshotRef.get()));
        }
        LOGGER.info("Committed new snapshot {}", (Object)newPersistedSnapshot.getId());
        this.snapshotMetrics.incrementSnapshotCount();
        this.observeSnapshotSize(newPersistedSnapshot);
        LOGGER.trace("Purging snapshots older than {}", (Object)newPersistedSnapshot.getMetadata().getSnapshotIdAsString());
        if (currentPersistedSnapshot != null) {
            LOGGER.debug("Deleting previous snapshot {}", (Object)currentPersistedSnapshot.getMetadata().getSnapshotIdAsString());
            currentPersistedSnapshot.delete();
        }
        this.purgePendingSnapshots(newPersistedSnapshot.getMetadata());
        this.listeners.forEach(listener -> listener.onNewSnapshot(newPersistedSnapshot));
        return newPersistedSnapshot;
    }

    private void moveToSnapshotDirectory(Path directory, Path destination) {
        try {
            this.tryAtomicDirectoryMove(directory, destination);
        }
        catch (FileAlreadyExistsException e) {
            LOGGER.debug("Expected to move snapshot from {} to {}, but it already exists", new Object[]{directory, destination, e});
        }
        catch (IOException e) {
            this.rollbackPartialSnapshot(destination);
            throw new UncheckedIOException(e);
        }
    }

    private void rollbackPartialSnapshot(Path destination) {
        try {
            FileUtil.deleteFolderIfExists((Path)destination);
        }
        catch (IOException ioException) {
            LOGGER.debug("Pending snapshot {} could not be deleted on rollback, but will be safely ignored as a partial snapshot", (Object)destination, (Object)ioException);
        }
    }

    private void purgePendingSnapshot(Path pendingSnapshot) {
        try {
            FileUtil.deleteFolder((Path)pendingSnapshot);
            LOGGER.debug("Deleted not completed (orphaned) snapshot {}", (Object)pendingSnapshot);
        }
        catch (IOException e) {
            LOGGER.warn("Failed to delete not completed (orphaned) snapshot {}", (Object)pendingSnapshot, (Object)e);
        }
    }

    private void tryAtomicDirectoryMove(Path directory, Path destination) throws IOException {
        try {
            FileUtil.moveDurably((Path)directory, (Path)destination, (CopyOption[])new CopyOption[]{StandardCopyOption.ATOMIC_MOVE});
        }
        catch (AtomicMoveNotSupportedException e) {
            LOGGER.warn("Atomic move not supported. Moving the snapshot files non-atomically", (Throwable)e);
            FileUtil.moveDurably((Path)directory, (Path)destination, (CopyOption[])new CopyOption[0]);
        }
    }

    private Path buildPendingSnapshotDirectory(SnapshotId id) {
        return this.pendingDirectory.resolve(id.getSnapshotIdAsString());
    }

    private Path buildSnapshotDirectory(FileBasedSnapshotMetadata metadata) {
        return this.snapshotsDirectory.resolve(metadata.getSnapshotIdAsString());
    }

    private Path buildSnapshotsChecksumPath(FileBasedSnapshotMetadata metadata) {
        return this.snapshotsDirectory.resolve(metadata.getSnapshotIdAsString() + CHECKSUM_SUFFIX);
    }

    SnapshotMetrics getSnapshotMetrics() {
        return this.snapshotMetrics;
    }
}

