/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.node;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.net.InetAddresses;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.airlift.configuration.ConfigurationLoader;
import io.airlift.configuration.ConfigurationUtils;
import io.airlift.node.AddressToHostname;
import io.airlift.node.NodeConfig;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Function;
import org.weakref.jmx.Managed;

@Singleton
public class NodeInfo {
    private final String environment;
    private final String pool;
    private final String nodeId;
    private final String location;
    private final String binarySpec;
    private final String configSpec;
    private final String instanceId = UUID.randomUUID().toString();
    private final String internalAddress;
    private final String externalAddress;
    private final InetAddress bindIp;
    private final long startTime = System.currentTimeMillis();
    private final Map<String, String> annotations;
    private final boolean preferIpv6Address;

    public NodeInfo(String environment) {
        this(new NodeConfig().setEnvironment(environment));
    }

    @Inject
    public NodeInfo(NodeConfig config) {
        this(config.getEnvironment(), config.getPool(), config.getNodeId(), config.getNodeInternalAddress(), config.getNodeBindIp(), config.getNodeExternalAddress(), config.getLocation(), config.getBinarySpec(), config.getConfigSpec(), config.getInternalAddressSource(), config.getAnnotationFile(), config.getPreferIpv6Address());
    }

    public NodeInfo(String environment, String pool, String nodeId, String internalAddress, InetAddress bindIp, String externalAddress, String location, String binarySpec, String configSpec, NodeConfig.AddressSource internalAddressSource, String annotationFile, boolean preferIpv6Address) {
        Objects.requireNonNull(environment, "environment is null");
        Objects.requireNonNull(pool, "pool is null");
        Objects.requireNonNull(internalAddressSource, "internalAddressSource is null");
        Preconditions.checkArgument((boolean)environment.matches("[a-z0-9][_a-z0-9]*"), (String)"environment '%s' %s", (Object)environment, (Object)"should match [a-z0-9][_a-z0-9]*");
        Preconditions.checkArgument((boolean)pool.matches("[a-z0-9][_a-z0-9]*"), (String)"pool '%s' %s", (Object)pool, (Object)"should match [a-z0-9][_a-z0-9]*");
        this.environment = environment;
        this.pool = pool;
        this.preferIpv6Address = preferIpv6Address;
        if (nodeId != null) {
            Preconditions.checkArgument((boolean)nodeId.matches("[A-Za-z0-9][_A-Za-z0-9-]*"), (String)"nodeId '%s' %s", (Object)nodeId, (Object)"should match [A-Za-z0-9][_A-Za-z0-9-]*");
            this.nodeId = nodeId;
        } else {
            this.nodeId = UUID.randomUUID().toString();
        }
        this.location = location != null ? location : "/" + this.nodeId;
        this.binarySpec = binarySpec;
        this.configSpec = configSpec;
        this.internalAddress = internalAddress != null ? internalAddress : this.findInternalAddress(internalAddressSource);
        this.bindIp = bindIp != null ? bindIp : InetAddresses.fromInteger((int)0);
        this.externalAddress = externalAddress != null ? externalAddress : this.internalAddress;
        if (annotationFile != null) {
            try {
                this.annotations = ImmutableMap.copyOf((Map)ConfigurationUtils.replaceEnvironmentVariables((Map)ConfigurationLoader.loadPropertiesFrom((String)annotationFile)));
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        } else {
            this.annotations = ImmutableMap.of();
        }
    }

    @Managed
    public String getEnvironment() {
        return this.environment;
    }

    @Managed
    public String getPool() {
        return this.pool;
    }

    @Managed
    public String getNodeId() {
        return this.nodeId;
    }

    @Managed
    public String getLocation() {
        return this.location;
    }

    @Managed
    public String getBinarySpec() {
        return this.binarySpec;
    }

    @Managed
    public String getConfigSpec() {
        return this.configSpec;
    }

    @Managed
    public String getInstanceId() {
        return this.instanceId;
    }

    @Managed
    public String getInternalAddress() {
        return this.internalAddress;
    }

    @Managed
    public String getExternalAddress() {
        return this.externalAddress;
    }

    @Managed
    public InetAddress getBindIp() {
        return this.bindIp;
    }

    @Managed
    public long getStartTime() {
        return this.startTime;
    }

    public Map<String, String> getAnnotations() {
        return this.annotations;
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("nodeId", (Object)this.nodeId).add("instanceId", (Object)this.instanceId).add("internalAddress", (Object)this.internalAddress).add("externalAddress", (Object)this.externalAddress).add("bindIp", (Object)this.bindIp).add("startTime", this.startTime).add("annotations", this.annotations).toString();
    }

    private String findInternalAddress(NodeConfig.AddressSource addressSource) {
        return switch (addressSource) {
            default -> throw new MatchException(null, null);
            case NodeConfig.AddressSource.IP -> InetAddresses.toAddrString((InetAddress)this.findInternalIp());
            case NodeConfig.AddressSource.IP_ENCODED_AS_HOSTNAME -> AddressToHostname.encodeAddressAsHostname(this.findInternalIp());
            case NodeConfig.AddressSource.HOSTNAME -> NodeInfo.getLocalHost().getHostName();
            case NodeConfig.AddressSource.FQDN -> NodeInfo.getLocalHost().getCanonicalHostName();
        };
    }

    private InetAddress findInternalIp() {
        ArrayList<Function<List, Optional>> searchOrder = new ArrayList<Function<List, Optional>>(3);
        if (this.preferIpv6Address) {
            searchOrder.add(NodeInfo::findIpv6address);
            searchOrder.add(NodeInfo::findLocalAddress);
            searchOrder.add(NodeInfo::findIpv4address);
        } else {
            searchOrder.add(NodeInfo::findLocalAddress);
            searchOrder.add(NodeInfo::findIpv4address);
            searchOrder.add(NodeInfo::findIpv6address);
        }
        List<InetAddress> goodAddresses = NodeInfo.getGoodAddresses();
        return searchOrder.stream().map(source -> (Optional)source.apply(goodAddresses)).filter(Optional::isPresent).findFirst().orElse(Optional.empty()).orElse(null);
    }

    private static Optional<InetAddress> findIpv4address(List<InetAddress> candidates) {
        for (InetAddress address : candidates) {
            if (!NodeInfo.isV4Address(address)) continue;
            return Optional.of(address);
        }
        return Optional.empty();
    }

    private static Optional<InetAddress> findIpv6address(List<InetAddress> candidates) {
        for (InetAddress address : candidates) {
            if (!NodeInfo.isV6Address(address)) continue;
            return Optional.of(address);
        }
        return Optional.empty();
    }

    private static Optional<InetAddress> findLocalAddress(List<InetAddress> candidates) {
        InetAddress localAddress = null;
        try {
            localAddress = InetAddress.getLocalHost();
            if (NodeInfo.isV4Address(localAddress) && NodeInfo.getGoodAddresses().contains(localAddress)) {
                return Optional.of(localAddress);
            }
        }
        catch (UnknownHostException unknownHostException) {
            // empty catch block
        }
        if (localAddress == null) {
            try {
                localAddress = InetAddress.getByAddress(new byte[]{127, 0, 0, 1});
            }
            catch (UnknownHostException e) {
                throw new AssertionError((Object)"Could not get local ip address");
            }
        }
        return Optional.ofNullable(localAddress);
    }

    private static List<InetAddress> getGoodAddresses() {
        ImmutableList.Builder list = ImmutableList.builder();
        for (NetworkInterface networkInterface : NodeInfo.getGoodNetworkInterfaces()) {
            for (InetAddress address : Collections.list(networkInterface.getInetAddresses())) {
                if (!NodeInfo.isGoodAddress(address)) continue;
                list.add((Object)address);
            }
        }
        return list.build();
    }

    private static List<NetworkInterface> getGoodNetworkInterfaces() {
        ImmutableList.Builder builder = ImmutableList.builder();
        try {
            for (NetworkInterface networkInterface : Collections.list(NetworkInterface.getNetworkInterfaces())) {
                try {
                    if (networkInterface.isLoopback() || !networkInterface.isUp()) continue;
                    builder.add((Object)networkInterface);
                }
                catch (Exception exception) {}
            }
        }
        catch (SocketException socketException) {
            // empty catch block
        }
        return builder.build();
    }

    private static boolean isV4Address(InetAddress address) {
        return address instanceof Inet4Address;
    }

    private static boolean isV6Address(InetAddress address) {
        return address instanceof Inet6Address;
    }

    private static boolean isGoodAddress(InetAddress address) {
        return !address.isAnyLocalAddress() && !address.isLinkLocalAddress() && !address.isLoopbackAddress() && !address.isMulticastAddress();
    }

    private static InetAddress getLocalHost() {
        try {
            return InetAddress.getLocalHost();
        }
        catch (UnknownHostException e) {
            throw new UncheckedIOException(e);
        }
    }
}

