/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.compress.snappy;

import com.google.common.base.Charsets;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.UnsignedBytes;
import io.airlift.compress.TestingModule;
import io.airlift.compress.benchmark.DataSet;
import io.airlift.compress.snappy.Crc32C;
import io.airlift.compress.snappy.RandomGenerator;
import io.airlift.compress.snappy.SnappyCompressor;
import io.airlift.compress.snappy.SnappyFramed;
import io.airlift.compress.snappy.SnappyFramedInputStream;
import io.airlift.compress.snappy.SnappyFramedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.List;
import org.testng.Assert;
import org.testng.annotations.Test;

public class TestSnappyStream {
    private static final File TEST_DATA_DIR = new File("testdata");

    protected static byte[] getRandom(double compressionRatio, int length) {
        RandomGenerator gen = new RandomGenerator(compressionRatio);
        gen.getNextPosition(length);
        byte[] random = Arrays.copyOf(gen.data, length);
        Assert.assertEquals((int)random.length, (int)length);
        return random;
    }

    protected byte[] getMarkerFrame() {
        return SnappyFramed.HEADER_BYTES;
    }

    @Test
    public void testSimple() throws Exception {
        byte[] original = "aaaaaaaaaaaabbbbbbbaaaaaa".getBytes(Charsets.UTF_8);
        byte[] compressed = TestSnappyStream.compress(original);
        byte[] uncompressed = TestSnappyStream.uncompress(compressed);
        Assert.assertEquals((byte[])uncompressed, (byte[])original);
        Assert.assertEquals((int)compressed.length, (int)37);
        Assert.assertEquals((byte[])Arrays.copyOf(compressed, 10), (byte[])SnappyFramed.HEADER_BYTES);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[10]), (int)0);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[11]), (int)23);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[12]), (int)0);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[13]), (int)0);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[17]), (int)146);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[16]), (int)116);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[15]), (int)205);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[14]), (int)168);
    }

    @Test
    public void testUncompressible() throws Exception {
        byte[] random = TestSnappyStream.getRandom(1.0, 5000);
        byte[] compressed = TestSnappyStream.compress(random);
        byte[] uncompressed = TestSnappyStream.uncompress(compressed);
        Assert.assertEquals((byte[])uncompressed, (byte[])random);
        Assert.assertEquals((int)compressed.length, (int)(random.length + 10 + 4 + 4));
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[10]), (int)1);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[13]), (int)0);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[12]), (int)19);
        Assert.assertEquals((int)UnsignedBytes.toInt((byte)compressed[11]), (int)140);
    }

    @Test
    public void testEmptyCompression() throws Exception {
        byte[] empty = new byte[]{};
        Assert.assertEquals((byte[])TestSnappyStream.compress(empty), (byte[])SnappyFramed.HEADER_BYTES);
        Assert.assertEquals((byte[])TestSnappyStream.uncompress(SnappyFramed.HEADER_BYTES), (byte[])empty);
    }

    @Test(expectedExceptions={EOFException.class}, expectedExceptionsMessageRegExp=".*block header.*")
    public void testShortBlockHeader() throws Exception {
        this.uncompressBlock(new byte[]{0});
    }

    @Test(expectedExceptions={EOFException.class}, expectedExceptionsMessageRegExp=".*reading frame.*")
    public void testShortBlockData() throws Exception {
        this.uncompressBlock(new byte[]{1, 8, 0, 0, 0, 0, 0, 0, 120, 120});
    }

    @Test
    public void testUnskippableChunkFlags() throws Exception {
        for (int i = 2; i <= 127; ++i) {
            try {
                this.uncompressBlock(new byte[]{(byte)i, 5, 0, 0, 0, 0, 0, 0, 0});
                Assert.fail((String)("no exception thrown with flag: " + Integer.toHexString(i)));
                continue;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testSkippableChunkFlags() throws Exception {
        for (int i = 128; i <= 254; ++i) {
            try {
                this.uncompressBlock(new byte[]{(byte)i, 5, 0, 0, 0, 0, 0, 0, 0});
                continue;
            }
            catch (IOException e) {
                Assert.fail((String)("exception thrown with flag: " + Integer.toHexString(i)), (Throwable)e);
            }
        }
    }

    @Test(expectedExceptions={IOException.class}, expectedExceptionsMessageRegExp="invalid length.*4.*")
    public void testInvalidBlockSizeZero() throws Exception {
        this.uncompressBlock(new byte[]{1, 4, 0, 0, 0, 0, 0, 0});
    }

    @Test(expectedExceptions={IOException.class}, expectedExceptionsMessageRegExp="Corrupt input: invalid checksum")
    public void testInvalidChecksum() throws Exception {
        this.uncompressBlock(new byte[]{1, 5, 0, 0, 0, 0, 0, 0, 97});
    }

    @Test
    public void testInvalidChecksumIgnoredWhenVerificationDisabled() throws Exception {
        byte[] block = new byte[]{1, 5, 0, 0, 0, 0, 0, 0, 97};
        ByteArrayInputStream inputData = new ByteArrayInputStream(TestSnappyStream.blockToStream(block));
        Assert.assertEquals((byte[])ByteStreams.toByteArray((InputStream)new SnappyFramedInputStream((InputStream)inputData, false)), (byte[])new byte[]{97});
    }

    @Test
    public void testLargerFrames_raw_() throws IOException {
        byte[] random = TestSnappyStream.getRandom(0.5, 100000);
        byte[] stream = new byte[SnappyFramed.HEADER_BYTES.length + 8 + random.length];
        System.arraycopy(SnappyFramed.HEADER_BYTES, 0, stream, 0, SnappyFramed.HEADER_BYTES.length);
        stream[10] = 1;
        int length = random.length + 4;
        stream[11] = (byte)length;
        stream[12] = (byte)(length >>> 8);
        stream[13] = (byte)(length >>> 16);
        int crc32c = Crc32C.maskedCrc32c((byte[])random);
        stream[14] = (byte)crc32c;
        stream[15] = (byte)(crc32c >>> 8);
        stream[16] = (byte)(crc32c >>> 16);
        stream[17] = (byte)(crc32c >>> 24);
        System.arraycopy(random, 0, stream, 18, random.length);
        byte[] uncompressed = TestSnappyStream.uncompress(stream);
        Assert.assertEquals((byte[])random, (byte[])uncompressed);
    }

    @Test
    public void testLargerFrames_compressed_() throws IOException {
        byte[] random = TestSnappyStream.getRandom(0.5, 500000);
        byte[] compressed = TestSnappyStream.blockCompress(random);
        byte[] stream = new byte[SnappyFramed.HEADER_BYTES.length + 8 + compressed.length];
        System.arraycopy(SnappyFramed.HEADER_BYTES, 0, stream, 0, SnappyFramed.HEADER_BYTES.length);
        stream[10] = 0;
        int length = compressed.length + 4;
        stream[11] = (byte)length;
        stream[12] = (byte)(length >>> 8);
        stream[13] = (byte)(length >>> 16);
        int crc32c = Crc32C.maskedCrc32c((byte[])random);
        stream[14] = (byte)crc32c;
        stream[15] = (byte)(crc32c >>> 8);
        stream[16] = (byte)(crc32c >>> 16);
        stream[17] = (byte)(crc32c >>> 24);
        System.arraycopy(compressed, 0, stream, 18, compressed.length);
        byte[] uncompressed = TestSnappyStream.uncompress(stream);
        Assert.assertEquals((byte[])random, (byte[])uncompressed);
    }

    @Test
    public void testLargerFrames_compressed_smaller_raw_larger() throws IOException {
        byte[] random = TestSnappyStream.getRandom(0.5, 100000);
        byte[] compressed = TestSnappyStream.blockCompress(random);
        byte[] stream = new byte[SnappyFramed.HEADER_BYTES.length + 8 + compressed.length];
        System.arraycopy(SnappyFramed.HEADER_BYTES, 0, stream, 0, SnappyFramed.HEADER_BYTES.length);
        stream[10] = 0;
        int length = compressed.length + 4;
        stream[11] = (byte)length;
        stream[12] = (byte)(length >>> 8);
        stream[13] = (byte)(length >>> 16);
        int crc32c = Crc32C.maskedCrc32c((byte[])random);
        stream[14] = (byte)crc32c;
        stream[15] = (byte)(crc32c >>> 8);
        stream[16] = (byte)(crc32c >>> 16);
        stream[17] = (byte)(crc32c >>> 24);
        System.arraycopy(compressed, 0, stream, 18, compressed.length);
        byte[] uncompressed = TestSnappyStream.uncompress(stream);
        Assert.assertEquals((byte[])random, (byte[])uncompressed);
    }

    private byte[] uncompressBlock(byte[] block) throws IOException {
        return TestSnappyStream.uncompress(TestSnappyStream.blockToStream(block));
    }

    private static byte[] blockToStream(byte[] block) {
        byte[] stream = new byte[SnappyFramed.HEADER_BYTES.length + block.length];
        System.arraycopy(SnappyFramed.HEADER_BYTES, 0, stream, 0, SnappyFramed.HEADER_BYTES.length);
        System.arraycopy(block, 0, stream, SnappyFramed.HEADER_BYTES.length, block.length);
        return stream;
    }

    @Test
    public void testLargeWrites() throws Exception {
        int c;
        byte[] random = TestSnappyStream.getRandom(0.5, 500000);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        SnappyFramedOutputStream snappyOut = new SnappyFramedOutputStream((OutputStream)out);
        int small = 1000;
        snappyOut.write(random, 0, small);
        snappyOut.write(random, small, random.length - small);
        snappyOut.close();
        byte[] compressed = out.toByteArray();
        Assert.assertTrue((compressed.length < random.length ? 1 : 0) != 0);
        byte[] uncompressed = TestSnappyStream.uncompress(compressed);
        Assert.assertEquals((byte[])uncompressed, (byte[])random);
        SnappyFramedInputStream in = new SnappyFramedInputStream((InputStream)new ByteArrayInputStream(compressed), true);
        int i = 0;
        while ((c = in.read()) != -1) {
            uncompressed[i++] = (byte)c;
        }
        Assert.assertEquals((int)i, (int)random.length);
        Assert.assertEquals((byte[])uncompressed, (byte[])random);
    }

    @Test
    public void testSingleByteWrites() throws Exception {
        byte[] random = TestSnappyStream.getRandom(0.5, 500000);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        SnappyFramedOutputStream snappyOut = new SnappyFramedOutputStream((OutputStream)out);
        for (byte b : random) {
            snappyOut.write(b);
        }
        snappyOut.close();
        byte[] compressed = out.toByteArray();
        Assert.assertTrue((compressed.length < random.length ? 1 : 0) != 0);
        byte[] uncompressed = TestSnappyStream.uncompress(compressed);
        Assert.assertEquals((byte[])uncompressed, (byte[])random);
    }

    @Test
    public void testExtraFlushes() throws Exception {
        byte[] random = TestSnappyStream.getRandom(0.5, 500000);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        SnappyFramedOutputStream snappyOut = new SnappyFramedOutputStream((OutputStream)out);
        snappyOut.write(random);
        for (int i = 0; i < 10; ++i) {
            snappyOut.flush();
        }
        snappyOut.close();
        byte[] compressed = out.toByteArray();
        Assert.assertTrue((compressed.length < random.length ? 1 : 0) != 0);
        byte[] uncompressed = TestSnappyStream.uncompress(compressed);
        Assert.assertEquals((byte[])uncompressed, (byte[])random);
    }

    @Test
    public void testUncompressibleRange() throws Exception {
        int max = 131072;
        byte[] random = TestSnappyStream.getRandom(1.0, max);
        for (int i = 1; i <= max; i += 102) {
            byte[] original = Arrays.copyOfRange(random, 0, i);
            byte[] compressed = TestSnappyStream.compress(original);
            byte[] uncompressed = TestSnappyStream.uncompress(compressed);
            Assert.assertEquals((byte[])uncompressed, (byte[])original);
        }
    }

    @Test
    public void testByteForByteTestData() throws Exception {
        List<DataSet> dataSets = new TestingModule().dataSets();
        for (DataSet dataSet : dataSets) {
            byte[] original = dataSet.getUncompressed();
            byte[] compressed = TestSnappyStream.compress(original);
            byte[] uncompressed = TestSnappyStream.uncompress(compressed);
            Assert.assertEquals((byte[])uncompressed, (byte[])original);
        }
    }

    @Test(expectedExceptions={EOFException.class}, expectedExceptionsMessageRegExp=".*stream header.*")
    public void testEmptyStream() throws Exception {
        TestSnappyStream.uncompress(new byte[0]);
    }

    @Test(expectedExceptions={IOException.class}, expectedExceptionsMessageRegExp=".*stream header.*")
    public void testInvalidStreamHeader() throws Exception {
        TestSnappyStream.uncompress(new byte[]{98, 0, 0, 103, 117, 115, 0});
    }

    @Test
    public void testCloseIsIdempotent() throws Exception {
        byte[] random = TestSnappyStream.getRandom(0.5, 500000);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        SnappyFramedOutputStream snappyOut = new SnappyFramedOutputStream((OutputStream)out);
        snappyOut.write(random);
        snappyOut.close();
        snappyOut.close();
        byte[] compressed = out.toByteArray();
        SnappyFramedInputStream snappyIn = new SnappyFramedInputStream((InputStream)new ByteArrayInputStream(compressed), true);
        byte[] uncompressed = ByteStreams.toByteArray((InputStream)snappyIn);
        Assert.assertEquals((byte[])uncompressed, (byte[])random);
        snappyIn.close();
        snappyIn.close();
    }

    @Test
    public void testMarkerFrameInStream() throws IOException {
        int toWrite;
        int size = 500000;
        byte[] random = TestSnappyStream.getRandom(0.5, size);
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        SnappyFramedOutputStream os = new SnappyFramedOutputStream((OutputStream)out);
        byte[] markerFrame = this.getMarkerFrame();
        for (int i = 0; i < size; i += toWrite) {
            toWrite = Math.max((size - i) / 4, 512);
            os.write(random, i, Math.min(size - i, toWrite));
            os.flush();
            out.write(markerFrame);
        }
        byte[] compressed = out.toByteArray();
        byte[] uncompressed = TestSnappyStream.uncompress(compressed);
        Assert.assertEquals((byte[])random, (byte[])uncompressed);
    }

    public static byte[] blockCompress(byte[] data) {
        SnappyCompressor compressor = new SnappyCompressor();
        byte[] compressedOut = new byte[compressor.maxCompressedLength(data.length)];
        int compressedSize = compressor.compress(data, 0, data.length, compressedOut, 0, compressedOut.length);
        byte[] trimmedBuffer = Arrays.copyOf(compressedOut, compressedSize);
        return trimmedBuffer;
    }

    private static byte[] compress(byte[] original) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        SnappyFramedOutputStream snappyOut = new SnappyFramedOutputStream((OutputStream)out);
        snappyOut.write(original);
        snappyOut.close();
        return out.toByteArray();
    }

    private static byte[] uncompress(byte[] compressed) throws IOException {
        return ByteStreams.toByteArray((InputStream)new SnappyFramedInputStream((InputStream)new ByteArrayInputStream(compressed)));
    }

    static File[] getTestFiles() {
        File[] testFiles = TEST_DATA_DIR.listFiles();
        Assert.assertTrue((testFiles != null && testFiles.length > 0 ? 1 : 0) != 0, (String)("No test files at " + TEST_DATA_DIR.getAbsolutePath()));
        return testFiles;
    }
}

