package io.aicactus.adsnetwork.models.bid

import android.content.Context
import android.os.Build
import android.webkit.WebSettings
import androidx.annotation.NonNull
import com.google.gson.annotations.SerializedName
import io.aicactus.adsnetwork.internal.Exclude
import io.aicactus.adsnetwork.utils.SingletonHolder
import kotlin.math.roundToInt

/**
 * This object provides information pertaining to the device through which the user is interacting.
 * Device information includes its hardware, platform, location, and carrier data.
 * The device can refer to a mobile handset, a desktop computer, set top box, or other digital device.
 */
data class Device (
    @NonNull @Exclude val context: Context,
) {

    /**
     * Browser user agent string.
     */
    @field:SerializedName("ua")
    private val userAgent: String = WebSettings.getDefaultUserAgent(context)

    /**
     * Location of the device assumed to be the user’s current location defined by a [Geo] object
     */
    @field:SerializedName("geo")
    private val geo: Geo? = null

    /**
     * Standard “Do Not Track” flag as set in the header by the browser,
     * where 0 = tracking is unrestricted, 1 = do not track.
     */
    @field:SerializedName("dnt")
    private val doNotTrack: Int? = null

    /**
     * “Limit Ad Tracking” signal commercially endorsed (e.g., iOS, Android),
     * where 0 = tracking is unrestricted, 1 = tracking must be limited per commercial guidelines.
     */
    @field:SerializedName("lmt")
    var limitAdTracking: Int = 0

    /**
     * IPv4 address closest to device.
     */
    @field:SerializedName("ip")
    var ip: String? = null

    /**
     * IP address closest to device as IPv6.
     */
    @field:SerializedName("ipv6")
    var ipV6: String? = null

    /**
     * The general type of device.
     */
    @field:SerializedName("devicetype")
    var deviceType: Int = Type.PHONE.value

    /**
     * Device make (e.g., “Google”).
     */
    @field:SerializedName("make")
    private val make: String = Build.BRAND

    /**
     * Device model (e.g., “Pixel 6”).
     */
    @field:SerializedName("model")
    private val model: String = Build.MODEL

    /**
     * Device operating system
     */
    @field:SerializedName("os")
    private val osName: String = "Android"

    /**
     * Device operating system version (e.g., “11.0.0”).
     */
    @field:SerializedName("osv")
    private val osVersion: String = Build.VERSION.RELEASE

    /**
     * Hardware version of the device
     */
    @field:SerializedName("hwv")
    private val hardwareVersion: String = Build.HARDWARE

    /**
     * Physical height of the screen in pixels.
     */
    @field:SerializedName("h")
    private val height: Int

    /**
     * Physical width of the screen in pixels.
     */
    @field:SerializedName("w")
    private val width: Int

    /**
     * Screen size as pixels per linear inch.
     */
    @field:SerializedName("ppi")
    private val ppi: Int

    /**
     * The ratio of physical pixels to device independent pixels.
     */
    @field:SerializedName("pxratio")
    private val pixelRatio: Float

    /**
     * Support for JavaScript, where 0 = no, 1 = yes.
     */
    @field:SerializedName("js")
    private val supportJS: Int = 1

    /**
     * Indicates if the geolocation API will be available to JavaScript code running in the banner,
     * where 0 = no, 1 = yes.
     */
    @field:SerializedName("geofetch")
    private val geoFetch: Int? = null

    /**
     * Version of Flash supported by the browser.
     */
    @field:SerializedName("flashver")
    private val flashVersion: String? = null

    /**
     * Browser language using ISO-639-1-alpha-2.
     */
    @field:SerializedName("language")
    private val language: String? = null

    /**
     * Carrier or ISP (e.g., “VERIZON”) using exchange curated string names
     * which should be published to bidders a priori.
     */
    @field:SerializedName("carrier")
    private val carrier: String? = null

    /**
     * Mobile carrier as the concatenated MCC-MNC code (e.g., “310-005” identifies Verizon Wireless CDMA in the USA).
     * Refer to https://en.wikipedia.org/wiki/Mobile_country_code for further examples.
     * Note that the dash between the MCC and MNC parts is required to remove parsing ambiguity.
     */
    @field:SerializedName("mccmnc")
    private val mccmnc: String? = null

    /**
     * Network connection type.
     */
    @field:SerializedName("connectiontype")
    var connectionType: Int = ConnectionType.ETHERNET.value

    /**
     * ID sanctioned for advertiser use in the clear (i.e., not hashed).
     */
    @field:SerializedName("ifa")
    var idForAdvertiser: String = "00000000-0000-0000-0000-000000000000"

    /**
     * Hardware device ID (e.g., IMEI); hashed via SHA1.
     */
    @field:SerializedName("didsha1")
    var deviceIDSHA1: String? = null

    /**
     * Hardware device ID (e.g., IMEI); hashed via MD5.
     */
    @field:SerializedName("didmd5")
    var deviceIDMD5: String? = null

    /**
     * Platform device ID (e.g., Android ID); hashed via SHA1.
     */
    @field:SerializedName("dpidsha1")
    var devicePlatformSHA1: String? = null

    /**
     * Platform device ID (e.g., Android ID); hashed via MD5.
     */
    @field:SerializedName("dpidmd5")
    var devicePlatformMD5: String? = null

    /**
     * MAC address of the device; hashed via SHA1.
     */
    @field:SerializedName("macsha1")
    var macAddressSHA1: String? = null

    /**
     * MAC address of the device; hashed via MD5.
     */
    @field:SerializedName("macmd5")
    var macAddressMD5: String? = null

    /**
     * Placeholder for exchange-specific extensions to OpenRTB.
     */
    @field:SerializedName("ext")
    var extension: Any? = null

    init {
        val displayMetrics = context.resources.displayMetrics
        val widthPixels = displayMetrics.widthPixels
        val heightPixels = displayMetrics.heightPixels
        val density = displayMetrics.density

        width = (widthPixels / density).roundToInt()
        height = (heightPixels / density).roundToInt()
        ppi = displayMetrics.densityDpi
        pixelRatio = density
    }

    val stringPresentation: String
        get() {
            return "UserAgent: $userAgent / OS: $osName / OS Version: $osVersion / " +
                    "Model: $model / Make: $make / Hardware: $hardwareVersion" +
                    "Width: $width / Height: $height / PixelRatio: $pixelRatio / PPI: $ppi"
        }

    companion object: SingletonHolder<Device, Context>(::Device)

    enum class Type(val value: Int) {
        MOBILE_OR_TABLET(1),
        PERSONAL_COMPUTER(2),
        CONNECTED_TV(3),
        PHONE(4),
        TABLET(5),
        CONNECTED_DEVICE(6),
        SET_TOP_BOX(7),
    }
}