package io.aicactus.adsnetwork.internal

import android.content.Context
import android.net.Uri
import android.util.Log
import androidx.annotation.NonNull
import androidx.annotation.WorkerThread
import com.google.gson.Gson
import io.aicactus.adsnetwork.utils.SingletonHolder
import io.aicactus.adsnetwork.utils.UriUtils
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.util.zip.GZIPInputStream
import javax.net.ssl.HttpsURLConnection

open class ServiceHttpClient(@NonNull private val context: Context) {

    private var userAgentGenerator = UserAgentGenerator(context)

    val gson = Gson()

    @WorkerThread
    @NonNull
    inline fun <reified T> post(
        @NonNull uri: Uri,
        @NonNull requestHeaders: Map<String, String>,
        @NonNull postData: Map<String, String>,
    ): AicactusApiResponse<T> {
        val postDataBytes = convertPostDataToBytes(postData)
        var conn: HttpsURLConnection? = null
        try {
            conn = openPostConnection(uri, postDataBytes.size)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            conn.outputStream.apply {
                write(postDataBytes)
                flush()
            }
            return getServiceResponse(conn)
        } catch (e: IOException) {
            return AicactusApiResponse.createAsError(
                AicactusApiResponseCode.NETWORK_ERROR,
                AicactusApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @WorkerThread
    @NonNull
    inline fun <reified T> postWithJson(
        @NonNull uri: Uri,
        @NonNull requestHeaders: Map<String, String>,
        @NonNull postData: String,
    ): AicactusApiResponse<T> {
        Log.d(TAG, "POST: $postData")
        return sendRequestWithJson(
            HttpMethod.POST,
            uri,
            requestHeaders,
            postData,
        )
    }

    @WorkerThread
    @NonNull
    inline fun <reified T> putWithJson(
        @NonNull uri: Uri,
        @NonNull requestHeaders: Map<String, String>,
        @NonNull postData: String,
    ): AicactusApiResponse<T> {
        return sendRequestWithJson(
            HttpMethod.PUT,
            uri,
            requestHeaders,
            postData,
        )
    }

    @WorkerThread
    @NonNull
    inline fun <reified T> sendRequestWithJson(
        @NonNull method: HttpMethod,
        @NonNull uri: Uri,
        @NonNull requestHeaders: Map<String, String>,
        @NonNull postData: String
    ): AicactusApiResponse<T> {
        val postDataBytes = postData.toByteArray()
        var conn: HttpsURLConnection? = null
        try {
            conn = openConnectionWithJson(uri, postDataBytes.size, method)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            conn.outputStream.apply {
                write(postDataBytes)
                flush()
            }
            return getServiceResponse(conn)
        } catch (e: IOException) {
            return AicactusApiResponse.createAsError(
                AicactusApiResponseCode.NETWORK_ERROR,
                AicactusApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @WorkerThread
    @NonNull
    inline fun <reified T> get(
        @NonNull uri: Uri,
        @NonNull requestHeaders: Map<String, String>,
        @NonNull queryParams: Map<String, String>
    ): AicactusApiResponse<T> {
        val fullUri = UriUtils.appendQueryParams(uri, queryParams)
        var conn: HttpsURLConnection? = null
        return try {
            conn = openGetConnection(fullUri)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            getServiceResponse(conn)
        } catch (e: IOException) {
            AicactusApiResponse.createAsError(
                AicactusApiResponseCode.NETWORK_ERROR,
                AicactusApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @WorkerThread
    @NonNull
    inline fun <reified T> delete(
        @NonNull uri: Uri,
        @NonNull requestHeaders: Map<String, String>
    ): AicactusApiResponse<T> {
        var conn: HttpsURLConnection? = null
        return try {
            conn = openDeleteConnection(uri)
            setRequestHeaders(conn, requestHeaders)
            conn.connect()
            getServiceResponse(conn)
        } catch (e: IOException) {
            AicactusApiResponse.createAsError(
                AicactusApiResponseCode.NETWORK_ERROR,
                AicactusApiError(e)
            )
        } finally {
            conn?.disconnect()
        }
    }

    @Throws(IOException::class)
    @NonNull
    fun openPostConnection(
        @NonNull uri: Uri,
        @NonNull postDataSizeByte: Int
    ): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = HttpMethod.POST.name
            doOutput = true
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
            setRequestProperty("Content-Type", "application/x-www-form-urlencoded")
            setRequestProperty("Content-Length", postDataSizeByte.toString())
        }
        return conn
    }

    @Throws(IOException::class)
    @NonNull
    fun openConnectionWithJson(
        @NonNull uri: Uri,
        @NonNull postDataSizeByte: Int,
        @NonNull method: HttpMethod,
    ): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = method.name
            doOutput = true
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
            setRequestProperty("Content-Type", "application/json")
            setRequestProperty("Content-Length", postDataSizeByte.toString())
        }
        return conn
    }

    @Throws(IOException::class)
    @NonNull
    fun openGetConnection(@NonNull uri: Uri): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        Log.d(TAG, userAgentGenerator.userAgent)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = HttpMethod.GET.name
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
        }
        return conn
    }

    @Throws(IOException::class)
    @NonNull
    fun openDeleteConnection(@NonNull uri: Uri): HttpsURLConnection {
        val conn = openHttpsConnection(uri)
        conn.apply {
            instanceFollowRedirects = true
            connectTimeout = connectTimeoutMillis
            readTimeout = readTimeoutMillis
            requestMethod = HttpMethod.DELETE.name
            setRequestProperty("Accept-Encoding", "gzip")
            setRequestProperty("User-Agent", userAgentGenerator.userAgent)
        }
        return conn
    }

    private fun openHttpsConnection(@NonNull uri: Uri): HttpsURLConnection {
        Log.d("ServiceHttpClient", uri.toString())
        val urlConnection = URL(uri.toString()).openConnection()
        if (urlConnection !is HttpsURLConnection) {
            throw IllegalArgumentException("The scheme of the server url must be https.$uri")
        }
        return urlConnection
    }

    fun convertPostDataToBytes(@NonNull postData: Map<String, String>): ByteArray {
        if (postData.isEmpty()) {
            return EMPTY_DATA
        }
        val uri = UriUtils.appendQueryParams("", postData)
        try {
            return uri.encodedQuery?.toByteArray(Charsets.UTF_8) ?: EMPTY_DATA
        } catch (e: UnsupportedEncodingException) {
            throw RuntimeException(e)
        }
    }

    fun setRequestHeaders(
        @NonNull conn: HttpsURLConnection,
        @NonNull requestHeaders: Map<String, String>
    ) {
        for (headerEntry in requestHeaders) {
            conn.setRequestProperty(headerEntry.key, headerEntry.value)
        }
    }

    @Throws(IOException::class)
    @NonNull
    inline fun <reified T> getServiceResponse(@NonNull conn: HttpsURLConnection): AicactusApiResponse<T> {
        val responseCode = conn.responseCode
        try {
            Log.d(TAG, "Response Code: $responseCode")
            if (responseCode == HttpURLConnection.HTTP_INTERNAL_ERROR) {
                return AicactusApiResponse.createAsError(
                    AicactusApiResponseCode.SERVER_ERROR,
                    AicactusApiError.createWithHttpResponseCode(
                        responseCode,
                        "Server Internal Error"
                    )
                )
            }

            if (responseCode == HttpURLConnection.HTTP_NO_CONTENT) {
                return AicactusApiResponse.createAsError(
                    AicactusApiResponseCode.SERVER_ERROR,
                    AicactusApiError.createWithHttpResponseCode(
                        responseCode,
                        "No Content"
                    )
                )
            }

            val inputStream = getInputStreamFrom(conn)
            val reader = InputStreamReader(inputStream)
            return AicactusApiResponse.createAsSuccess(gson.fromJson(reader, T::class.java))
        } catch (e: Exception) {
            return when(e) {
                is IOException -> AicactusApiResponse.createAsError(
                    AicactusApiResponseCode.NETWORK_ERROR,
                    AicactusApiError(e, AicactusApiError.ErrorCode.HTTP_RESPONSE_PARSE_ERROR)
                )
                else -> AicactusApiResponse.createAsError(
                    AicactusApiResponseCode.INTERNAL_ERROR,
                    AicactusApiError("Internal Server Error")
                )
            }

        }
    }

    fun getInputStreamFrom(@NonNull conn: HttpsURLConnection): InputStream {
        val responseCode = conn.responseCode
        val inputStream = if (responseCode < 400) conn.inputStream else conn.errorStream
        return if (isGzipUsed(conn)) GZIPInputStream(inputStream) else inputStream
    }

    private fun isGzipUsed(@NonNull conn: HttpsURLConnection): Boolean {
        val contentEncodings = conn.headerFields["Content-Encoding"]
        contentEncodings?.let {
            for (contentEncoding in contentEncodings) {
                if (contentEncoding.equals("gzip", true)) {
                    return true
                }
            }
        }
        return false
    }

    private var connectTimeoutMillis = DEFAULT_CONNECT_TIMEOUT_MILLIS
    private var readTimeoutMillis = DEFAULT_READ_TIMEOUT_MILLIS

    enum class HttpMethod {
        POST, GET, DELETE, PUT
    }

    companion object: SingletonHolder<ServiceHttpClient, Context>(::ServiceHttpClient) {
        const val CORE_API_SERVER_BASE_URI = "https://staging-adnetwork-core.aicactus.io"
        const val TAGS_API_SERVER_BASE_URI = "https://staging-tags-lb7.aicactus.io/v1/b"

        private const val DEFAULT_CONNECT_TIMEOUT_MILLIS = 10 * 1000
        private const val DEFAULT_READ_TIMEOUT_MILLIS = 10 * 1000
        private val EMPTY_DATA = ByteArray(0)

        val TAG = ServiceHttpClient::class.simpleName
    }
}