package io.aicactus.adsnetwork.internal

import android.content.Context
import android.icu.text.SimpleDateFormat
import android.icu.util.TimeZone
import android.util.Log
import com.google.gson.Gson
import io.aicactus.adsnetwork.api.BatchRequest
import io.aicactus.adsnetwork.models.batch.Batch
import io.aicactus.adsnetwork.models.batch.BatchContext
import io.aicactus.adsnetwork.models.batch.BatchType
import io.aicactus.adsnetwork.utils.UriUtils
import java.util.*
import kotlin.collections.HashMap
import kotlin.concurrent.fixedRateTimer

class BatchController(
    private val context: Context,
    private val serviceHttpClient: ServiceHttpClient,
    private val gson: Gson,
) {
    private var batches = mutableListOf<Batch>()

    private var timer: Timer? = null

    private fun createBatch(type: BatchType, writeKey: String): Batch {
        val anonymousId = UUID.randomUUID().toString()
        val batchContext = BatchContext.getInstance(context).apply {
            this.writeKey = writeKey
        }
        val messageId = UUID.randomUUID().toString()
        val timeStamp = getCurrentISODatetime()
        return Batch(writeKey, anonymousId, messageId, "track", type.value, timeStamp).apply {
            this.context = batchContext
        }
    }

    private fun getCurrentISODatetime(): String {
        val simpleDateFormat = SimpleDateFormat("yyyy-MM-dd'T'HH:mm'Z'", Locale.getDefault())
        simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return simpleDateFormat.format(Date())
    }

    fun addBatch(type: BatchType, containerId: String) {
        val writeKey = "${containerId}@android"

        val batch = createBatch(type, writeKey)
        batches.add(batch)

        val sentAt = getCurrentISODatetime()
        val batchRequest = BatchRequest(writeKey, batches, sentAt)

        val body = gson.toJson(batchRequest)
        val size = body.toByteArray().size
        Log.d(TAG, "Size of package: $size - $body")

        if (size >= MAX_BATCH_PACKAGE_SIZE) {
            sendBatches(body)
        } else {
            timer?.cancel()
            timer = fixedRateTimer("SendBatches", true, 60000, 60000) {
                sendBatches(body)
                timer?.cancel()
            }
        }
    }

    private fun sendBatches(body: String) {
        val uri = UriUtils.buildUri(ServiceHttpClient.TAGS_API_SERVER_BASE_URI)
        val headers = HashMap<String, String>()
        headers["Accept-Encoding"] = "gzip, deflate"
        serviceHttpClient.postWithJson<String>(uri, headers, body)
        batches.clear()
    }

    companion object {
        private val TAG = BatchController::class.simpleName
        private const val MAX_BATCH_PACKAGE_SIZE = 500 * 1024
    }
}