package io.aicactus.adsnetwork.internal

import android.os.Parcel
import android.os.Parcelable
import androidx.annotation.Nullable
import java.lang.Exception
import java.util.*
import kotlin.properties.Delegates

class AicactusApiError() : Parcelable {

    enum class ErrorCode {
        ACTIVITY_NOT_FOUND,
        HTTP_RESPONSE_PARSE_ERROR,
        NOT_DEFINED,
    }

    private var responseCode by Delegates.notNull<Int>()
    private var message: String = ""
    private var errorCode: ErrorCode? = null

    constructor(parcel: Parcel) : this() {
        this.responseCode = parcel.readInt()
        this.message = parcel.readString().toString()
        val tempErrorCode = parcel.readInt()
        this.errorCode = if (tempErrorCode == -1) null else ErrorCode.values()[tempErrorCode]
    }

    constructor(@Nullable e: Exception) : this(DEFAULT_HTTP_RESPONSE_CODE, e.toString(), ErrorCode.NOT_DEFINED)

    constructor(@Nullable message: String) : this(DEFAULT_HTTP_RESPONSE_CODE, message, ErrorCode.NOT_DEFINED)

    constructor(@Nullable e: Exception, errorCode: ErrorCode) : this(DEFAULT_HTTP_RESPONSE_CODE, e.toString(), errorCode)

    constructor(responseCode: Int, @Nullable message: String, errorCode: ErrorCode) : this() {
        this.responseCode = responseCode
        this.message = message
        this.errorCode = errorCode
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        parcel.writeInt(responseCode)
        parcel.writeString(message)
        parcel.writeInt(errorCode?.ordinal ?: -1)
    }

    override fun describeContents(): Int {
        return 0
    }

    fun getMessage(): String {
        return this.message
    }

    override fun hashCode(): Int {
        return Objects.hash(responseCode, getMessage(), errorCode)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is AicactusApiError) return false
        return responseCode == other.responseCode &&
                Objects.equals(getMessage(), other.getMessage()) &&
                errorCode == other.errorCode
    }

    companion object {
        @JvmField
        val CREATOR: Parcelable.Creator<AicactusApiError> = object : Parcelable.Creator<AicactusApiError> {
            override fun createFromParcel(parcel: Parcel): AicactusApiError {
                return AicactusApiError(parcel)
            }

            override fun newArray(size: Int): Array<AicactusApiError?> {
                return Array(size) { AicactusApiError() }
            }
        }

        private const val DEFAULT_HTTP_RESPONSE_CODE = -1

        val DEFAULT = AicactusApiError(DEFAULT_HTTP_RESPONSE_CODE, "Not Defined Error", ErrorCode.NOT_DEFINED)

        fun createWithHttpResponseCode(responseCode: Int, @Nullable errorString: String): AicactusApiError {
            return AicactusApiError(responseCode, errorString, ErrorCode.NOT_DEFINED)
        }

        fun createWithHttpResponseCode(responseCode: Int, @Nullable e: Exception): AicactusApiError {
            return createWithHttpResponseCode(responseCode, e.toString())
        }
    }
}