package io.aicactus.adsnetwork.internal

import android.content.Context
import android.util.Log
import androidx.annotation.NonNull
import io.aicactus.adsnetwork.api.CoreApiClient
import io.aicactus.adsnetwork.models.bid.ApplicationInfo
import io.aicactus.adsnetwork.models.bid.Device
import io.aicactus.adsnetwork.models.container.Container
import io.aicactus.adsnetwork.utils.SingletonHolder
import kotlinx.coroutines.*

class AicactusAdsNetwork(@NonNull val context: Context) {

    private var sdkConfiguration: SdkConfiguration? = null

    var container: Container? = null
    private lateinit var device: Device
    private lateinit var app: ApplicationInfo
    private lateinit var cappingManager: CappingManager

    private var inProcess: Boolean = false

    init {
        SdkConfigurationParser().parse(context)?.let { sdkConfiguration ->
            this.sdkConfiguration = sdkConfiguration
            cappingManager = CappingManager.getInstance(context)
        }
    }

    companion object : SingletonHolder<AicactusAdsNetwork, Context>(::AicactusAdsNetwork) {
        private val TAG = AicactusAdsNetwork::class.simpleName

        suspend fun setup(context: Context, containerID: String?, callback: (Boolean, String?) -> Unit) {
            val instance = AicactusAdsNetwork.getInstance(context)
            if (instance.inProcess) {
                callback(false, "Duplicate Request")
                return
            }
            if (instance.cappingManager.isBlackList()) {
                callback(false, "Client is in blacklist")
                return
            }
            instance.inProcess = true
            containerID?.let {
                instance.sdkConfiguration = SdkConfiguration(it)
            }
            instance.sdkConfiguration?.let { sdkConfiguration ->
                instance.device = Device.getInstance(context)
                instance.app = ApplicationInfo.getInstance(context)
                withContext(Dispatchers.IO) {
                    val coreApiClient = CoreApiClient.getInstance(context)
                    val result = coreApiClient.init(sdkConfiguration.containerId)
                    result.responseData?.let { container ->
                        instance.container = container
                        instance.container?.configurationID = sdkConfiguration.containerId
                        instance.cappingManager.config = container.capping
                        withContext(Dispatchers.Main) {
                            callback(true, null)
                        }
                        return@withContext
                    }
                    result.errorData.getMessage().let { message ->
                        if (message.isNotEmpty()) {
                            Log.e(TAG, "Couldn't init container. Error: $message")
                            withContext(Dispatchers.Main) {
                                callback(false, message)
                            }
                        }
                    }
                    instance.inProcess = false
                }
            } ?: run {
                Log.e(TAG, "AicactusAdsNetwork couldn't be initialized. Please recheck your app manifest")
            }
        }

        suspend fun setup(context: Context, callback: (Boolean, String?) -> Unit) {
            setup(context, null, callback)
        }
    }
}