package io.aicactus.adsnetwork.api

import androidx.annotation.NonNull
import com.google.gson.annotations.SerializedName
import io.aicactus.adsnetwork.models.bid.SeatBid

/**
 * This object is the top-level bid response object (i.e., the unnamed outer JSON object).
 * The id attribute is a reflection of the bid request ID for logging purposes.
 * Similarly, bidid is an optional response tracking ID for bidders.
 * If specified, it can be included in the subsequent win notice call if the bidder wins.
 * At least one seatbid object is required, which contains at least one bid for an impression.
 * Other attributes are optional.
 */
data class BidResponse(
    /**
     * ID of the bid request to which this is a response.
     */
    @field:SerializedName("id")
    @NonNull val id: String,

    /**
     * Array of seatbid objects; 1+ required if a bid is to be made.
     */
    @field:SerializedName("seatbid")
    val seatBids: List<SeatBid>? = null,

    /**
     * Bidder generated response ID to assist with logging/tracking.
     */
    @field:SerializedName("bidid")
    val bidId: String? = null,

    /**
     * Bid currency using ISO-4217 alpha codes.
     */
    @field:SerializedName("cur")
    val currency: String? = "USD",

    /**
     * Optional feature to allow a bidder to set data in the exchange’s cookie.
     * The string must be in base85 cookie safe characters and be in any format.
     * Proper JSON encoding must be used to include “escaped” quotation marks.
     */
    @field:SerializedName("customdata")
    val customData: String? = null,

    /**
     * Reason for not bidding.
     */
    @field:SerializedName("nbr")
    val noBidReasonCode: NoBidReasonCode? = null,

    /**
     * Placeholder for exchange-specific extensions to OpenRTB.
     */
    @field:SerializedName("ext")
    val extension: Any? = null,
) {

    data class Extension(

        @field:SerializedName("tmaxrequest")
        val timeMaxRequest: Int? = null,

        @field:SerializedName("responsetimemillis")
        val responseTimeMillis: ResponseTimeMillis? = null,

        @field:SerializedName("prebid")
        val preBid: PreBid? = null
    ) {

        data class ResponseTimeMillis(

            @field:SerializedName("aicactus")
            val aicactus: Int? = null,

            @field:SerializedName("cache")
            val cache: Int? = null,
        )

        data class PreBid(
            @field:SerializedName("auctiontimestamp")
            val auctionTimeStamp: Long? = null,
        )
    }

    enum class NoBidReasonCode(val value: Int) {
        UNKNOWN(0),
        TECHNICAL(1),
        INVALID(2),
        KNOWN_WEB_SPIDER(3),
        SUSPECTED_NO_HUMAN_TRAFFIC(4),
        CLOUD_DATA_CENTER_PROXY(5),
        UNSUPPORTED_DEVICE(6),
        BLOCKED_PUBLISHER(7),
        UNMATCHED_USER(8),
        DAILY_READER_CAP_MET(9),
        DAILY_DOMAIN_CAP_MET(10),
    }
}