package io.aicactus.adsnetwork.api

import android.content.Context
import android.util.Log
import androidx.annotation.NonNull
import com.google.gson.GsonBuilder
import io.aicactus.adsnetwork.ads.AdRequest
import io.aicactus.adsnetwork.ads.AdSize
import io.aicactus.adsnetwork.internal.*
import io.aicactus.adsnetwork.models.batch.BatchType
import io.aicactus.adsnetwork.models.bid.*
import io.aicactus.adsnetwork.utils.SingletonHolder
import io.aicactus.adsnetwork.utils.UriUtils
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.map
import java.util.*

sealed class Result<out R> {
    data class Success<out T>(val data: T) : Result<T>()
    data class Error(val error: AicactusApiError) : Result<Nothing>()
}

class BidRequestApiClient private constructor(@NonNull val context: Context) {

    @NonNull
    private val serviceHttpClient = ServiceHttpClient.getInstance(context)

    private val gson = GsonBuilder().setExclusionStrategies(AnnotationExclusionStrategy()).create()

    @NonNull
    private val batchController = BatchController(context, serviceHttpClient, gson)

    suspend fun postBidRequest(inventoryId: Int, bidRequest: BidRequest): Result<Bid> {
        val cappingManager = CappingManager.getInstance(context)
        if (cappingManager.isBlackList()) {
            return Result.Error(AicactusApiError("Client is in blacklist"))
        }

        val container = AicactusAdsNetwork.getInstance(context).container
            ?: return Result.Error(AicactusApiError("Not found container"))

        val core = container.core

        // Check if Container has Inventory for Ad Unit ID
        val inventory = container.inventories.firstOrNull { it.id == inventoryId }

        // Check inventory Ad Type
        // if (type.value != inventory.format) {
            // return Result.Error(AicactusApiError("Inventory ${inventory.format} not support ${type.value} Ad"))
        // }

        // Get Ad Default
        val adDefault = inventory?.defaultAds

        // Check if Inventory has DSP
        val inventoryDsps = inventory?.platforms
        // if (inventoryDsps == null || inventoryDsps.isEmpty()) {
            // return Result.Error(AicactusApiError("Not found dsps for inventory"))
        // }

        // Map DSP information
        val dsps = inventoryDsps?.map { dsp ->
            dsp.url = container.platforms?.find { it.id == dsp.id }?.url
            return@map dsp
        }

        // Create Endpoints for request
        val endpoints = arrayOf(Endpoint("core", core.url, "core"))
        dsps?.forEach { dsp ->
            endpoints.plus(Endpoint(dsp.id, dsp.url!!, "dsp", dsp.directDeal > 0))
        }

        // Perform requests to all endpoints
        val flow = endpoints.asFlow().map { endpoint ->
            val uri = UriUtils.buildUri(endpoint.url)
            if (endpoint.type == "core") {
                // If request to core, append extension info to Impression
                appendImpressionExtension(bidRequest, container.id, inventoryId)
            }
            val body = gson.toJson(bidRequest)
            val response = serviceHttpClient.postWithJson<BidResponse>(uri, emptyMap(), body)

            //  Return response with endpoint id for collect later
            Pair(endpoint.id, response)
        }

        // Collect success response in 2500 milliseconds
        val successResponses: MutableList<Pair<String, BidResponse>> = mutableListOf()
        withTimeoutOrNull(2500) {
            flow.collect { (id, response) ->
                if (response.isSuccess && response.responseData != null) {
                    val pair = Pair(id, response.responseData)
                    successResponses.add(pair)
                    batchController.addBatch(BatchType.BID_RESPONSE, container.configurationID!!)
                }
            }
        }

        val bids = mutableListOf<Bid>()

        // Collect bids from direct deal DSP
        val endpoint = endpoints.firstOrNull { it.directDeal }
        if (endpoint != null) {
            val responseData = successResponses.first { it.first == endpoint.id }.second
            responseData.seatBids?.let { seatBids ->
                for (seatBid in seatBids) {
                    bids.addAll(seatBid.bids)
                }
            }
        } else {
            // Or all bids if there's no direct deal DSP
            for (responseData in successResponses) {
                responseData.second.seatBids?.let { seatBids ->
                    for (seatBid in seatBids) {
                        bids.addAll(seatBid.bids)
                    }
                }
            }
        }

        // Filter by floor price
        var floorPrice = 0.0f
        inventory?.let { floorPrice = it.floorPrice}
        val filteredBids = bids.filter { it.price >= floorPrice }.toMutableList()

        // Sort bis by price
        filteredBids.sortByDescending { it.price }

        // Print log for checking
        filteredBids.forEach {
            Log.d(TAG, "Bid ${it.id}: ${it.price}")
        }

        val chosenBid = bids.firstOrNull()
        if (chosenBid != null) {
            batchController.addBatch(BatchType.BID_WINNER, container.configurationID!!)
            return Result.Success(chosenBid)
        } else if (adDefault !== null && adDefault.bid != null) {
            // Return default ad if no bids return and default
            val bidResponse = adDefault.bid
            val bidsDefault = mutableListOf<Bid>()
            if (bidResponse?.seatBids != null) {
                for (seatBid in bidResponse.seatBids) {
                    for (bid in seatBid.bids) {
                        bidsDefault.add(bid)
                    }
                }
                bidsDefault.sortByDescending { it.price }
                bidsDefault.firstOrNull()?.let {
                    return Result.Success(it)
                }
            }
        }

        batchController.addBatch(BatchType.BID_ERROR, container.configurationID!!)
        return Result.Error(AicactusApiError("No available ads"))
    }

    private fun makeBidRequest(adRequest: AdRequest): BidRequest {
        Log.d(TAG, "Make Bid request with ${adRequest.id}")
        val bidRequestId = UUID.randomUUID().toString()
        val bidRequest = BidRequest(bidRequestId)
        bidRequest.apply {
            device = Device.getInstance(context)
            app = ApplicationInfo.getInstance(context)
            impression = arrayOf(Impression(id = UUID.randomUUID().toString()))
            user = User()
        }

        return bidRequest
    }

    private fun appendImpressionExtension(
        bidRequest: BidRequest,
        containerId: Int,
        inventoryId: Int
    ) {
        bidRequest.impression?.first()?.apply {
            extension = Impression.Extension().apply {
                preBid = PreBid()
                aicactusAds = Impression.Extension.AiCactusAd().apply {
                    this.containerId = containerId
                    this.inventoryId = inventoryId
                    this.headerBids = emptyList()
                }
            }
        }
    }

    fun bannerAdRequest(size: AdSize, adRequest: AdRequest): BidRequest {
        val bidRequest = makeBidRequest(adRequest)
        bidRequest.impression!!.first().apply {
            banner = Banner().apply {
                formats = arrayOf(Format().apply {
                    width = size.width
                    height = size.height
                })
            }
        }

        return bidRequest
    }

    fun videoAdRequest(size: AdSize, adRequest: AdRequest): BidRequest {
        Log.d(TAG, "Video size: ${size.width}x${size.height}")
        val bidRequest = makeBidRequest(adRequest)
        bidRequest.impression!!.first().apply {
            video = Video(arrayOf("video/mp4")).apply {
                linearity = VideoLinearity.IN_STREAM
                playbackMethod = arrayOf(PlaybackMethod.ENTER_VIEWPORT_SOUND_OFF)
                placement = VideoPlacement.IN_BANNER
                protocols = arrayOf(Protocol.VAST_2)
                width = size.width
                height = size.height
            }
        }

        return bidRequest
    }

    fun nativeAdRequest(adAssets: Array<Asset.Request>, adRequest: AdRequest): BidRequest {
        val nativeAdRequest = NativeAdRequest(adAssets)
        val request = gson.toJson(nativeAdRequest)
        val bidRequest = makeBidRequest(adRequest)
        bidRequest.impression!!.first().apply {
            native = Native(request)
        }

        return bidRequest
    }

    companion object : SingletonHolder<BidRequestApiClient, Context>(::BidRequestApiClient) {
        private val TAG = BidRequestApiClient::class.simpleName
    }
}