package io.aicactus.adsnetwork.ads

import android.content.Context
import android.util.Log
import io.aicactus.adsnetwork.api.BidRequest
import io.aicactus.adsnetwork.api.BidRequestApiClient
import io.aicactus.adsnetwork.api.Result
import io.aicactus.adsnetwork.models.bid.Asset
import io.aicactus.adsnetwork.models.bid.Bid
import io.aicactus.adsnetwork.utils.SingletonHolder
import kotlinx.coroutines.*

class AdViewController(context: Context) {

    private val bidRequestApiClient = BidRequestApiClient.getInstance(context)

    private val requestJob = Job()
    private val scope = CoroutineScope(Dispatchers.IO + requestJob)

    fun request(
        adUnitID: Int,
        adSize: AdSize,
        adType: AdType,
        adRequest: AdRequest,
        callback: (Result<Bid>) -> Unit
    ) {
        val request = if (adType === AdType.BANNER)
            bidRequestApiClient.bannerAdRequest(adSize, adRequest)
        else bidRequestApiClient.videoAdRequest(adSize, adRequest)
        postRequestJob(adUnitID, request, adType, callback)
    }

    fun requestNativeAd(
        adUnitID: Int,
        adAssets: Array<Asset.Request>,
        adRequest: AdRequest,
        callback: (Result<Bid>) -> Unit,
    ) {
        val request = bidRequestApiClient.nativeAdRequest(adAssets, adRequest)
        postRequestJob(adUnitID, request, AdType.NATIVE, callback)
    }

    private fun postRequestJob(
        adUnitID: Int,
        request: BidRequest,
        type: AdType,
        callback: (Result<Bid>) -> Unit
    ) {
        Log.d(TAG, "Post $type Ad Request")
        scope.launch {
            val response = bidRequestApiClient.postBidRequest(adUnitID, request)
            withContext(Dispatchers.Main) {
                callback(response)
            }
        }
    }

    companion object : SingletonHolder<AdViewController, Context>(::AdViewController) {
        private val TAG = AdViewController::class.simpleName
    }
}