package io.ably.lib.objects.type.livemap

import io.ably.lib.objects.ObjectsMapSemantics
import io.ably.lib.objects.ObjectsMapOp
import io.ably.lib.objects.ObjectOperation
import io.ably.lib.objects.ObjectOperationAction
import io.ably.lib.objects.ObjectState
import io.ably.lib.objects.isInvalid
import io.ably.lib.objects.objectError
import io.ably.lib.objects.type.map.LiveMapUpdate
import io.ably.lib.objects.type.noOp
import io.ably.lib.util.Log

internal class LiveMapManager(private val liveMap: DefaultLiveMap): LiveMapChangeCoordinator() {

  private val objectId = liveMap.objectId

  private val tag = "LiveMapManager"

  /**
   * @spec RTLM6 - Overrides object data with state from sync
   */
  internal fun applyState(objectState: ObjectState, serialTimestamp: Long?): LiveMapUpdate {
    val previousData = liveMap.data.toMap()

    if (objectState.tombstone) {
      liveMap.tombstone(serialTimestamp)
    } else {
      // override data for this object with data from the object state
      liveMap.createOperationIsMerged = false // RTLM6b
      liveMap.data.clear()

      objectState.map?.entries?.forEach { (key, entry) ->
        liveMap.data[key] = LiveMapEntry(
          isTombstoned = entry.tombstone ?: false,
          tombstonedAt = if (entry.tombstone == true) entry.serialTimestamp ?: System.currentTimeMillis() else null,
          timeserial = entry.timeserial,
          data = entry.data
        )
      } // RTLM6c

      // RTLM6d
      objectState.createOp?.let { createOp ->
        mergeInitialDataFromCreateOperation(createOp)
      }
    }

    return calculateUpdateFromDataDiff(previousData, liveMap.data.toMap())
  }

  /**
   * @spec RTLM15 - Applies operations to LiveMap
   */
  internal fun applyOperation(operation: ObjectOperation, serial: String?, serialTimestamp: Long?) {
    val update = when (operation.action) {
      ObjectOperationAction.MapCreate -> applyMapCreate(operation) // RTLM15d1
      ObjectOperationAction.MapSet -> {
        if (operation.mapOp != null) {
          applyMapSet(operation.mapOp, serial) // RTLM15d2
        } else {
          throw objectError("No payload found for ${operation.action} op for LiveMap objectId=${objectId}")
        }
      }
      ObjectOperationAction.MapRemove -> {
        if (operation.mapOp != null) {
          applyMapRemove(operation.mapOp, serial, serialTimestamp) // RTLM15d3
        } else {
          throw objectError("No payload found for ${operation.action} op for LiveMap objectId=${objectId}")
        }
      }
      ObjectOperationAction.ObjectDelete -> liveMap.tombstone(serialTimestamp)
      else -> throw objectError("Invalid ${operation.action} op for LiveMap objectId=${objectId}") // RTLM15d4
    }

    liveMap.notifyUpdated(update) // RTLM15d1a, RTLM15d2a, RTLM15d3a
  }

  /**
   * @spec RTLM16 - Applies map create operation
   */
  private fun applyMapCreate(operation: ObjectOperation): LiveMapUpdate {
    if (liveMap.createOperationIsMerged) {
      // RTLM16b
      // There can't be two different create operation for the same object id, because the object id
      // fully encodes that operation. This means we can safely ignore any new incoming create operations
      // if we already merged it once.
      Log.v(
        tag,
        "Skipping applying MAP_CREATE op on a map instance as it was already applied before; objectId=${objectId}"
      )
      return noOpMapUpdate
    }

    validateMapSemantics(operation.map?.semantics) // RTLM16c

    return mergeInitialDataFromCreateOperation(operation) // RTLM16d
  }

  /**
   * @spec RTLM7 - Applies MAP_SET operation to LiveMap
   */
  private fun applyMapSet(
      mapOp: ObjectsMapOp, // RTLM7d1
      timeSerial: String?, // RTLM7d2
  ): LiveMapUpdate {
    val existingEntry = liveMap.data[mapOp.key]

    // RTLM7a
    if (existingEntry != null && !canApplyMapOperation(existingEntry.timeserial, timeSerial)) {
      // RTLM7a1 - the operation's serial <= the entry's serial, ignore the operation
      Log.v(tag,
        "Skipping update for key=\"${mapOp.key}\": op serial $timeSerial <= entry serial ${existingEntry.timeserial};" +
          " objectId=${objectId}"
      )
      return noOpMapUpdate
    }

    if (mapOp.data.isInvalid()) {
      throw objectError("Invalid object data for MAP_SET op on objectId=${objectId} on key=${mapOp.key}")
    }

    // RTLM7c
    mapOp.data?.objectId?.let {
      // this MAP_SET op is setting a key to point to another object via its object id,
      // but it is possible that we don't have the corresponding object in the pool yet (for example, we haven't seen the *_CREATE op for it).
      // we don't want to return undefined from this map's .get() method even if we don't have the object,
      // so instead we create a zero-value object for that object id if it not exists.
      liveMap.objectsPool.createZeroValueObjectIfNotExists(it) // RTLM7c1
    }

    if (existingEntry != null) {
      // RTLM7a2 - Replace existing entry with new one instead of mutating
      liveMap.data[mapOp.key] = LiveMapEntry(
        isTombstoned = false, // RTLM7a2c
        timeserial = timeSerial, // RTLM7a2b
        data = mapOp.data // RTLM7a2a
      )
    } else {
      // RTLM7b, RTLM7b1
      liveMap.data[mapOp.key] = LiveMapEntry(
        isTombstoned = false, // RTLM7b2
        timeserial = timeSerial,
        data = mapOp.data
      )
    }

    return LiveMapUpdate(mapOf(mapOp.key to LiveMapUpdate.Change.UPDATED))
  }

  /**
   * @spec RTLM8 - Applies MAP_REMOVE operation to LiveMap
   */
  private fun applyMapRemove(
      mapOp: ObjectsMapOp, // RTLM8c1
      timeSerial: String?, // RTLM8c2
      timeStamp: Long?, // RTLM8c3
  ): LiveMapUpdate {
    val existingEntry = liveMap.data[mapOp.key]

    // RTLM8a
    if (existingEntry != null && !canApplyMapOperation(existingEntry.timeserial, timeSerial)) {
      // RTLM8a1 - the operation's serial <= the entry's serial, ignore the operation
      Log.v(
        tag,
        "Skipping remove for key=\"${mapOp.key}\": op serial $timeSerial <= entry serial ${existingEntry.timeserial}; " +
          "objectId=${objectId}"
      )
      return noOpMapUpdate
    }

    val tombstonedAt = if (timeStamp != null) timeStamp else {
      Log.w(
        tag,
        "No timestamp provided for MAP_REMOVE op on key=\"${mapOp.key}\"; using current time as tombstone time; " +
          "objectId=${objectId}"
      )
      System.currentTimeMillis()
    }

    if (existingEntry != null) {
      // RTLM8a2 - Replace existing entry with new one instead of mutating
      liveMap.data[mapOp.key] = LiveMapEntry(
        isTombstoned = true, // RTLM8a2c
        tombstonedAt = tombstonedAt,
        timeserial = timeSerial, // RTLM8a2b
        data = null // RTLM8a2a
      )
    } else {
      // RTLM8b, RTLM8b1
      liveMap.data[mapOp.key] = LiveMapEntry(
        isTombstoned = true, // RTLM8b2
        tombstonedAt = tombstonedAt,
        timeserial = timeSerial
      )
    }

    return LiveMapUpdate(mapOf(mapOp.key to LiveMapUpdate.Change.REMOVED))
  }

  /**
   * For Lww CRDT semantics (the only supported LiveMap semantic) an operation
   * Should only be applied if incoming serial is strictly greater than existing entry's serial.
   * @spec RTLM9 - Serial comparison logic for map operations
   */
  private fun canApplyMapOperation(existingMapEntrySerial: String?, timeSerial: String?): Boolean {
    if (existingMapEntrySerial.isNullOrEmpty() && timeSerial.isNullOrEmpty()) { // RTLM9b
      return false
    }
    if (existingMapEntrySerial.isNullOrEmpty()) { // RTLM9d - If true, means timeSerial is not empty based on previous checks
      return true
    }
    if (timeSerial.isNullOrEmpty()) { // RTLM9c - Check reached here means existingMapEntrySerial is not empty
      return false
    }
    return timeSerial > existingMapEntrySerial // RTLM9e - both are not empty
  }

  /**
   * @spec RTLM17 - Merges initial data from create operation
   */
  private fun mergeInitialDataFromCreateOperation(operation: ObjectOperation): LiveMapUpdate {
    if (operation.map?.entries.isNullOrEmpty()) { // no map entries in MAP_CREATE op
      return noOpMapUpdate
    }

    val aggregatedUpdate = mutableListOf<LiveMapUpdate>()

    // RTLM17a
    // in order to apply MAP_CREATE op for an existing map, we should merge their underlying entries keys.
    // we can do this by iterating over entries from MAP_CREATE op and apply changes on per-key basis as if we had MAP_SET, MAP_REMOVE operations.
    operation.map?.entries?.forEach { (key, entry) ->
      // for a MAP_CREATE operation we must use the serial value available on an entry, instead of a serial on a message
      val opTimeserial = entry.timeserial
      val update = if (entry.tombstone == true) {
        // RTLM17a2 - entry in MAP_CREATE op is removed, try to apply MAP_REMOVE op
        applyMapRemove(ObjectsMapOp(key), opTimeserial, entry.serialTimestamp)
      } else {
        // RTLM17a1 - entry in MAP_CREATE op is not removed, try to set it via MAP_SET op
        applyMapSet(ObjectsMapOp(key, entry.data), opTimeserial)
      }

      // skip noop updates
      if (update.noOp) {
        return@forEach
      }

      aggregatedUpdate.add(update)
    }

    liveMap.createOperationIsMerged = true // RTLM17b

    return LiveMapUpdate(
      aggregatedUpdate.map { it.update }.fold(emptyMap()) { acc, map -> acc + map }
    )
  }

  internal fun calculateUpdateFromDataDiff(
    prevData: Map<String, LiveMapEntry>,
    newData: Map<String, LiveMapEntry>
  ): LiveMapUpdate {
    val update = mutableMapOf<String, LiveMapUpdate.Change>()

    // Check for removed entries
    for ((key, prevEntry) in prevData) {
      if (!prevEntry.isTombstoned && !newData.containsKey(key)) {
        update[key] = LiveMapUpdate.Change.REMOVED
      }
    }

    // Check for added/updated entries
    for ((key, newEntry) in newData) {
      if (!prevData.containsKey(key)) {
        // if property does not exist in current map, but new data has it as non-tombstoned property - got updated
        if (!newEntry.isTombstoned) {
          update[key] = LiveMapUpdate.Change.UPDATED
        }
        // otherwise, if new data has this prop tombstoned - do nothing, as property didn't exist anyway
        continue
      }

      // properties that exist both in current and new map data need to have their values compared to decide on update type
      val prevEntry = prevData[key]!!

      // compare tombstones first
      if (prevEntry.isTombstoned && !newEntry.isTombstoned) {
        // prev prop is tombstoned, but new is not. it means prop was updated to a meaningful value
        update[key] = LiveMapUpdate.Change.UPDATED
        continue
      }
      if (!prevEntry.isTombstoned && newEntry.isTombstoned) {
        // prev prop is not tombstoned, but new is. it means prop was removed
        update[key] = LiveMapUpdate.Change.REMOVED
        continue
      }
      if (prevEntry.isTombstoned && newEntry.isTombstoned) {
        // props are tombstoned - treat as noop, as there is no data to compare
        continue
      }

      // both props exist and are not tombstoned, need to compare values to see if it was changed
      val valueChanged = prevEntry.data != newEntry.data
      if (valueChanged) {
        update[key] = LiveMapUpdate.Change.UPDATED
        continue
      }
    }

    return LiveMapUpdate(update)
  }

  internal fun validate(state: ObjectState) {
    liveMap.validateObjectId(state.objectId)
    validateMapSemantics(state.map?.semantics)
    state.createOp?.let { createOp ->
      liveMap.validateObjectId(createOp.objectId)
      validateMapCreateAction(createOp.action)
      validateMapSemantics(createOp.map?.semantics)
    }
  }

  private fun validateMapCreateAction(action: ObjectOperationAction) {
    if (action != ObjectOperationAction.MapCreate) {
      throw objectError("Invalid create operation action $action for LiveMap objectId=${objectId}")
    }
  }

  private fun validateMapSemantics(semantics: ObjectsMapSemantics?) {
    if (semantics != liveMap.semantics) {
      throw objectError(
        "Invalid object: incoming object map semantics=$semantics; current map semantics=${ObjectsMapSemantics.LWW}"
      )
    }
  }
}
