package eu.codlab.google.maps

import eu.codlab.google.maps.models.GoogleMapsError
import eu.codlab.google.maps.models.SessionErrorResult
import eu.codlab.google.maps.models.SessionParam
import eu.codlab.google.maps.models.SessionResult
import eu.codlab.http.Configuration
import eu.codlab.http.createClient
import io.ktor.client.call.body
import io.ktor.client.request.accept
import io.ktor.client.request.get
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.client.statement.bodyAsChannel
import io.ktor.client.statement.bodyAsText
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.http.isSuccess
import io.ktor.utils.io.toByteArray
import kotlinx.io.Buffer

class InternalSession {
    private val client = createClient {
        Configuration(
            enableLogs = true
        )
    }

    private fun getTileUrl(
        apiKey: String,
        session: String,
        row: Int, col: Int, zoomLevel: Int
    ) = "${zoomLevel}/$col/$row".let {
        "https://tile.googleapis.com/v1/2dtiles/$it?key=$apiKey&session=$session"
    }

    suspend fun getTile(
        apiKey: String,
        session: String,
        row: Int, col: Int, zoomLevel: Int
    ): ByteArray {
        val url = getTileUrl(apiKey, session, row, col, zoomLevel)

        val request = client.get(url)

        if (!request.status.isSuccess()) {
            val errorResult = request.body<SessionErrorResult>()

            throw GoogleMapsError(errorResult.error)
        }

        return request.bodyAsChannel().toByteArray()
    }


    suspend fun getTileAsBuffer(
        apiKey: String,
        session: String,
        row: Int, col: Int, zoomLvl: Int
    ): Buffer {
        val result = getTile(apiKey, session, row, col, zoomLvl)

        return  Buffer().also {
            it.write(result)
        }
    }

    suspend fun createSession(
        apiKey: String,
        param: SessionParam
    ): SessionResult {
        val request = client.post("https://tile.googleapis.com/v1/createSession?key=$apiKey") {
            contentType(ContentType.Application.Json)
            accept(ContentType.Application.Json)

            setBody(param)
        }

        if (!request.status.isSuccess()) {
            val errorResult = request.body<SessionErrorResult>()
            throw GoogleMapsError(errorResult.error)
        }

        return request.body()
    }
}