/*
 * Decompiled with CFR 0.152.
 */
package dev.sigstore.encryption;

import com.google.common.annotations.VisibleForTesting;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EllipticCurve;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Locale;
import java.util.logging.Logger;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ECKeyParameters;
import org.bouncycastle.crypto.params.Ed25519PublicKeyParameters;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import org.bouncycastle.crypto.util.PublicKeyFactory;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.ECPointUtil;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.spec.ECNamedCurveSpec;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;

public class Keys {
    private static final Logger log = Logger.getLogger(Keys.class.getName());

    public static PublicKey parsePublicKey(byte[] keyBytes) throws InvalidKeySpecException, IOException, NoSuchAlgorithmException {
        PemReader pemReader = new PemReader((Reader)new InputStreamReader((InputStream)new ByteArrayInputStream(keyBytes), StandardCharsets.UTF_8));
        PemObject section = pemReader.readPemObject();
        if (pemReader.readPemObject() != null) {
            throw new InvalidKeySpecException("sigstore public keys must be only a single PEM encoded public key");
        }
        if (section.getType().equals("RSA PUBLIC KEY")) {
            ASN1Sequence sequence = ASN1Sequence.getInstance((Object)section.getContent());
            ASN1Integer modulus = ASN1Integer.getInstance((Object)sequence.getObjectAt(0));
            ASN1Integer exponent = ASN1Integer.getInstance((Object)sequence.getObjectAt(1));
            RSAPublicKeySpec keySpec = new RSAPublicKeySpec(modulus.getPositiveValue(), exponent.getPositiveValue());
            KeyFactory factory = KeyFactory.getInstance("RSA");
            return factory.generatePublic(keySpec);
        }
        byte[] content = section.getContent();
        X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(content);
        AsymmetricKeyParameter keyParameters = PublicKeyFactory.createKey((byte[])content);
        String keyAlgorithm = Keys.extractKeyAlgorithm(keyParameters);
        KeyFactory keyFactory = KeyFactory.getInstance(keyAlgorithm);
        return keyFactory.generatePublic(publicKeySpec);
    }

    public static PublicKey constructTufPublicKey(byte[] contents, String scheme) throws NoSuchAlgorithmException, InvalidKeySpecException {
        PublicKey publicKey = null;
        switch (scheme) {
            case "rsassa-pss-sha256": {
                throw new RuntimeException("rsassa-pss-sha256 not currently supported");
            }
            case "ed25519": {
                KeyFactory kf = KeyFactory.getInstance("Ed25519");
                X509EncodedKeySpec keySpec = new X509EncodedKeySpec(contents);
                publicKey = kf.generatePublic(keySpec);
                break;
            }
            case "ecdsa-sha2-nistp256": {
                ECNamedCurveParameterSpec spec = ECNamedCurveTable.getParameterSpec((String)"P-256");
                KeyFactory kf = null;
                try {
                    kf = KeyFactory.getInstance("ECDSA", "BC");
                }
                catch (NoSuchProviderException e) {
                    throw new RuntimeException(e);
                }
                ECNamedCurveSpec params = new ECNamedCurveSpec("P-256", spec.getCurve(), spec.getG(), spec.getN());
                ECPoint point = ECPointUtil.decodePoint((EllipticCurve)params.getCurve(), (byte[])contents);
                ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(point, (ECParameterSpec)params);
                publicKey = kf.generatePublic(pubKeySpec);
                break;
            }
        }
        return publicKey;
    }

    private static String extractKeyAlgorithm(AsymmetricKeyParameter keyParameters) throws NoSuchAlgorithmException {
        if (keyParameters instanceof RSAKeyParameters) {
            return "RSA";
        }
        if (keyParameters instanceof Ed25519PublicKeyParameters) {
            return "EdDSA";
        }
        if (keyParameters instanceof ECKeyParameters) {
            return "EC";
        }
        String error = String.format(Locale.ROOT, "The key provided was of type: %s. We only support RSA, EdDSA, and EC ", keyParameters);
        log.warning(error);
        throw new NoSuchAlgorithmException(error);
    }

    @VisibleForTesting
    protected static int getJavaVersion() {
        return Keys.getJavaVersion(System.getProperty("java.version"));
    }

    @VisibleForTesting
    protected static int getJavaVersion(String version) {
        return Integer.parseInt(version.substring(0, version.indexOf(".")));
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }
}

