package dev.langchain4j.kotlin.model.chat

import dev.langchain4j.internal.VirtualThreadUtils
import dev.langchain4j.kotlin.model.chat.request.ChatRequestBuilder
import dev.langchain4j.kotlin.model.chat.request.chatRequest
import dev.langchain4j.model.chat.request.ChatRequest
import dev.langchain4j.model.chat.response.ChatResponse
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.withContext
import kotlin.coroutines.CoroutineContext

/**
 * Asynchronously processes a chat request using the language model within
 * a coroutine scope. This extension function provides a structured
 * concurrency wrapper around the synchronous [dev.langchain4j.model.chat.ChatModel.chat] method.
 *
 * Example usage:
 * ```kotlin
 * val response = model.chatAsync(ChatRequest(messages))
 * val response2 = model.chatAsync(request = chatRequest, coroutineContext = Dispatchers.IO)
 * ```
 *
 * @param request The chat request containing messages and optional parameters
 *    for the model.
 * @param coroutineContext processes a chat request in provided [CoroutineContext]
 * @return [ChatResponse] containing the model's response and any additional
 *    metadata.
 * @throws Exception if the chat request fails or is interrupted.
 * @see dev.langchain4j.model.chat.ChatModel.chat(ChatRequest)
 * @see ChatRequest
 * @see ChatResponse
 * @author Konstantin Pavlov
 */
@JvmOverloads
public suspend fun dev.langchain4j.model.chat.ChatModel.chatAsync(
    request: ChatRequest,
    coroutineContext: CoroutineContext = defaultCoroutineContext()
): ChatResponse {
    val model = this
    return withContext(coroutineContext) { model.chat(request) }
}

/**
 * Asynchronously processes a chat request using a [ChatRequest.Builder] for
 * convenient request configuration. This extension function combines the
 * builder pattern with coroutine-based asynchronous execution.
 *
 * Example usage:
 * ```kotlin
 * val response = model.chat(
 *     ChatRequest.builder()
 *         .messages(listOf(UserMessage("Hello")))
 *         .temperature(0.7)
 *         .maxTokens(100)
 * )
 * ```
 *
 * @param requestBuilder The builder instance configured with desired chat
 *    request parameters.
 * @param coroutineContext processes a chat request in provided [CoroutineContext]
 * @return [ChatResponse] containing the model's response and any additional
 *    metadata.
 * @throws Exception if the chat request fails, is interrupted, or the builder
 *    produces an invalid configuration.
 * @see ChatRequest
 * @see ChatResponse
 * @see ChatRequest.Builder
 * @see chatAsync
 * @author Konstantin Pavlov
 */
@JvmOverloads
public suspend fun dev.langchain4j.model.chat.ChatModel.chat(
    requestBuilder: ChatRequest.Builder,
    coroutineContext: CoroutineContext = defaultCoroutineContext()
): ChatResponse = chatAsync(coroutineContext = coroutineContext, request = requestBuilder.build())

/**
 * Asynchronously processes a chat request by configuring a [ChatRequest]
 * using a provided builder block. This method facilitates the creation
 * of well-structured chat requests using a [ChatRequestBuilder] and
 * executes the request using the associated [dev.langchain4j.model.chat.ChatModel].
 *
 * Example usage:
 * ```kotlin
 * model.chat {
 *     messages += systemMessage("You are a helpful assistant")
 *     messages += userMessage("Say 'Hello'")
 *     parameters {
 *         temperature = 0.1
 *     }
 * }
 * ```
 *
 * @param block A lambda with receiver on [ChatRequestBuilder] used to
 *    configure the messages and parameters for the chat request.
 * @param coroutineContext processes a chat request in provided [CoroutineContext]
 * @return A [ChatResponse] containing the response from the model and any
 *    associated metadata.
 * @throws Exception if the chat request fails or encounters an error during execution.
 * @author Konstantin Pavlov
 */
public suspend fun dev.langchain4j.model.chat.ChatModel.chat(
    coroutineContext: CoroutineContext = defaultCoroutineContext(),
    block: ChatRequestBuilder.() -> Unit
): ChatResponse = chatAsync(coroutineContext = coroutineContext, request = chatRequest(block))

public suspend fun dev.langchain4j.model.chat.ChatModel.chat(block: ChatRequestBuilder.() -> Unit): ChatResponse =
    chatAsync(coroutineContext = defaultCoroutineContext(), request = chatRequest(block))

/**
 * Provides the default [CoroutineContext] for executing asynchronous operations.
 *
 * This method attempts to create a coroutine dispatcher backed by a virtual thread
 *  executor if virtual threads are available on the current platform (Java 21+).
 * If virtual threads are not supported, it defaults to using [Dispatchers.IO].
 *
 * @return A [CoroutineContext] appropriate for executing background tasks,
 *         defaulting to a virtual thread dispatcher when available or [Dispatchers.IO] otherwise.
 */
internal fun defaultCoroutineContext(): CoroutineContext =
    if (VirtualThreadUtils.isVirtualThreadsSupported()) {
        VirtualThreadUtils.createVirtualThreadExecutor().asCoroutineDispatcher()
    } else {
        Dispatchers.IO
    }

