package de.yourinspiration.configuration;

import java.util.Optional;

/**
 * Holds the value of a configuration property and provides type save access the
 * value.
 * 
 * @author Marcel Härle - marcel@yourinspiration.de
 *
 */
public interface ConfigurationValue {

    /**
     * Retrieves the property value as a string.
     * 
     * @return returns an empty optional of the value is not present
     */
    Optional<String> asString();

    /**
     * Retrieves the property value as an integer.
     * 
     * @return returns an empty optional of the value is not present or could
     *         not be parsed as an integer
     */
    Optional<Integer> asInt();

    /**
     * Retrieves the property value as a long.
     * 
     * @return returns an empty optional of the value is not present or could
     *         not be parsed as a long
     */
    Optional<Long> asLong();

    /**
     * Retrieves the property value as a boolean.
     * 
     * @return returns an empty optional of the value is not present or could
     *         not be parsed as a boolean
     */
    Optional<Boolean> asBoolean();

    /**
     * Retrieves the property value as string or the provided default value if
     * not present.
     * 
     * @param defaultValue
     *            the default value to be returned if the value is not present
     * @return returns the default value if the value is not present
     */
    String asStringOrDefault(String defaultValue);

    /**
     * Retrieves the property value as an integer or the provided default value
     * if not present or if the value could not be parsed,
     * 
     * @param defaultValue
     *            the default value to be returned if the value is not present
     *            or could not be parsed
     * @return returns the default value if the value is not present or could
     *         not parsed
     */
    int asIntOrDefault(int defaultValue);

    /**
     * Retrieves the property value as a long or the provided default value if
     * not present or if the value could not be parsed,
     * 
     * @param defaultValue
     *            the default value to be returned if the value is not present
     *            or could not be parsed
     * @return returns the default value if the value is not present or could
     *         not parsed
     */
    long asLongOrDefault(long defaultValue);

    /**
     * Retrieves the property value as a boolean or the provided default value
     * if not present or if the value could not be parsed,
     * 
     * @param defaultValue
     *            the default value to be returned if the value is not present
     *            or could not be parsed
     * @return returns the default value if the value is not present or could
     *         not parsed
     */
    boolean asBooleanOrDefault(boolean defaultValue);

}
