/**
 * Copyright (c) 2025 Wunderkopf Technologies GmbH
 * All rights reserved.
 *
 * Permission is hereby granted, free  of charge, to any person obtaining
 * a  copy  of this  software  and  associated  documentation files  (the
 * "Software"), to  deal in  the Software without  restriction, including
 * without limitation  the rights to  use, copy, modify,  merge, publish,
 * distribute,  sublicense, and/or sell  copies of  the Software,  and to
 * permit persons to whom the Software  is furnished to do so, subject to
 * the following conditions:
 *
 * The  above  copyright  notice  and  this permission  notice  shall  be
 * included in all copies or substantial portions of the Software.
 *
 * THE  SOFTWARE IS  PROVIDED  "AS  IS", WITHOUT  WARRANTY  OF ANY  KIND,
 * EXPRESS OR  IMPLIED, INCLUDING  BUT NOT LIMITED  TO THE  WARRANTIES OF
 * MERCHANTABILITY,    FITNESS    FOR    A   PARTICULAR    PURPOSE    AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE,  ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

package de.wunderkopfservices.creditscore.client.internal;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import java.util.Base64;
import java.util.Base64.Encoder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WkApiAuthenticator {

	private static final Logger logger = LoggerFactory.getLogger(WkApiAuthenticator.class);
	
	private static final String IAM_ENDPOINT = 
			"https://iam.eu-west-1.wunderkopfservices.de/v2/oauth2/token";
	
	private Token cachedAccessToken = null;
	private static WkApiAuthenticator instance = null;
	private HttpClient client;
	
	final Encoder encoder = Base64.getEncoder();
	
	private WkApiAuthenticator() {
		this.client = HttpClient.newBuilder()
				.connectTimeout(Duration.ofMinutes(2)).build();
	}
	
	public static WkApiAuthenticator getInstance() {
		if (instance == null) {
			synchronized(WkApiAuthenticator.class) {
				if (instance == null) {
					instance = new WkApiAuthenticator();
				}
			}
		}
		return instance;
	}
	
	public synchronized Token getAccessToken(String refreshToken) throws WkApiClientException {
		
		if (cachedAccessToken != null ) {
			if (this.cachedAccessToken.isValid(5)) {
				return this.cachedAccessToken;
			} else {
				this.cachedAccessToken = null;
			} 			
		}
		
		Token accessToken = this.exchangeAccessToken(refreshToken);
		this.cachedAccessToken = accessToken;
		
		return accessToken;
		
	}
	
	private Token exchangeAccessToken(String refreshToken) throws WkApiClientException {
		
		String postString = "token=" + encoder.encodeToString(refreshToken.getBytes());
		
		HttpRequest request = HttpRequest.newBuilder()
				.uri(URI.create(IAM_ENDPOINT))
				.header("Content-Type", "application/x-www-form-urlencoded")
				.POST(BodyPublishers.ofString(postString))
				.build();
		HttpResponse<String> response;
		try {
			response = client.send(request, BodyHandlers.ofString());
			if (response.statusCode() != 200) {
				String responseBody = response.body();
				int status = response.statusCode();
				
				logger.error("Exchange access token failed [responseBody={}, status={}]",
						responseBody, status);
				throw new WkApiClientException(
						"Exchange of access token failed", 
						responseBody, status);
			}
			return JSON.parse(response.body(), Token.class);
		} catch (IOException | InterruptedException e) {
			
			logger.debug("Network error while exchanging token, start retry [uri={}]",
					request.uri().toString(), e);
			
			// Retry
			try {
				response = client.send(request, BodyHandlers.ofString());
				if (response.statusCode() != 200) {
					String responseBody = response.body();
					int status = response.statusCode();
					
					logger.error("Exchange access token failed [responseBody={}, status={}]",
							responseBody, status);
					throw new WkApiClientException(
							"Exchange of access token failed", 
							responseBody, status);
				}
				return JSON.parse(response.body(), Token.class);
			}  catch (IOException | InterruptedException e2) {
				logger.error("Network error while exchanging token, retry failed [uri={}]",
						request.uri().toString());
				throw new WkApiClientException("Network error while exchanging token", e2);
			}
		}
	}
}
